/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.HistorySpace}.
 * @module baja/hist/HistorySpace
 */
define(["bajaScript/sys",
        "bajaScript/baja/nav/NavNodeContainer",
        "bajaScript/baja/comm/Callback"], function (
        baja,
        NavNodeContainer,
        Callback) {
  
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper,
      objectify = baja.objectify;
  
  /**
   * Represents a `history:HistorySpace` in BajaScript.
   *
   * @class
   * @alias module:baja/hist/HistorySpace
   * @extends baja.NavContainer
   */   
  var HistorySpace = function HistorySpace() {
    callSuper(HistorySpace, this, [{
      navName: "history",
      icon: "module://icons/x16/historyDatabase.png",
      ord: "local:|history:"
    }]);
  };
  
  subclass(HistorySpace, NavNodeContainer);

  function getOrdsJson(ords) {
    var ordArray = [],
        i;

    if (baja.hasType(ords)) {
      ordArray.push(ords.toString());
    }
    else {
      for (i = 0; i < ords.length; ++i) {
        ordArray.push(ords[i].toString());
      }
    }

    return ordArray;
  }

  /**
   * Clear records from the specified histories via their ORDs.
   *
   * @param {Object|String|Array|baja.Ord} [obj] The ORDs or object literal.
   * @param {String|baja.Ord|Array} obj.ords An array of baja.Ord. Or an array
   * of ORD Strings. Or a single baja.Ord. Or a single String ORD.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the
   * records have been successfully cleared.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the
   * history devices fails to resolve.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object.
   * 
   * @returns {Promise} A promise that's resolved once all of the records have
   * been cleared.
   */
  HistorySpace.prototype.clearAllRecords = function(obj) {
    obj = objectify(obj, "ords");

    var cb = new Callback(obj.ok, obj.fail, obj.batch);

    baja.comm.clearAllRecords(getOrdsJson(obj.ords), cb);
    return cb.promise();
  };

  /**
   * Clear the history records before the specified absolute time.
   *
   * @param {Object|String|Array|baja.Ord} [obj] The ORDs or object literal.
   * @param {String|baja.Ord|Array} obj.ords An array of baja.Ord. Or an array
   * of ORD Strings. Or a single baja.Ord. Or a single String ORD.
   * @param {baja.AbsTime} obj.before If the first argument is not an object
   * literal, the before date can be passed in as the second argument. The
   * before absolute time specifies the deletion range.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the
   * records have been successfully cleared.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the
   * history devices fails to resolve.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object.
   * 
   * @returns {Promise} A promise that's resolved once all of the records have
   * been cleared.
   */
  HistorySpace.prototype.clearOldRecords = function(obj, before) {
    obj = objectify(obj, "ords");

    var arg = {
        ords: getOrdsJson(obj.ords),
        before: (obj.before || before).encodeToString()
      },
      cb = new Callback(obj.ok, obj.fail, obj.batch);

    baja.comm.clearOldRecords(arg, cb);
    return cb.promise();
  };

  /**
   * Delete the specified histories.
   *
   * @param {Object|String|Array|baja.Ord} [obj] The ORDs or object literal.
   * @param {String|baja.Ord|Array} obj.ords An array of baja.Ord. Or an array
   * of ORD Strings. Or a single baja.Ord. Or a single String ORD.
   * @param {Function} [obj.ok] (Deprecated: use Promise) Called once the
   * records have been successfully cleared.
   * @param {Function} [obj.fail] (Deprecated: use Promise) Called if the
   * history devices fails to resolve.
   * @param {baja.comm.Batch} [obj.batch] if defined, any network calls will be
   * batched into this object.
   * 
   * @returns {Promise} A promise that's resolved once all of the histories
   * have been deleted.
   */
  HistorySpace.prototype.deleteHistories = function(obj) {
    obj = objectify(obj, "ords");

    var cb = new Callback(obj.ok, obj.fail, obj.batch);

    baja.comm.deleteHistories(getOrdsJson(obj.ords), cb);
    return cb.promise();
  };
  
  return HistorySpace;
});