/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines {@link baja.Slot}.
 * @module baja/comp/Slot
 */
define(["bajaScript/sys"], function (baja) {
  
  "use strict";
  
  var BaseBajaObj = baja.BaseBajaObj,
  
      subclass = baja.subclass;
  
  /**
   * A Niagara `Complex` is made up of `Slot`s. A `Slot` can be a `Property`, 
   * `Action` or a `Topic`.
   * 
   * This is the base class for all `Slot`s in BajaScript.
   * 
   * A new object should never be directly created with this Constructor.
   *
   * @class
   * @alias baja.Slot
   * @extends baja.BaseBajaObj
   *
   * @param {String} slotName  the name of the Slot.
   * @param {String} displayName  the display name of the Slot.
   */
  var Slot = function Slot(slotName, displayName) {
    this.$slotName = slotName || "";
    this.$displayName = displayName || "";
  };
  
  subclass(Slot, BaseBajaObj);

  /**
   * Return the name of the `Slot`.
   *
   * @returns {String}
   */
  Slot.prototype.getName = function () { 
    return this.$slotName; 
  };
  
  /**
   * Return a `String` representation of the `Slot`.
   *
   * @returns {String}
   */
  Slot.prototype.toString = function () { 
    return this.getName(); 
  };
  
  /**
   * Return the display name of the `Slot`.
   * 
   * Please note, this method is intended for INTERNAL use by Tridium only. An
   * external developer should never call this method.
   *   
   * @private 
   * @returns {String}
   */
  Slot.prototype.$getDisplayName = function () { 
    return this.$displayName; 
  };
  
  /**
   * Set the display name of the `Slot`.
   * 
   * Please note, this method is intended for INTERNAL use by Tridium only. An
   * external developer should never call this method.
   *  
   * @private 
   * @param {String} displayName
   */
  Slot.prototype.$setDisplayName = function (displayName) { 
    this.$displayName = displayName;
  };
      
  /**
   * Is the Slot frozen?
   *
   * @name baja.Slot#isFrozen
   * @function
   *
   * @returns {Boolean}
   */
  Slot.prototype.isFrozen = function () {  
    return false; 
  };
  
  /**
   * Is the Slot a Property?
   *
   * @name baja.Slot#isProperty
   * @function
   *
   * @returns {Boolean}
   */
  Slot.prototype.isProperty = function () {
    return false;
  };
  
  /**
   * Is the Slot a Topic?
   *
   * @name baja.Slot#isTopic
   * @function
   *
   * @returns {Boolean}
   */
  Slot.prototype.isTopic = function () {
    return false;
  };
  
  /**
   * Is the Slot an Action?
   *
   * @name baja.Slot#isAction
   * @function
   *
   * @returns {Boolean}
   */
  Slot.prototype.isAction = function () {
    return false;
  };

  /**
   * Return the Flags for the slot.
   *
   * @abstract
   * @function baja.Slot#getFlags
   * @returns {Number}
   */
  
  return Slot;
});