/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines a BajaScript implementation of a dynamic property (not exposed
 * on `baja` namespace).
 * @module baja/comp/DynamicProperty
 */
define(["bajaScript/sys",
        "bajaScript/baja/comp/Property",
        "bajaScript/baja/obj/Facets"], 
        function (baja, Property, Facets) {
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper,
      bajaDef = baja.def;
  
  /**
   * Dynamic Property Slot.
   * 
   * Property defines a Slot which is a storage location for a variable in a 
   * `Complex`.
   * 
   * A new object should never be directly created with this Constructor. All 
   * Slots are created internally by BajaScript.
   *
   * @class
   * @alias module:baja/comp/DynamicProperty
   * @extends baja.Property
   * @private
   * @inner
   */
  var DynamicProperty = function DynamicProperty(slotName, displayName, display, flags, facets, value) {  
    callSuper(DynamicProperty, this, [slotName, displayName]);
    this.$display = display || "";
    this.$flags = flags || 0;
    this.$facets = facets || Facets.DEFAULT;
    this.$val = bajaDef(value, null);
  };
  
  subclass(DynamicProperty, Property);
    
  /**   
   * Return the Property value.
   * 
   * Please note, this method is intended for INTERNAL use by Tridium only. An
   * external developer should never call this method. Access a Property's value
   * through the associated {@link baja.Complex} instead.
   *
   * @see baja.Complex#get
   *
   * @private  
   *
   * @returns value
   */
  DynamicProperty.prototype.$getValue = function () {
    return this.$val;
  };

  /**   
   * Set the Property value.
   * 
   * Please note, this method is intended for INTERNAL use by Tridium only. An
   * external developer should never call this method.
   *
   * @private  
   *
   * @param val value to be set.
   */
  DynamicProperty.prototype.$setValue = function (val) {
    this.$val = val;
  };   
    
  /**   
   * Return the Flags for the Property.
   *
   * @see baja.Flags
   *
   * @returns {Number}
   */
  DynamicProperty.prototype.getFlags = function () { 
    return this.$flags; 
  };
  
  /**   
   * Set the Flags for the Property.
   * 
   * Please note, this method is intended for INTERNAL use by Tridium only. An
   * external developer should never call this method.
   *
   * @private
   * @see baja.Flags
   *
   * @param {Number} flags
   */
  DynamicProperty.prototype.$setFlags = function (flags) { 
    this.$flags = flags; 
  };
  
  /**
   * Return the Facets for the Property.
   *
   * @see baja.Facets
   *
   * @returns the Slot Facets
   */
  DynamicProperty.prototype.getFacets = function () {
    return this.$facets; 
  };
  
  /**   
   * Set the Facets for the Property.
   * 
   * Please note, this method is intended for INTERNAL use by Tridium only. An
   * external developer should never call this method.
   *
   * @private
   * @see baja.Facets
   *
   * @param {baja.Facets} facets
   */
  DynamicProperty.prototype.$setFacets = function (facets) {
    this.$facets = facets; 
  };
  
  /**
   * Return the default flags for the Property.
   *
   * @returns {Number}
   */
  DynamicProperty.prototype.getDefaultFlags = function () {
    return this.getFlags();
  };
  
  /**
   * Return the default value for the Property.
   *
   * @returns the default value for the Property.
   */
  DynamicProperty.prototype.getDefaultValue = function () {   
    return this.$val;
  }; 
  
  /**
   * Return the Type for this Property.
   *
   * @returns the Type for the Property.
   */
  DynamicProperty.prototype.getType = function () { 
    return this.$val.getType(); 
  };
    
  /**
   * Return the display String for this Property.
   * 
   * Please note, this method is intended for INTERNAL use by Tridium only. An
   * external developer should never call this method.
   *
   * @private
   *
   * @returns {String}
   */
  DynamicProperty.prototype.$getDisplay = function () {
    return this.$display;
  };
  
  /**
   * Set the display for this Property.
   * 
   * Please note, this method is intended for INTERNAL use by Tridium only. An
   * external developer should never call this method.
   *
   * @private
   *
   * @param {String} display the display String.
   */
  DynamicProperty.prototype.$setDisplay = function (display) {
    this.$display = display;
  };
  
  return DynamicProperty;
});