/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * Defines `ServerSession` (not exposed on `baja` namespace).
 * @module baja/comm/ServerSession
 * @private
 */
define([], function () {
  "use strict";
  
  var eventHandlers = {}; // Server Session event handlers
  
  /**
   * ServerSession implementation used for polling Server data.
   *
   * @class
   * @alias module:baja/comm/ServerSession
   * @private
   * @inner
   */
  var ServerSession = function ServerSession(id) {
    this.$id = id;
  };
  
  /**
   * Add a `ServerSession` `make` network request.
   *
   * @private
   */
  ServerSession.make = function (cb) {
    cb.addReq("ssession", "make", {});
  };
        
  /**
   * Add a request to the `ServerSession` object.
   *
   * @private
   *
   * @param {String} key
   * @param {baja.comm.Callback} cb
   */
  ServerSession.prototype.addReq = function (key, cb) {
    return ServerSession.addReq(key, cb, { id: this.$id });
  };
  
  /**
   * Add a request to the `ServerSession` object.
   *
   * @private
   *
   * @param {String} key
   * @param {baja.comm.Callback} cb
   * @param arg call argument.
   */
  ServerSession.addReq = function (key, cb, arg) {

    // Please note, all messages sent at the Server Session layer are queued
    // when committed. This to try and keep the messaging in a nice tight order.
    cb.addReq("ssession", key, arg, /*queueOnCommit*/true);
    return arg;
  };
  
  /**
   * Add an event handler for an `eventHandlerId` to this `ServerSession`.
   * 
   * This is used to listen to events from the `ServerSession`.
   *
   * @private
   *
   * @param {String} eventHandlerId
   * @param {Function} eventHandler
   */
  ServerSession.addEventHandler = function (eventHandlerId, eventHandler) {
    eventHandlers[eventHandlerId] = eventHandler;
  };
  
  /**
   * Remove an event handler from the `ServerSession`.
   *
   * @private
   *
   * @param {String} eventHandlerId
   */
  ServerSession.removeEventHandler = function (eventHandlerId) {
    if (eventHandlers.hasOwnProperty(eventHandlerId)) {
      delete eventHandlers[eventHandlerId];
    }
  };
  
  /**
   * Return an event handler via its `eventHandlerId`.
   *
   * @private
   *
   * @param {String} eventHandlerId
   * @returns {Function} event handler or null if cannot be found.
   */
  ServerSession.findEventHandler = function (eventHandlerId) {
    return eventHandlers.hasOwnProperty(eventHandlerId) ? eventHandlers[eventHandlerId] : null;
  };
  
  return ServerSession;
});