/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/boxcs/SyncOp
 * @private
 */
define([
  "bajaScript/comm",
  "bajaScript/baja/comm/Callback" ], function (
    baja,
    Callback) {
  
  "use strict";
  
  var subclass = baja.subclass,
      BaseBajaObj = baja.BaseBajaObj;
  
  /**
   * Base SyncOp.
   *
   * All other SyncOps extend from this constructor.
   *
   * @class
   * @name SyncOp
   * @extends BaseBajaObj
   * @private
   */ 
  function SyncOp() {
    this.$arg = null;
  }
  
  subclass(SyncOp, BaseBajaObj); 
  
  // The unique identifier for the SyncOp  
  SyncOp.id = "";
  
  /**
   * Return the id for the SyncOp.
   *
   * The id is used for encoding and decoding a SyncOp.
   *
   * @private
   *
   * @returns {String}
   */
  SyncOp.prototype.getId = function () {
    return this.constructor.id;
  };
    
  /**
   * Perform a syncTo network call.
   *
   * @private
   *
   * @param space the Component Space we're syncing too.
   * @param {baja.comm.Callback} cb the callback handler.
   */    
  SyncOp.prototype.syncTo = function (space, cb) {

    // Create BSON SyncOp data structure
    var sync = {
          nm: "sync",
          ver: 1.0,
          ops: [ this.$arg ]
        },
        failedCalled = false,
        syncToResp,
        syncToCb;
    
    // TODO: at some point, we'll want to batch up the syncTo requests and have one syncFrom request
    // for greater efficiency. At the same time, we should also try and introduce the idea
    // of unsolicited messages that we'll subscribe too. All of the syncFrom events would then come from
    // unsolicited messages that have been registered for with a topic name. This sets us up for eventually
    // using a WebSocket instead of constant HTTP request/responses. When receiving a BOX frame with unsolicited
    // messages, the unsolicited messages should be processed first before dealing with the responses
    
    // Only call the ok if fail hasn't been called on the syncTo handler
    cb.addOk(function (ok/*, fail, resp*/) {
      if (!failedCalled) {
        // Pass results of the committed syncOps into the result of the original callback
        ok(syncToResp);      
      }
    });

    syncToCb = new Callback(baja.ok, cb.fail, cb.getBatch());
    
    syncToCb.addOk(function (ok, fail, resp) {
      // Get results of syncTo (which are the results of the committed syncOps)
      syncToResp = resp;
      ok(resp);
    });

    // Mark that fail has been called
    syncToCb.addFail(function (ok, fail, err) {
      failedCalled = true;
      fail(err);
    });
     
    // syncTo  
    baja.comm.serverHandlerCall(space, "syncTo", sync, syncToCb);

    // syncFrom
    space.getCallbacks().poll(cb);
  };
  
  return SyncOp;
});
