/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @module baja/boxcs/SetOp
 * @private
 */
define(["bajaScript/bson",
        "bajaScript/baja/boxcs/SyncOp"], 
        function (baja, SyncOp) {
  
  "use strict";
  
  var subclass = baja.subclass,
      callSuper = baja.callSuper,
      
      serverDecodeContext = baja.$serverDecodeContext,
      importUnknownTypes = baja.bson.importUnknownTypes,
      bsonDecodeValue = baja.bson.decodeValue;
  
  /**
   * @class Set SyncOp
   *
   * @name SetOp
   * @extends SyncOp
   * @inner
   * @private
   *
   * @param comp the Component the set is happening upon.
   * @param {Array} propPath an array of Property names for the set.
   * @param val the value being used in the set.
   */ 
  function SetOp(comp, propPath, val) {
    callSuper(SetOp, this, arguments);
    
    // Encode argument to a data structure
    var a = {
      nm: this.getId(),
      h: comp.getHandle(),
      n: propPath.reverse().join("/"),
      b: baja.bson.encodeValue(val) // Encode the value to BSON
    };
    
    this.$arg = a;  
  }
  
  subclass(SetOp, SyncOp); 
  
  SetOp.id = "s";
  
  /**
   * Decode and commit the SyncOp.
   *
   * @private
   *
   * @param comp the Component the set is happening on.
   * @param sp the syncOp data structure to be decoded.
   */
  SetOp.decodeAndCommit = function (comp, sp) {  
    if (comp !== null) {  
      // Decode the value and call 'set'
      var names = sp.n.split("/"),
          displayName = sp.dn,
          display = sp.b.d,
          target = comp,
          timestamp = sp.ts,
          n = null,
          i;
          
      for (i = 0; i < names.length; ++i) {
        if (n !== null) {
          target = target.get(n);
        }
        n = names[i];
      }
      
      return importUnknownTypes(sp.b)
        .then(function () {      
          // Set the desired target      
          target.set({
            "slot": n,
            "value": bsonDecodeValue(sp.b, serverDecodeContext), 
            "cx": { 
              commit: true, 
              serverDecode: true, 
              syncStructVals: true, 
              displayName: displayName, 
              display: display,
              timestamp: timestamp
            }
          });
        });
    }
  };
  
  return SetOp;
});