/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/webChart/rc/command/DialogWizardCommand
 */
define(['jquery', 'Promise', 'baja!', 'lex!', 'dialogs', 'bajaux/commands/Command', 'baja!' + 'baja:Boolean,' + 'baja:AbsTime,' + 'baja:AbsTimeRange,' + 'baja:TimeRange,' + 'baja:Weekday,' + 'baja:User,' + 'webChart:WebChartTimeRange,' + 'bql:DynamicTimeRangeType'], function ($, Promise, baja, lex, dialogs, Command) {
  "use strict";

  /**
   * DialogWizardCommand helps developers create field editors that allow users to select data
   *
   * @class
   * @alias module:nmodule/webChart/rc/command/DialogWizardCommand
   * @extends module:bajaux/commands/Command
   */

  var DialogWizardCommand = function DialogWizardCommand(chartWidget) {
    var lexParams = this.lexParams(),
        that = this;
    Command.call(this, {
      module: lexParams.module,
      lex: lexParams.lex,
      func: function func() {
        return that.resolveTab().then(function (tab) {
          return that.dialogCycle(tab);
        });
      }
    });
  };

  DialogWizardCommand.prototype = Object.create(Command.prototype);
  DialogWizardCommand.prototype.constructor = DialogWizardCommand;

  /**
   * Override resolveTab to resolve the first tab or tabs. If an array of tabs is returned, then
   * a TabbedEditor will be used to display tabs in a TabbedEditor for the user to complete.
   * @returns {*|Promise} An optional promise that if resolved to a tab will open another dialog when the
   * prior dialog completes.
   */
  DialogWizardCommand.prototype.dialogCycle = function (tab) {
    var that = this;
    if (!tab) {
      return; //end cycle
    }

    var dialog = dialogs.showOkCancel({
      title: tab.title,
      content: '<div class="DialogWizardCommand-Container"/>'
    }),
        dom = dialog.content().children(".DialogWizardCommand-Container");

    //initialize resolved tab
    tab.initialize(dom).then(function () {
      return tab.load();
    });

    //handle Ok
    return dialog.ok(function () {
      return Promise.resolve(that.handleRead(tab)).then(function (anotherTab) {
        if (!anotherTab) {
          return; //wizard complete
        }

        return Promise.resolve(that.dialogCycle(anotherTab));
      });
    }).cancel(function () {
      return tab.destroy();
    }).promise();
  };

  /**
   * Override resolveTab to resolve the first tab for the content.
   * @return {Promise}
   */
  DialogWizardCommand.prototype.resolveTab = function () {
    return Promise.resolve(null);
  };

  /**
   * Resolve a new tab if you want to continue the wizard, resolve a Falsy value to stop the
   * dialog wizard.
   * @param tab
   * @returns {*}
   */
  DialogWizardCommand.prototype.handleRead = function (tab) {
    return Promise.resolve("");
  };

  /**
   * Subclass Override to set lexicon parameters
   * @returns {Object}
   */
  DialogWizardCommand.prototype.lexParams = function () {
    return {};
  };

  DialogWizardCommand.prototype.options = function () {
    return baja.Facets.make();
  };

  /**
   * Override to tell Command what to do with selected results.
   * When work is complete, resolve a promise.
   * @param {enumSet} enumSet
   * @returns {Promise}
   */
  DialogWizardCommand.prototype.read = function (enumSet) {
    return Promise.resolve("");
  };

  return DialogWizardCommand;
});
