/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/fe/UxNodeSelectionEditor
 */
define(['baja!', 'jquery', 'Promise', 'd3', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'dialogs', 'underscore', 'bajaux/events', 'bajaux/dragdrop/dragDropUtils', 'nmodule/analytics/rc/report/util/reportUtils', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/analytics/rc/report/fe/AnalyticNavTree', 'nmodule/analytics/rc/report/fe/GroupNameFE', 'nmodule/webEditors/rc/fe/fe', 'hbs!nmodule/analytics/rc/report/templates/UxNodeSelection', 'nmodule/analytics/rc/report/util/AnalyticGroupTreeNode', 'nmodule/analytics/rc/report/util/AnalyticNodeTreeNode', 'nmodule/analytics/rc/report/util/reportWidgetEvents', 'nmodule/webEditors/rc/wb/menu/CommandGroupContextMenu', 'nmodule/webEditors/rc/wb/mixin/ContextMenuSupport', 'nmodule/webEditors/rc/wb/menu/menuAgents', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/js/rc/log/Log', 'lex!analytics', 'baja!analytics:AnalyticNodeGroup,analytics:AnalyticNode', 'css!nmodule/analytics/rc/report/styles/uxstyles'], function (baja, $, Promise, d3, Widget, subscriberMixIn, dialogs, _, events, dragDropUtils, reportUtils, BaseEditor, AnalyticNavTree, GroupNameFE, fe, uxNodeSelection, AnalyticGroupTreeNode, AnalyticNodeTreeNode, reportWidgetEvents, CommandGroupContextMenu, addContextMenuSupport, menuAgents, feDialogs, Log, lexicon, types) {

    'use strict';

    var lex = lexicon[0];
    var MENU_MODULE_ID = 'nmodule/analytics/rc/report/util/analyticGroupMenuAgent';
    var colorMap = {};
    var logWarning = _.partial(Log.logMessage, 'webEditors/rc/util', Log.Level.WARNING);

    /**
     * @class
     * @alias module:nmodule/analytics/rc/report/fe/UxNodeSelectionEditor
     * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
     */
    var UxNodeSelectionEditor = function UxNodeSelectionEditor() {
        BaseEditor.apply(this, arguments);
        subscriberMixIn(this);
        this.$registerContextMenu();
    };
    /**
     * Attach the parent's prototye and change the constructor and overidden methods
     * @type {BaseEditor}
     */
    UxNodeSelectionEditor.prototype = Object.create(BaseEditor.prototype);

    // Setting the constructor
    UxNodeSelectionEditor.prototype.constructor = UxNodeSelectionEditor;

    var builder = function builder(ed, index, group, dispOrd) {
        var jq = ed.jq();
        var elem = $(".nodefe-container", jq);
        var nodeP = ed.$getJSONNode(group, index);
        var prop = ed.properties();
        return nodeP.then(function (jsonNode) {
            var nodeName = "ux-node";
            elem.append('<div id="' + nodeName + index + '">');
            return fe.buildFor({
                value: jsonNode,
                dom: elem.find("#" + nodeName + index),
                type: AnalyticNavTree,
                properties: { showColorSelector: prop.getValue("showColorSelector"),
                    dispOrd: dispOrd,
                    wIndex: index
                },
                formFactor: 'mini'
            }).then(function (ced) {
                ced.jq().on(events.MODIFY_EVENT, function (evt) {
                    ed.setModified(true);
                });
                return Promise.resolve(ced);
            });
        }).error(function (r) {
            baja.err(r);
        });
    };
    /**
     *  Register the context menu agent
     */
    UxNodeSelectionEditor.prototype.$registerContextMenu = function () {
        var that = this,
            menuAgentModuleId = MENU_MODULE_ID;
        var reg1 = menuAgents.register("analytics:AnalyticNodeGroup", menuAgentModuleId);
        Promise.all([reg1]).then(function () {
            addContextMenuSupport(that);
        });
    };
    /**
     * Initalize the Node selection FE with blank node group place holder
     * @param dom
     */
    UxNodeSelectionEditor.prototype.doInitialize = function (dom) {
        var that = this;
        that.$clickedElements = [];
        dom.css("min-height", "100%").css("height", "100%");
        var container = $("<div class='nodefe-container'/>");
        dom.append(container);
        container.on('dragover dragenter', function (e) {
            e.preventDefault();
        }).on('drop', function (e) {
            var clipboard = e.originalEvent.dataTransfer;
            dragDropUtils.fromClipboard(clipboard).then(function (envelope) {
                return envelope.toValues();
            }).then(function (values) {
                that.$createNewGroups(values).then(function () {
                    that.setModified(true);
                }).catch(function (err) {
                    if (typeof err === "function") {
                        err();
                    }
                });
            });
            return false;
        });
        // Set the limits for Groups and Nodes.
        var props = that.properties(),
            groupLimit = props.getValue("groupLimit"),
            nodeLimit = props.getValue("nodeLimit"),
            groupLimitMsg = props.getValue("groupLimitMessage") || "";
        this.groupLimit = groupLimit || -1;
        this.nodeLimit = nodeLimit || -1;
        this.groupLimitMsg = groupLimitMsg;
        that.jq().on(reportWidgetEvents.DELETE_GROUP_REQUEST, function (event, root) {
            return UxNodeSelectionEditor.prototype.doLoad.apply(that, [that.$root]).then(function () {
                that.setModified(true);
                return Promise.resolve(true);
            });
        });
        return BaseEditor.prototype.doInitialize.apply(this, arguments);
    };

    /**
     *
     * @param {baja.Folder} folder a `baja:Folder` value, which contains a list
     * of 'analytics:AnalyticNodeGroup'
     */
    UxNodeSelectionEditor.prototype.doLoad = function (root) {
        var that = this;
        that.$root = root;
        that.jq().find(".nodefe-container").empty();
        if (root.getType().is("baja:Folder")) {
            //Get all the Groups underneath.
            var npList = [];
            var groups = root.getSlots().is("analytics:AnalyticNodeGroup").toArray();
            var len = groups.length;
            len = that.groupLimit === -1 || len < that.groupLimit ? len : that.groupLimit;
            _.each(groups, function (g, i) {
                var group = root.get(g);
                var color = group.getSeriesColor();
                colorMap[color] = (colorMap[color] || 0) + 1;
                var nodes = group.getSlots().properties().is("analytics:AnalyticNode").toValueArray();
                if (nodes.length === 1) {
                    var navNode = nodes[0],
                        dn = navNode.getDisplayName();
                    npList.push(reportUtils.getDisplayNodePath(baja.SlotPath.unescape(dn), baja.Ord.make(navNode.getNodePath())).then(function (dispOrd) {
                        return builder(that, i, group, dispOrd);
                    }));
                } else {
                    npList.push(builder(that, i, group));
                }
            });
            return Promise.all(npList).then(function () {
                that.$registerMouseDownEvent(that.jq());
                return Promise.resolve(root);
            });
        }
        return Promise.reject("Unable to load Node selector");
    };

    /**
     * Read the value of the node
     * @param root
     * @returns {*}
     */
    UxNodeSelectionEditor.prototype.doRead = function (root) {
        return this.getChildEditors().readAll();
    };
    /**
     *
     */
    UxNodeSelectionEditor.prototype.getContextMenuSelector = function () {
        return "span.group-content";
    };

    UxNodeSelectionEditor.prototype.getSubject = function (dom) {
        var that = this,
            clickedGroupElems = that.$clickedElements,
            values = [];
        _.each(clickedGroupElems, function (groupElement) {
            var editor = groupElement.parent().data('widget');
            if (editor) {
                values.push(editor.value());
            }
        });
        return values;
    };
    /**
     *
     * @param navNodes
     */
    UxNodeSelectionEditor.prototype.$createNewGroups = function (navNodes) {
        var that = this;
        var root = this.$root;
        var existingGroups = root.getSlots().is("analytics:AnalyticNodeGroup").toValueArray();
        // Validation for the maximum number oof groups that can be created
        if (that.groupLimit !== -1 && (existingGroups.length >= that.groupLimit || navNodes.length > that.groupLimit)) {
            return Promise.reject(function () {
                dialogs.showOk({
                    title: lex.get("report.settings.error.title"),
                    content: that.groupLimitMsg
                }).ok(function () {
                    return Promise.resolve(root);
                });
            });
        }
        var promiseList = [],
            dMap = {};
        _.each(navNodes, function (navNode, index) {
            var nodeName = navNode.getNavDisplayName(),
                dn = baja.SlotPath.escape(nodeName);
            var existingGroup = null,
                groupExists = false;
            if ((existingGroup = root.get(dn)) !== null) {
                var existingNodes = existingGroup.getSlots().properties().is("analytics:AnalyticNode").toValueArray();
                groupExists = existingNodes.length === 1 && existingNodes[0].getNodePath().equals(navNode.getNavOrd());
            }
            if (!groupExists) {
                var nodeComponent = baja.$("analytics:AnalyticNode", {
                    nodePath: baja.Ord.make(navNode.getNavOrd())
                }),
                    group = baja.$("analytics:AnalyticNodeGroup");
                group.setGuid(reportUtils.getRandomUuid(index * 1000));
                var cp = reportUtils.getDisplayNodePath(nodeName, navNode.getNavOrd()).then(function (nodeOrd) {
                    return group.add({
                        slot: dn + "?",
                        value: nodeComponent
                    }).then(function (prop) {
                        group.setSeriesColor(reportUtils.getColorEditorForGroup(colorMap));
                        return group.setFlags({
                            slot: prop.getName(),
                            flags: baja.Flags.HIDDEN
                        }).then(function () {
                            var addPromise = root.add({
                                slot: dn + "?",
                                value: group
                            }).then(function (prop) {
                                dMap[prop.getName()] = dn;
                                return Promise.resolve({ prop: prop, disp: nodeOrd });
                            });
                            return addPromise;
                        });
                    });
                });
                promiseList.push(cp);
            } else {
                logWarning(lex.get({ key: "report.warning.group.exists",
                    args: [dn] }));
            }
        });
        return Promise.all(promiseList).then(function (propList) {
            // Now update the display names
            var dnObj = root.get('displayNames'),
                method = "add";
            if (dnObj) {
                // Update the existing map
                var keys = dnObj.list();
                _.each(keys, function (key) {
                    dMap[key] = dnObj.get(key);
                });
                method = "set";
            }
            return root[method]({
                slot: "displayNames",
                value: baja.NameMap.make(dMap),
                flags: baja.Flags.HIDDEN | baja.Flags.READONLY | baja.Flags.TRANSIENT
            }).then(function () {
                var groups = root.getSlots().is("analytics:AnalyticNodeGroup").toValueArray();
                var buildList = [],
                    len = propList.length;
                for (var pIndex = 0; pIndex < len; pIndex++) {
                    var prop = propList[pIndex].prop,
                        dispOrd = propList[pIndex].disp;
                    buildList.push(builder(that, groups.length - len + pIndex, root.get(prop), dispOrd));
                }
                return Promise.all(buildList).then(function (edList) {
                    for (var edIndex = 0; edIndex < edList.length; edIndex++) {
                        var edRoot = edList[edIndex].jq();
                        that.$registerMouseDownOnGroup($('span.group-content', edRoot));
                    }
                    return Promise.resolve(true);
                });
            });
        });
    };
    /**
     * Register right click event handler
     */
    UxNodeSelectionEditor.prototype.$registerMouseDownEvent = function (parentElement) {
        // Now give a right click flag
        var that = this;
        var groups = $('span.group-content', parentElement);
        _.each([groups], function (groupElem) {
            that.$registerMouseDownOnGroup(groupElem);
        });
    };
    /**
     * Register the right-click on individual elements
     */
    UxNodeSelectionEditor.prototype.$registerMouseDownOnGroup = function (elem) {
        var that = this,
            selectedClass = 'selected-content';
        elem.mousedown(function (event) {
            switch (event.which) {
                case 1:
                case 3:
                    if (event.ctrlKey) {
                        that.$clickedElements.push($(this));
                    } else {
                        if (event.which === 3) {
                            if (!$(this).hasClass(selectedClass)) {
                                that.$clickedElements.push($(this));
                            } else {
                                $(this).toggleClass(selectedClass);
                            }
                        } else {
                            that.$clickedElements = [$(this)];
                            $('.' + selectedClass, that.jq()).toggleClass(selectedClass);
                        }
                    }
                    $(this).toggleClass(selectedClass);
                    break;
                default:
                    that.$clickedElements = [];
            }
        });
    };
    /**
     * Get the JSON Node for the node group
     * @param group
     */
    UxNodeSelectionEditor.prototype.$getJSONNode = function (group, index) {
        return group.loadSlots().then(function () {
            var groupUINode = new AnalyticGroupTreeNode(group, {});
            groupUINode.setIndex(index);
            return groupUINode.getKids().then(function () {
                var nodes = group.getSlots().is("analytics:AnalyticNode").toArray();
                var promiseList = [];
                for (var i = 0; i < nodes.length; i++) {
                    var node = group.get(nodes[i]);
                    // Right now only one level hierarchy is supported.
                    if (group.getFlags(node.getName()) !== baja.Flags.HIDDEN) {
                        promiseList.push(groupUINode.add(new AnalyticNodeTreeNode(node, {})));
                    }
                }
                return Promise.all(promiseList).then(function () {
                    return Promise.resolve(groupUINode);
                });
            });
        }).error(function () {
            return Promise.reject("Error Loading the node list");
        });
    };

    /**
     * Get the limit on the total number of groups that can be added to node selection FE
     * @param dom
     */
    UxNodeSelectionEditor.prototype.getGroupLimit = function () {
        return this.groupLimit;
    };
    /**
     * Get the limit on the total number of nodes that can be added to a given node in  node selection FE
     * @param dom
     */
    UxNodeSelectionEditor.prototype.getNodeLimit = function () {
        return this.nodeLimit;
    };
    /**
     * Get the current color map (in its actual state)
     * @param dom
     */
    UxNodeSelectionEditor.prototype.getColorMap = function () {
        return colorMap;
    };
    /**
     * Set the limit on the total number of groups that can be added to node selection FE
     * @param dom
     */
    UxNodeSelectionEditor.prototype.setGroupLimit = function (groupLimit) {
        this.groupLimit = groupLimit !== 0 ? groupLimit : -1;
    };
    /**
     * Set the limit on the total number of nodes that can be added to a given node in  node selection FE
     * @param dom
     */
    UxNodeSelectionEditor.prototype.setNodeLimit = function (nodeLimit) {
        this.nodeLimit = nodeLimit !== 0 ? nodeLimit : -1;
    };

    /**
     * Initalize the Node selection FE with blank node group place holder
     * @param dom
     */
    UxNodeSelectionEditor.prototype.dropNode = function (dom) {};

    /**
     * Set all child editors readonly/writable.
     *
     * @param {Boolean} readonly
     * @returns {Promise}
     */
    UxNodeSelectionEditor.prototype.doReadonly = function (readonly) {
        return this.getChildEditors().setAllReadonly(readonly);
    };

    /**
     * Set all child editors enabled/disabled.
     *
     * @param {Boolean} enabled
     * @returns {Promise}
     */
    UxNodeSelectionEditor.prototype.doEnabled = function (enabled) {
        return this.getChildEditors().setAllEnabled(enabled);
    };

    /**
     * Destroy all child editors.
     *
     * @returns {Promise}
     */
    UxNodeSelectionEditor.prototype.doDestroy = function () {
        return this.getChildEditors().destroyAll();
    };

    return UxNodeSelectionEditor;
});
