/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * @file Custom time range field editor for analytic web chart
 * @author Upender Paravastu
 */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/fe/AnalyticReportTimeRangeFE
 */
define(['baja!', 'bajaux/Widget', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'Promise', 'jquery', 'underscore', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'bajaux/events', 'moment', 'dialogs', 'nmodule/analytics/rc/report/fe/AnalyticReportTimeRangeSelector', 'nmodule/analytics/rc/report/fe/AnalyticDayOfWeekEditor', 'nmodule/analytics/rc/chart/base/analyticEvents', 'nmodule/analytics/rc/report/util/reportUtils', 'hbs!nmodule/analytics/rc/report/templates/UxAnalyticReportTimeRange', 'lex!analytics', 'baja!analytics:AnalyticReportTimeRange,baja:AbsTime', 'css!nmodule/analytics/rc/report/styles/uxstyles', 'css!nmodule/js/rc/dialogs/dialogs'], function (baja, Widget, BaseEditor, Promise, $, _, fe, feDialogs, events, moment, dialogs, AnalyticReportTimeRangeSelector, AnalyticDayOfWeekEditor, analyticEvents, reportUtils, UxAnalyticReportTimeRange, lexicon, types) {

    'use strict';

    /**
     *  <pre> AnalyticReportPeriodSelectionFE </pre> is the field  editor for time range selection in
     *  analytics UX reports.
     *  It builds upon the functionality of the frozenEnum Editor and adds
     *  extra time range options with their respective handling
     * @class
     * @alias module:nmodule/analytics/rc/report/fe/AnalyticReportTimeRangeFE
     * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
     */

    var AnalyticReportPeriodSelectionFE = function AnalyticReportPeriodSelectionFE() {
        BaseEditor.apply(this, arguments);
    };
    var lex = lexicon[0];
    var MODIFY_EVENT = events.MODIFY_EVENT,
        CLICK_EVENT = "click";

    // Inheriting AnalyticDataEditorFE from BaseEditor
    AnalyticReportPeriodSelectionFE.prototype = Object.create(BaseEditor.prototype);
    // Setting the constructor
    AnalyticReportPeriodSelectionFE.prototype.constructor = AnalyticReportPeriodSelectionFE;

    /**
     * From the time range string get the dates
     * @param element
     */
    AnalyticReportPeriodSelectionFE.prototype.getDatesFromTimeRange = function (dateStr) {
        return reportUtils.getDatesFromTimeRange(dateStr);
    };
    /**
     * Initialize the time range editor
     * @param element
     */
    AnalyticReportPeriodSelectionFE.prototype.doInitialize = function (dom) {
        this.$eventsRegistered = false;
        dom.append(UxAnalyticReportTimeRange({}));
    };

    /**
     * This method takes in string format of the time range string and makes additional
     * time range info for 'from', 'current' , 'previous' and 'timeRange' options.
     * @param value
     */
    AnalyticReportPeriodSelectionFE.prototype.doLoad = function (value) {
        var that = this;
        var jq = that.jq();
        var properties = that.properties();

        var editors = [];
        var dowDom = ".dow-container";
        var showDOWInclusion = properties.getValue("showDOWInclusion");
        if (showDOWInclusion) {
            var funcDOW = function funcDOW() {
                return fe.buildFor({
                    dom: $(dowDom, jq),
                    value: value.getDow(),
                    type: AnalyticDayOfWeekEditor
                });
            };
            if (that.$dowFE) {
                that.$dowFE.then(function (ed) {
                    ed.destroy().then(function () {
                        that.$dowFE = funcDOW();
                    });
                });
            } else {
                that.$dowFE = funcDOW();
            }
            editors.push(that.$dowFE);
        }
        if (!that.$eventsRegistered) {
            that.$registerEvents();
            that.$eventsRegistered = true;
        }
        return Promise.join(editors, function () {
            // Populate the text boxe
            return that.$updateDateTime(value.getTimeRange());
        });
    };

    /**
     *  Register events
     */
    AnalyticReportPeriodSelectionFE.prototype.$registerEvents = function () {
        var that = this,
            jq = this.jq(),
            trLaunchEl = '.tr-select-launch';
        jq.on(MODIFY_EVENT, '.dow-container', function () {
            that.setModified(true);
            return false;
        });
        jq.off(CLICK_EVENT, trLaunchEl);
        jq.on(CLICK_EVENT, trLaunchEl, function () {
            var value = that.value();
            var stEndStr = value.getMetaData(),
                stEndJSON = JSON.parse(stEndStr);
            feDialogs.showFor({
                value: value.getTimeRange(),
                properties: {
                    stIntrnl: stEndJSON.stIntrnl,
                    etIntrnl: stEndJSON.etIntrnl
                },
                title: lex.get("timeRange"),
                type: AnalyticReportTimeRangeSelector,
                formFactor: 'mini'
            }).then(function (reportTimeRangeSelectorVal) {
                if (reportTimeRangeSelectorVal) {
                    var val = that.value();
                    val.setTimeRange(reportTimeRangeSelectorVal.tag);
                    that.setModified(true);
                    that.$updateDateTime(reportTimeRangeSelectorVal.tag);
                }
            });
        });
    };

    /**
     * Update the date time string
     * @param timeRangeStr
     */
    AnalyticReportPeriodSelectionFE.prototype.$updateDateTime = function (timeRangeStr) {
        var that = this,
            jq = that.jq();
        return that.getDatesFromTimeRange(timeRangeStr).then(function (timerange) {
            var startTime = timerange.getStartTime(),
                endTime = timerange.getEndTime(),
                format = timerange.getFormat();
            // If time range is selected as 'all', then the end_time is set to baja's 'now', this is only for visual purposes.
            var timeRangeToValidate = 'all, thisWeek, thisMonth, thisYear';
            if (timeRangeToValidate.indexOf(timeRangeStr) !== -1) {
                endTime = baja.AbsTime.now();
            }
            var st = that.$format(startTime, format),
                et = that.$format(endTime, format);
            $(".start-end-value", jq).html(lex.getSafe({
                key: "report.settings.timeformat",
                args: [that.getTimeRangeDisplay(timeRangeStr), st, et]
            }));
            var val = that.value();
            var stEndJSON = {
                st: st,
                et: et,
                stIntrnl: startTime.encodeToString(),
                etIntrnl: endTime.encodeToString()
            };
            val.setMetaData(JSON.stringify(stEndJSON));
            return Promise.resolve(val);
        });
    };
    /**
     * Show the display string for Time Range
     * @param timeRangeStr
     * @returns {String|Promise.<String>}
     */
    AnalyticReportPeriodSelectionFE.prototype.getTimeRangeDisplay = function (timeRangeStr) {
        var trEnum = null,
            typeSpec = "analytics:UxAnalyticTimeRangeType";
        if (timeRangeStr.indexOf("from") === 0) {
            trEnum = baja.$(typeSpec).make("from");
        } else if (timeRangeStr.indexOf("current") === 0) {
            trEnum = baja.$(typeSpec).make("current");
        } else if (timeRangeStr.indexOf("previous") === 0) {
            trEnum = baja.$(typeSpec).make("previous");
        } else if (timeRangeStr === "yesterday" || timeRangeStr === "yearToDate" || timeRangeStr === "monthToDate" || timeRangeStr === "weekToDate" || timeRangeStr === "today" || timeRangeStr === "thisWeek" || timeRangeStr === "thisMonth" || timeRangeStr === "thisYear" || timeRangeStr === "lastWeek" || timeRangeStr === "lastMonth" || timeRangeStr === "lastYear" || timeRangeStr === "all") {
            trEnum = baja.$(typeSpec).make(timeRangeStr);
        } else {
            trEnum = baja.$(typeSpec).make("timeRainge");
        }
        return trEnum.getDisplayTag();
    };

    /**
    * Returns the formatted time stamp.
    */
    AnalyticReportPeriodSelectionFE.prototype.$format = function (timestamp, format) {
        var jsDate = timestamp.getJsDate();
        return moment(jsDate).format(format);
    };
    /**
     * read the value from the time range and time range info
     * @param element
     */
    AnalyticReportPeriodSelectionFE.prototype.doRead = function () {
        var that = this;
        return this.getChildEditors().readAll().spread(function (dowVal) {
            var timeRangeConfig = that.value();
            if (dowVal) {
                timeRangeConfig.setDow(dowVal);
            }
            return Promise.resolve(timeRangeConfig);
        });
    };

    return AnalyticReportPeriodSelectionFE;
});
