/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/container/base/AnalyticsBaseReport
 */
define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'Promise', 'jquery', 'd3', 'dialogs', 'underscore', 'nmodule/webEditors/rc/fe/fe', 'nmodule/analytics/rc/util/analyticsUtil', 'nmodule/analytics/rc/report/container/builder/AnalyticsReportBuilder', 'nmodule/analytics/rc/chart/base/AnalyticsMultiNodeDataModel', 'nmodule/analytics/rc/report/base/JSONAnalyticReportCfgAreaMixin', 'nmodule/analytics/rc/chart/utils/AnalyticsDataUtils', 'nmodule/analytics/rc/chart/base/analyticEvents', 'nmodule/analytics/rc/report/util/reportWidgetEvents', 'nmodule/analytics/rc/report/util/reportConstants', 'nmodule/analytics/rc/util/ModelFactory', 'bajaux/commands/Command', 'nmodule/webChart/rc/grid/GridEditor', 'bajaux/events', 'hbs!nmodule/analytics/rc/report/templates/report/analyticsBaseRptTmpl', 'lex!baja,analytics', 'css!nmodule/analytics/rc/report/styles/reportStyles', 'css!nmodule/analytics/rc/report/styles/printReport'], function (baja, Widget, subscriberMixIn, Promise, $, d3, dialogs, _, fe, analyticsUtil, AnalyticsReportBuilder, AnalyticsMultiNodeDataModel, JSONAnalyticReportCfgAreaMixin, AnalyticsDataUtils, analyticEvents, reportWidgetEvents, reportConstants, ModelFactory, Command, GridEditor, events, reportBaseTemplate, lexicons) {

  "use strict";

  var dashboard = "dashboard",
      dashboardReportData = "dashboardReportData";
  var lex = lexicons[1];

  /**
   * @class
   * @alias module:nmodule/analytics/rc/report/container/base/AnalyticsBaseReport
   * @extends module:bajaux/Widget
   */
  var AnalyticsBaseReport = function AnalyticsBaseReport() {
    var that = this;
    Widget.apply(this, arguments);

    that.$widgetBar = new GridEditor();
    addDashboardProperties(that);
    subscriberMixIn(this);
  };

  AnalyticsBaseReport.prototype = Object.create(Widget.prototype);
  AnalyticsBaseReport.prototype.constructor = AnalyticsBaseReport;

  /**
   * @private
   * @param {event}
   */
  var resizeTableContainer = function resizeTableContainer(event) {
    var that = event.data.report;
    var tableContainer = that.jq().find('.masterTableContainer');
    var isVisible = tableContainer.hasClass('isToggled');
    var tableHeight = "0%",
        chartHeight = reportConstants.REPORT_MAX_HEIGHT;
    if (isVisible) {
      tableHeight = reportConstants.REPORT_TABLE_HEIGHT;
      chartHeight = reportConstants.REPORT_CHART_HEIGHT;
      that.jq().find('.toggleTableContainer').addClass('icon-icons-x16-restoreWhite').removeClass('icon-icons-x16-maximizeWhite');
    } else {
      that.jq().find('.toggleTableContainer').removeClass('icon-icons-x16-restoreWhite').addClass('icon-icons-x16-maximizeWhite');
    }
    tableContainer.toggleClass('isToggled').animate({ "height": tableHeight });
    var chartContainer = that.jq().find('.chartContainer');
    chartContainer.animate({ "height": chartHeight }, 400, function () {
      that.jq().find('.chartContainer').trigger(analyticEvents.CHART_EXPANDED);
    });
  };

  /**
   *
   */
  var resizeNavContainer = function resizeNavContainer(event) {
    var that = event.data.report;
    var controlContainer = that.jq().find('.controlContainer-parent');
    var isVisible = controlContainer.hasClass('isToggled');
    var controlContainerWidth = "0%",
        dataContainerWidth = reportConstants.REPORT_MAX_WIDTH;
    if (isVisible) {
      controlContainerWidth = reportConstants.REPORT_CONFIG_WIDTH;
      dataContainerWidth = reportConstants.REPORT_CONTENT_WIDTH;
      $('.controlContainer').children().show();
      $('.reportTitle').show();
      that.jq().find('.toggleNavContainer').removeClass('icon-icons-x16-restoreWhite').addClass('icon-icons-x16-maximizeWhite');
    } else {
      $('.controlContainer').children().hide();
      $('.reportTitle').hide();
      that.jq().find('.toggleNavContainer').removeClass('icon-icons-x16-maximizeWhite').addClass('icon-icons-x16-restoreWhite');
    }
    controlContainer.toggleClass('isToggled').animate({ "width": controlContainerWidth });
    that.jq().find('.dataContainer').animate({ "width": dataContainerWidth }, 400, function () {
      that.jq().find('.chartContainer').trigger(analyticEvents.CHART_EXPANDED);
    });
  };

  /**
   *
   * @param e
   */
  var verticalDragBarMouseDownHandler = function verticalDragBarMouseDownHandler(e) {
    var that = e.data.report;
    e.preventDefault();
    that.isDragged = true;
    var verticalGhostBar = $('<div>', {
      class: 'verticalGhostBar',
      css: {
        top: Math.abs(e.pageY - that.jq().find('.dataContainer').offset().top) + 2
      }
    }).appendTo(that.jq().find('.dataContainer'));

    that.jq().find('.dataContainer').on('mousemove', function (e) {
      verticalGhostBar.css('top', Math.abs(e.pageY - that.jq().find('.dataContainer').offset().top) + 2);
    });
  };

  /**
   *
   * @param e
   */
  var dataContainerMouseUpEventHandler = function dataContainerMouseUpEventHandler(e) {
    var that = e.data.report;
    if (that.isDragged) {
      var curPos = Math.abs(e.pageY - that.jq().find('.dataContainer').offset().top);
      var currentHeight = that.jq().height();
      var chartOccupancyPercentage = Math.ceil(curPos / currentHeight * 100);
      var tableOccupancyPercentage = 100 - chartOccupancyPercentage;
      chartOccupancyPercentage = chartOccupancyPercentage - 2;
      tableOccupancyPercentage = tableOccupancyPercentage - 2;

      that.jq().find('.verticalGhostBar').remove();

      var tableContainer = that.jq().find('.masterTableContainer');
      tableContainer.toggleClass('isToggled').animate({ "height": "" + tableOccupancyPercentage + "%" });
      var chartContainer = that.jq().find('.chartContainer');
      chartContainer.animate({ "height": "" + chartOccupancyPercentage + "%" }, 400, function () {
        that.jq().find('.chartContainer').trigger(analyticEvents.CHART_EXPANDED);
      });
      that.jq().find('.dataContainer').unbind('mousemove');
      that.isDragged = false;
    }
  };

  /**
   * Called when the widget is created. This adds any dashboard
   * specific properties to the chart widget.
   *
   * @inner
   * @private
   *
   * @param widget The chart widget instance to add the dasboard properties too.
   */
  function addDashboardProperties(widget) {
    widget.properties().add({ // Enabled if on a Dashboard.
      name: dashboard,
      value: false,
      hidden: true,
      transient: true,
      readonly: true
    }).add({ //-------------------------- *DASHBOARD-ONLY NOTE*
      name: dashboardReportData,
      value: null,
      dashboard: true
    }).add({
      name: "hideCommandBar",
      value: true,
      hidden: true,
      transient: true,
      readonly: true
    });
  }

  /**
   * Return true if the web chart is being loaded from a dashboard.
   *
   * @inner
   * @private
   *
   * @param widget The widget instance.
   * @returns {Boolean} Returns true if the chart widget is on a Dashboard.
   */
  function isOnDashboard(widget) {
    return widget.properties().getValue(dashboard, false);
  }

  /**
   * Attempt to resolve the dashboard clear command. If the widget
   * isn't on a dashboard, the returned promise will resolve with no
   * value.
   *
   * @inner
   * @private
   *
   * @param widget The widget instance.
   * @returns {Promise} A promise that will resolve to a clear command.
   */
  function resolveClearCmd(widget) {
    return new Promise(function (resolve, reject) {
      // If being used on a Dashboard then resolve the dashboard
      // resources before initializing the widget bar.
      if (isOnDashboard(widget)) {
        require(["nmodule/dashboard/rc/dashboard"], function (db) {
          var clearCmd = db.makeClearCmd(widget);

          // Add to the widget and the widget bar.
          widget.getCommandGroup().add(clearCmd);

          resolve(clearCmd);
        }, reject);
      } else {
        resolve();
      }
    });
  }

  /**
   *
   * @param element
   * @returns {*}
   */
  AnalyticsBaseReport.prototype.doInitialize = function (element) {
    var that = this;
    element.html(reportBaseTemplate({
      reportName: _.escape(that.getReportName())
    }));
    that.isDragged = false;
    that.jq().find('.hideNavContainer').on('click', { report: this }, resizeNavContainer);
    that.jq().find('.hideTableContainer').on('click', { report: this }, resizeTableContainer);
    that.jq().find('.verticalDragBar').on('mousedown', { report: this }, verticalDragBarMouseDownHandler);
    that.jq().find('.dataContainer').on('mouseup', { report: this }, dataContainerMouseUpEventHandler);
    return baja.Ord.make("station:|service:analytics:AnalyticService").get().then(function (service) {
      that.$areaTag = service.getAreaTag();
      that.$oatTag = service.getOatTag();
      return Promise.resolve(true);
    });
  };

  /**
   *
   * @param source
   */
  AnalyticsBaseReport.prototype.doLoad = function (source) {
    var that = this;
    source = JSON.parse(that.properties().getValue(dashboardReportData)) || {};
    var reportWidgets = AnalyticsReportBuilder.buildReport(source, this);
    return reportWidgets.then(function (widgetMap) {
      that.$registerSettingsTriggerEvent(widgetMap);
      analyticsUtil.parseObjectForReport(source);
      return that.$doLoad(source, widgetMap.metaData).then(function () {
        var widgetBarDiv = that.jq().find(".widgetBar");
        return resolveClearCmd(that).then(function (clearCmd) {
          that.initCommands(widgetBarDiv, clearCmd);
          return Promise.resolve(true);
        });
      });
    });
  };

  /**
   * Upon click of 'Run Report' the changes should apply on the chart.
   * @param widgetMap
   * @returns {*}
   */
  AnalyticsBaseReport.prototype.$registerSettingsTriggerEvent = function (widgetMap) {
    var settingsWidget = widgetMap.settings;
    var that = this;
    if (settingsWidget) {
      settingsWidget.jq().on(reportWidgetEvents.DELETE_GROUP_REQUEST, function (e, grpComponent) {
        settingsWidget.jq().find('.baseline-enabler').prop('checked', false).trigger('change');
      });
      settingsWidget.jq().on(reportWidgetEvents.REPORT_CONFIG_CHANGED, function (event, jsonConfig, metaData) {
        var jsonStr = JSON.stringify(jsonConfig);
        that.properties().setValue(dashboardReportData, jsonStr);
        var jsonTemp = JSON.parse(jsonStr);
        analyticsUtil.parseObjectForReport(jsonTemp);
        that.getChartContainer().removeData();
        that.getTableContainer().removeData();
        that.$doLoad(jsonTemp, metaData);
      });
    }
  };

  /**
   * Empties the data area and put some text if data is unvailable, else returns the input data set.
   */
  AnalyticsBaseReport.prototype.renderEmptyForNoData = function (modelArr) {
    var that = this;
    if (_.every(modelArr, function (m) {
      return _.isEmpty(m.getAnalyticTrendArray());
    })) {
      var chartContainer = that.getChartContainer();
      var message = $("<div class='no-data-text'>"),
          text = lex.get("analytics.chart.nodata");
      message.text(text);
      chartContainer.unbind().empty().append(message);
      return Promise.reject(text);
    }
    return Promise.resolve(modelArr);
  };

  /**
   *
   * @param source
   */
  AnalyticsBaseReport.prototype.$doLoad = function (source, metaData) {
    var that = this;
    if (metaData) {
      that.jq().find(".timeRangeLabel").html(_.escape(metaData.timeRange.st + " - " + metaData.timeRange.et));
    }
    var promises = [];
    if (that.$chart) {
      promises.push(that.$chart.destroy());
    }
    if (that.$table) {
      promises.push(that.$table.destroy());
    }
    var allPromise = Promise.all(promises).then(function () {
      if (!_.isEmpty(source)) {
        source.aggMode = that.getAggMode();
        source.areaTag = that.$areaTag;
        source.oatTag = that.$oatTag;
        source.type = "report";
        source.reportName = that.getName();
        var dataModelPromise = AnalyticsDataUtils.getModel(that, source, {}),
            baselineModelPromise = AnalyticsDataUtils.getBaselineModel(that, source, {});
        return Promise.all([dataModelPromise, baselineModelPromise]).spread(function (inputModelArr, baselineModel) {
          var dataModel = inputModelArr[0],
              configModel = inputModelArr[1];
          if (that.isMultiPartSupported()) {
            // Right now we dont need baseline for Aggregation report
            return AnalyticsDataUtils.convertMultiPartData(that, dataModel).then(function (modelArray) {
              var obj;
              if (modelArray.length > 0) {
                obj = {};
                obj[modelArray[0].getSeriesName()] = modelArray[0].getAreaCfg();
                obj.useGrpName = true;
              }

              return that.renderEmptyForNoData(modelArray).then(function (modelArr) {
                return [modelArr, configModel, obj];
              });
            });
          } else {
            if (baselineModel) {
              dataModel.updateWithBaselineModel(baselineModel);
            }
            return AnalyticsDataUtils.getDataForMultiNode(dataModel).then(function (multiNodeModel) {
              var nodeModel = multiNodeModel.$nodeModel;
              var areaCfg = {};
              _.each(nodeModel, function (nodeCfg, key) {
                if (!nodeCfg.bl) {
                  areaCfg.useGrpName = false;
                  areaCfg[nodeCfg.guid] = nodeCfg.areaCfg;
                }
              });
              if (!that.isColorChoiceAllowed() && multiNodeModel.$nodeModel) {
                multiNodeModel.$nodeModel[_.keys(multiNodeModel.$nodeModel)[0]].color = undefined;
              }

              var returnPromise = that.renderEmptyForNoData(AnalyticsDataUtils.convertMultiNodeToTrendData(that, multiNodeModel, configModel)).then(function (modelArr) {
                return [modelArr, configModel, [areaCfg, false]];
              });
              return returnPromise;
            });
          }
        });
      } else {
        var message = "";
        if (that.$chart) {
          that.$chart.renderEmpty();
        }
        return Promise.resolve(message);
      }
    });
    var dataWidgetPromise = allPromise.then(function (values) {
      return Promise.all([fe.buildFor({
        dom: that.getChartContainer(),
        type: that.getChartType(),
        properties: {
          isCmdBarRequired: false,
          source: 'report',
          model: values[0],
          configModel: values[1],
          showLegend: that.showLegend(),
          showInterpolationStatus: source.interpolationStatus, // TODO: Need to move this piece of code to the configuration section of Reports.
          // Move the chartValue to the first index of the array.
          colKeys: AnalyticsDataUtils.moveKeyToFirstIndex(_.keys(source.combinationMap), source.chartValue)
        },
        value: source
      }).then(function (chart) {
        return chart;
      }), fe.buildFor({
        dom: that.getTableContainer(),
        type: that.getTableType(),
        properties: {
          isCmdBarRequired: false,
          source: 'report',
          xKeyFormatter: that.getTableKeyFormatter(),
          configModel: ModelFactory.newConfigModel(that.getTableType(), { exclusionList: [lex.get("table.interpolationStatus.columnText")] }),
          showInterpolationStatus: source.interpolationStatus // TODO: Need to move this piece of code to the configuration section of Reports.
        },
        value: {}
      }).then(function (table) {
        return table;
      })]).spread(function (chart, table) {
        that.$chart = chart;
        that.$table = table;
        var c = 'ux-fullscreen';
        chart.jq().removeClass(c);
        table.jq().removeClass(c);
        // For aggregation report, this if condition passes as it has multi parts involved.
        if (that.isMultiPartSupported()) {
          _.each(chart.series, function (series, index) {
            if (index !== 0) {
              return false;
            }
            series.data = that.processDataForTable(chart, source);
            return series;
          });
        } else {
          if (chart.series.length > 0) {
            chart.series[0].data = that.processDataForTable(chart, source);
          }
        }
        if (!_.isEmpty(chart.getModel())) {
          that.jq().find('.tableContainer').trigger(analyticEvents.CHART_CHANGED, [chart]);
        }
        // Once the chart and table are rendered, trigger a report area configuration changed event.
        that.getSettingsContainer().data('widget').jq().trigger(reportWidgetEvents.REPORT_AREA_VAL_LOADED, values[2]);
        return Promise.resolve([chart, table]);
      });
    }).catch(function (e) {
      baja.outln(e);
    });
    dialogs.showLoading(200, dataWidgetPromise);
    return dataWidgetPromise;
  };

  /**
   * This needs to be overridden by specific reports to handle data manipulation for table
   * @param chartDataModel
   * @param source
   */
  AnalyticsBaseReport.prototype.processDataForTable = function (chart, source) {
    return chart.series[0].data;
  };

  /**
   * Format the X Value
   * @returns {Function}
   */
  AnalyticsBaseReport.prototype.getTableKeyFormatter = function () {
    var that = this;
    return function (xVal, key) {
      if (that.$chart) {
        return that.$chart.getFormattedTooltipValue(xVal);
      }
      return xVal;
    };
  };

  AnalyticsBaseReport.prototype.getChartContainer = function () {
    return this.jq().find('.chartContainer');
  };

  AnalyticsBaseReport.prototype.getTableContainer = function () {
    return this.jq().find('.masterTableContainer');
  };

  /**
   * I am outside.
   * @returns {*}
   */
  AnalyticsBaseReport.prototype.getSettingsContainer = function () {
    return this.jq().find('.controlContainer');
  };

  /**
   * Get the list of settings applicable for Spectrum Chart
   * @returns {*}
   */
  AnalyticsBaseReport.prototype.getApplicableSettings = function () {
    var that = this;
    return {
      node: { req: true, value: "{}" },
      timeRange: { req: true, value: "today" },
      dataTag: { req: true, value: "n:history" },
      interval: { req: true, value: "", selected: false },
      rollup: { req: true, value: "", selected: false },
      aggregation: { req: true, value: "", selected: false },
      units: { req: true, value: baja.$("baja:Unit").DEFAULT },
      mdConfig: {
        req: true,
        value: reportConstants.DEFAULT_MISSING_DATA_CFG
      },
      hisTotEnabled: { req: true, value: false },
      interpolationStatus: { req: true, value: false },
      dataFilter: { req: true, value: "" },
      legendPosition: {
        req: true,
        value: reportConstants.DEFAULT_LEGEND_POSITION
      },
      configType: that.getConfigType()
    };
  };

  /**
   * Returns the configuration type of the chart.
   */
  AnalyticsBaseReport.prototype.getConfigType = function () {
    return "analytics:AnalyticReportConfiguration";
  };

  /**
   * Returns the requirement of the report to aggregate the MultiNode or MultiOrd Requests.
   * @returns {boolean}
   */
  AnalyticsBaseReport.prototype.getAggMode = function () {
    return false;
  };

  /**
   * Return the default ord scheme supported by the report.
   * Other reports can override the ord scheme.
   * @returns {string}
   */
  AnalyticsBaseReport.prototype.getOrdScheme = function () {
    return "analyticMultiTrend:";
  };

  /**
   * Some child reports might support multi part rollup and data tags.
   * @returns {boolean}
   */
  AnalyticsBaseReport.prototype.isColorChoiceAllowed = function () {
    return true;
  };

  AnalyticsBaseReport.prototype.isMultiPartSupported = function () {
    return false;
  };
  /**
   * Some child reports might support multi part rollup and data tags.
   * @returns {boolean}
   */
  AnalyticsBaseReport.prototype.getSettingsWidgetType = function () {
    return JSONAnalyticReportCfgAreaMixin;
  };

  /**
   * Returns the report name
   * @returns {string}
   */
  AnalyticsBaseReport.prototype.getName = function () {
    return "AnalyticsBaseReport";
  };

  AnalyticsBaseReport.prototype.initCommands = function (element, clearCmd) {
    var that = this;

    // A hack but as the chart widget implements its own widget bar, this is what we need to do.
    // if (clearCmd) {
    //     element.show();
    //     jqClear().show();
    // }


    function jqClear() {
      // return $(".js-tab-" + saveCmd.getId(), that.jq());
      return $(".ux-btn.clear-btn", that.jq());
    }

    function disableButtons() {
      return that.properties().getValue(dashboardReportData) === undefined || that.properties().getValue(dashboardReportData) === 'null' || false;
    }

    //Enable the clear command is enabled
    jqClear().prop("disabled", disableButtons());

    that.jq().on([events.ENABLE_EVENT, events.DISABLE_EVENT, events.READONLY_EVENT, events.WRITABLE_EVENT, events.INITIALIZE_EVENT, events.LOAD_EVENT, events.DESTROY_EVENT].join(' '), '*', false);

    that.jq().on([events.MODIFY_EVENT, reportWidgetEvents.REPORT_CONFIG_CHANGED].join(' '), function () {
      that.save().then(function (status) {
        jqClear().prop("disabled", false);
      });
    }).on([events.LOAD_EVENT].join(' '), function (e) {
      jqClear().parent().toggle(isOnDashboard(that));
    });

    jqClear().on('click', function (e) {
      e.preventDefault();
      var ok = function ok(status) {
        if (status && status[1] && status[1] === "yes") {
          jqClear().prop("disabled", true);
        } else {
          jqClear().prop("disabled", false);
        }
      };
      var fail = function fail() {
        jqClear().prop("disabled", false);
      };
      clearCmd.invoke().then(ok, fail);
    });
  };

  AnalyticsBaseReport.prototype.showLegend = function () {
    return false;
  };

  /**
   * Returns default ord scheme for all reports.
   * @returns {string}
   */
  AnalyticsBaseReport.prototype.getOrdScheme = function () {
    return "analyticMultiTrend:";
  };

  /**
   * Returns the base data model
   * @returns {*}
   */
  AnalyticsBaseReport.prototype.getDataModel = function () {
    return AnalyticsMultiNodeDataModel;
  };

  AnalyticsBaseReport.prototype.doDestroy = function () {
    return Promise.all([this.$chart.destroy(), this.$table.destroy()]);
  };

  return AnalyticsBaseReport;
});
