/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/command/AnalyticUngroupNodesCommand
 */
define(['baja!', 'lex!analytics', 'underscore', 'jquery', 'Promise', 'nmodule/analytics/rc/report/util/reportUtils', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/commands/ComponentEditorCommand', 'nmodule/analytics/rc/report/util/reportWidgetEvents'], function (baja, lexs, _, $, Promise, reportUtil, feDialogs, ComponentEditorCommand, reportWidgetEvents) {

    'use strict';

    var lex = lexs[0];

    ////////////////////////////////////////////////////////////////
    // Exports
    ////////////////////////////////////////////////////////////////

    /**
     * A command for grouping an analytic node
     * @class
     * @alias module:nmodule/analytics/rc/report/command/AnalyticUngroupNodesCommand
     * @extends module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
     * @param ed
     * @param component
     */
    var AnalyticUngroupNodesCommand = function AnalyticUngroupNodesCommand(ed, component) {
        var that = this;
        that.$groupComponents = [component];
        that.$ed = ed;
        ComponentEditorCommand.call(that, {}, component);
        that.$displayName = lex.get('report.commands.ungroup');
    };

    AnalyticUngroupNodesCommand.prototype = Object.create(ComponentEditorCommand.prototype);
    AnalyticUngroupNodesCommand.prototype.constructor = AnalyticUngroupNodesCommand;

    /**
     *
     * @param {baja.Component} comp
     * @param {baja.Slot} slot
     * @returns {Boolean} true if I can ungroup nodes
     */
    AnalyticUngroupNodesCommand.prototype.canPerformCommand = function (comp, slot) {
        return true;
    };

    /**
     * Shows a dialog for the component's currently configured slot name,
     * and renames the slot after the user clicks OK.
     *
     * @param {baja.Component} comp
     * @param {Array.<baja.Slot>} slots the slots to rename (*Note*: only the
     * first given slot will be renamed)
     * @returns {Promise}
     */
    AnalyticUngroupNodesCommand.prototype.performCommand = function (rootFolder, slots, params) {
        var that = this,
            groups = that.$groupComponents,
            addPromiseList = [],
            dMap = {};
        if (groups.length > 0) {
            _.each(groups, function (group, gIndex) {
                var nodes = group.getSlots().properties().is("analytics:AnalyticNode").toValueArray();
                _.each(nodes, function (n, nIndex) {
                    var repGroup = baja.$("analytics:AnalyticNodeGroup", {
                        guid: reportUtil.getRandomUuid(nIndex * 1000)
                    });
                    repGroup.setSeriesColor(reportUtil.getColorEditorForGroup(that.$ed.getColorMap()));
                    var p = repGroup.add({
                        slot: n.getDisplayName() + "?",
                        value: n.newCopy(true), // Make an exact copy
                        flags: baja.Flags.HIDDEN
                    }).then(function (hiddenProp) {
                        var obj = {};
                        obj[hiddenProp.getName()] = n.getDisplayName();
                        return repGroup.add({
                            slot: "displayNames",
                            value: baja.NameMap.make(obj)
                        }).then(function () {
                            return addGroupsToRootFolder(rootFolder, repGroup, n).then(function (prop) {
                                dMap[prop.getName()] = n.getDisplayName();
                                return Promise.resolve(prop);
                            });
                        });
                    });
                    addPromiseList.push(p);
                });
            });
            return Promise.all(addPromiseList).then(function (groupProperties) {
                var toBeDeletedGroups = {};
                var cleanUpPromises = [];
                _.each(groups, function (gp) {
                    toBeDeletedGroups[gp.getName()] = true;
                    cleanUpPromises.push(rootFolder.remove({
                        slot: gp.getName()
                    }));
                });
                return Promise.all(cleanUpPromises).then(function () {
                    var dnObj = rootFolder.get("displayNames"),
                        method = "add";
                    if (dnObj) {
                        var keys = dnObj.list();
                        _.each(keys, function (key, i) {
                            if (!dMap.hasOwnProperty(key) && !toBeDeletedGroups[key]) {
                                dMap[key] = dnObj.get(key).valueOf();
                            }
                        });
                        method = "set";
                    }
                    return rootFolder[method]({
                        slot: "displayNames",
                        value: baja.NameMap.make(dMap)
                    }).then(function () {
                        that.$ed.jq().trigger(reportWidgetEvents.DELETE_GROUP_REQUEST, rootFolder);
                        return Promise.resolve(true);
                    });
                });
            });
        }
        return Promise.resolve(false);
    };

    /**
     * Merge only if it is a rename command
     * @param comp
     * @param slot
     */
    AnalyticUngroupNodesCommand.prototype.merge = function (cmd) {
        if (cmd instanceof AnalyticUngroupNodesCommand) {
            this.$groupComponents = _.flatten([this.$groupComponents, cmd.$groupComponents]);
            if (reportUtil.hasAnyChildren(this.$groupComponents)) {
                this.setEnabled(true);
            }
            return this;
        }
        return null;
    };

    /**
     * Function is used to add groups to the root folder
     * @param rootFolder
     * @param repGroup
     * @param n
     * @returns {*}
     */
    function addGroupsToRootFolder(rootFolder, repGroup, n) {
        return rootFolder.add({
            slot: n.getName() + "?",
            value: repGroup
        });
    }

    return AnalyticUngroupNodesCommand;
});
