/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/analytics/rc/report/command/AnalyticNodeGroupCommand
 */
define(['baja!', 'lex!analytics,webEditors', 'underscore', 'jquery', 'Promise', 'nmodule/analytics/rc/report/util/reportUtils', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/commands/ComponentEditorCommand', 'nmodule/analytics/rc/report/util/reportWidgetEvents'], function (baja, lexs, _, $, Promise, reportUtil, feDialogs, ComponentEditorCommand, reportWidgetEvents) {

    'use strict';

    var lex = lexs[0],
        webEditorsLex = lexs[1];

    ////////////////////////////////////////////////////////////////
    // Exports
    ////////////////////////////////////////////////////////////////

    /**
     * A command for grouping an analytic node
     * @class
     * @alias module:nmodule/analytics/rc/report/command/AnalyticNodeGroupCommand
     * @extends module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
     * @param ed
     * @param component
     */
    var AnalyticNodeGroupCommand = function AnalyticNodeGroupCommand(ed, component) {
        var that = this;
        that.$groupComponents = [component];
        that.$ed = ed;
        ComponentEditorCommand.call(that, {}, component);
        that.$displayName = lex.get('report.commands.group');
    };

    AnalyticNodeGroupCommand.prototype = Object.create(ComponentEditorCommand.prototype);
    AnalyticNodeGroupCommand.prototype.constructor = AnalyticNodeGroupCommand;

    /**
     *
     * @param {baja.Component} comp
     * @param {baja.Slot} slot
     * @returns {Boolean} true if I can group nodes
     */
    AnalyticNodeGroupCommand.prototype.canPerformCommand = function (comp, slot) {
        return true;
    };

    /**
     * Shows a dialog for the component's currently configured slot name,
     * and renames the slot after the user clicks OK.
     *
     * @param {baja.Component} comp
     * @param {Array.<baja.Slot>} slots the slots to rename (*Note*: only the
     * first given slot will be renamed)
     * @returns {Promise}
     */
    AnalyticNodeGroupCommand.prototype.performCommand = function (node, slots, params) {
        var that = this,
            groups = that.$groupComponents;
        if (groups.length > 0) {
            var ref = groups[0];
            return feDialogs.showFor({
                title: lex.get('report.commands.group.title'),
                value: baja.SlotPath.unescape(ref.getName() || ""),
                formFactor: 'mini'
            }).then(function (groupName) {
                if (groupName === null) {
                    return;
                }
                if (groupName.trim() === '') {
                    throw new Error(webEditorsLex.get('enterName'));
                }
                groupName = baja.SlotPath.escape(groupName.trim());
                return that.regroup(node, groupName);
            }).catch(feDialogs.error);
        }
        return Promise.resolve(false);
    };
    /**
     *
     * @param node
     * @param slots
     * @param params
     */
    AnalyticNodeGroupCommand.prototype.regroup = function (rootFolder, groupName) {
        var that = this,
            groups = that.$groupComponents,
            addPromiseList = [];
        if (groups.length > 0) {
            var newGroupComp = baja.$("analytics:AnalyticNodeGroup", {
                guid: reportUtil.getRandomUuid(1000)
            }),
                grpSize = groups.length,
                dMap = {},
                rMap = {};
            newGroupComp.add({
                slot: "displayNames",
                value: baja.NameMap.make({})
            }).then(function (ngp) {
                _.each(groups, function (group, gIndex) {
                    var nodes = group.getSlots().properties().is("analytics:AnalyticNode").toValueArray();
                    if (grpSize === 1 && nodes.length === 1) {
                        return Promise.resolve(false);
                    }
                    _.each(nodes, function (n, nIndex) {
                        newGroupComp.setSeriesColor(reportUtil.getColorEditorForGroup(that.$ed.getColorMap()));
                        addPromiseList.push(newGroupComp.add({
                            slot: n.getName() + "?",
                            value: n.newCopy(true) // Make an exact copy
                        }).then(function (prop) {
                            dMap[prop.getName()] = n.getDisplayName();
                            return Promise.resolve(true);
                        }));
                    });
                });
                if (addPromiseList.length > 0) {
                    var displayNamesToBeDeleted = {};
                    return Promise.all(addPromiseList).then(function () {
                        return newGroupComp.set({
                            slot: "displayNames",
                            value: baja.NameMap.make(dMap)
                        }).then(function () {
                            var cleanUpPromises = [];
                            _.each(groups, function (gp) {
                                displayNamesToBeDeleted[gp.getName()] = true;
                                cleanUpPromises.push(rootFolder.remove({
                                    slot: gp.getName()
                                }));
                            });
                            cleanUpPromises.push(rootFolder.add({
                                slot: groupName + "?",
                                value: newGroupComp
                            }).then(function (ngp) {
                                rMap[ngp.getName()] = groupName;
                                return Promise.resolve(true);
                            }));
                            return Promise.all(cleanUpPromises).then(function () {
                                var dnObj = rootFolder.get("displayNames");
                                var keys = dnObj.list();
                                _.each(keys, function (key) {
                                    if (!displayNamesToBeDeleted[key]) {
                                        rMap[key] = dnObj.get(key).valueOf();
                                    }
                                });
                                return rootFolder.set({
                                    slot: "displayNames",
                                    value: baja.NameMap.make(rMap)
                                }).then(function () {
                                    that.$ed.jq().trigger(reportWidgetEvents.DELETE_GROUP_REQUEST, rootFolder);
                                    return Promise.resolve(true);
                                });
                            });
                        });
                    });
                }
            });
        }
        return Promise.resolve(false);
    };
    /**
     * Merge only if it is a rename command
     * @param comp
     * @param slot
     */
    AnalyticNodeGroupCommand.prototype.merge = function (cmd) {
        if (cmd instanceof AnalyticNodeGroupCommand) {
            this.$groupComponents = _.flatten([this.$groupComponents, cmd.$groupComponents]);
            if (this.$groupComponents.length > 1) {
                this.setEnabled(true);
            }
            return this;
        }
        return null;
    };

    return AnalyticNodeGroupCommand;
});
