/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/StringEditor
 */
define(['baja!', 'bajaux/Widget', 'jquery', 'underscore', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/StringEditor'], function (baja, Widget, $, _, BaseEditor, StringEditor) {

  'use strict';

  /**
  * A field editor for working with strings.
  *
  * It accepts the following Properties:
  *
  * - `datalist`: (string|object) a semicolon-separated list of strings to use
  *   to generate an HTML `datalist` element. Alternatively, it can be an
  *   object literal from values to display strings; the display strings will
  *   be used to create the `datalist`, but `doRead()` will resolve the actual
  *   value if it corresponds to the entered display string. Useful for enum
  *   values with display tags, etc.
  * - `max`: (number) max length for the input.
  * - `min`: (number) min length for the input.
  * - `multiLine`: (boolean) if true, editor will be a `textarea`.
  * - `mustBeSlotName`: (boolean) input value must be a valid slot name.
  * - `placeholder`: (string) if given, the `placeholder` attribute of the
  *   text input will be set.
  *
  * It accepts the following bajaux Properties:
  *
  * - `inputType`: (string) when not in multiline mode, set the `input` tag's
  *   `type` attribute. Defaults to `text`.
  *
  * @class
  * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
  * @alias module:nmodule/webEditors/rc/fe/baja/AnalyticsStringEditor
  *
  * @param {baja.Facets} [params.facets] if a semicolon-separated string
  * `datalist` facet is provided, a `<datalist>` element will be appended for
  * autocomplete purposes.
  */

  var TimeRangeStringLabelEditor = function TimeRangeStringLabelEditor(params) {
    var that = this;

    BaseEditor.call(that, $.extend(true, {
      keyName: 'AnalyticsStringEditor'
    }, {
      properties: {
        inputType: { value: 'label', hidden: true, readonly: true }
      }
    }, params));
  };

  TimeRangeStringLabelEditor.prototype = Object.create(StringEditor.prototype);
  TimeRangeStringLabelEditor.prototype.constructor = TimeRangeStringLabelEditor;

  /**
   * Creates a text element. If the `multiLine` facet is set, will create a
   * `<textarea>` element (setting the `cols` property to the
   * `fieldWidth` facet, if present). Otherwise, creates
   * `<input type="text">`.
   *
   * @param {jQuery} dom
   */
  TimeRangeStringLabelEditor.prototype.doInitialize = function (dom) {
    var id = this.generateId(),
        label = $('<label></label>').prop('for', id);
    dom.html(label);
  };

  /**
   * Loads the given string into the text element. If value is not already
   * a string, it will be toString-ed (or have `encodeToString` called if a
   * `baja:Simple`.
   *
   * @param {String} val
   */
  TimeRangeStringLabelEditor.prototype.doLoad = function (val) {
    this.jq().children('label').text(val.getTimeRangeLabel() || "");
  };

  return TimeRangeStringLabelEditor;
});
