/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/**
 * SpectrumChart.js
 * This javascript creates a SpectrumChart/HeatMap using d3 javascript.
 */
define(['baja!', 'bajaux/Widget', 'bajaux/mixin/subscriberMixIn', 'Promise', 'jquery', 'underscore', 'd3', 'moment', 'nmodule/analytics/rc/chart/base/AnalyticC3BaseChart', 'nmodule/analytics/rc/chart/utils/AnalyticsDataUtils', 'nmodule/analytics/rc/util/analyticsUtil', 'lex!analytics', 'css!nmodule/analytics/rc/chart/aggregation/aggregationChart'], function (baja, Widget, subscriberMixIn, Promise, $, _, d3, moment, AnalyticC3BaseChart, AnalyticsDataUtils, analyticsUtil, lexicon) {

  "use strict";

  var minValue, maxValue, minDuration, maxDuration;
  var lex = lexicon[0];
  var AggregationChart = function AggregationChart(params) {
    AnalyticC3BaseChart.call(this, $.extend({}, params));
    minValue = baja.Float.MAX_VALUE.valueOf();
    maxValue = baja.Float.MIN_VALUE.valueOf();
    minDuration = baja.Float.MAX_VALUE.valueOf();
    maxDuration = baja.Float.MIN_VALUE.valueOf();
  };

  AggregationChart.prototype = Object.create(AnalyticC3BaseChart.prototype);
  AggregationChart.prototype.constructor = AggregationChart;

  /**
   * Get the chart margins
   * @type {{top: number, right: number, bottom: number, left: number}}
   */
  AggregationChart.prototype.chartMargins = function () {
    return {
      top: 0,
      right: 0,
      bottom: 0,
      left: 0
    };
  };

  /**
   *
   * @param chartSettingsCollection
   * @param isRedrawRequest
   * @returns {{types, groups, xs, columns, names, y2, colors, y2Label, yLabel}|*}
   */
  AggregationChart.prototype.processData = function (chartSettingsCollection, isRedrawRequest) {
    var that = this;
    return AnalyticC3BaseChart.prototype.processData.apply(that, arguments).then(function (chartColl) {
      chartSettingsCollection = chartColl;
      _.each(chartSettingsCollection, function (csModel) {
        if (!isRedrawRequest) {
          analyticsUtil.normalize(csModel, csModel.getAnalyticTrendArray());
        }
      });
      var model = chartSettingsCollection[0];
      that.series = [];
      var color = model.getBrush();
      var processedSeries = {
        color: color,
        unitSymbol: model.getUnitDisplay(),
        name: model.getSeriesName(),
        ord: model.getOrd(),
        unit: model.getUnitDisplay(),
        chartType: model.getChartType(),
        showOnY2: model.getShowOnY2()
      };
      that.xArr = [];
      processedSeries.data = $.map(model.getAnalyticTrendArray(), function (item) {
        return {
          date: item.date,
          value: item.value,
          metaData: { interpolationStatus: item.interpolationStatus },
          color: color
        };
      });
      that.series.push(processedSeries);
      return Promise.resolve(analyticsUtil.getRenderableC3Data(that, that.series));
    });
  };

  /**
   *
   * @returns {*}
   */
  AggregationChart.prototype.getXAxisLabel = function () {
    return lex.get("chart.aggregation.xaxis.label");
  };

  /**
   *
   * @returns {*}
   */
  AggregationChart.prototype.getYAxisLabel = function () {
    var that = this,
        returnObj = { yLabel: [], y2Label: [] },
        labelName = "";
    if (_.isEmpty(that.colKeys)) {
      return AnalyticC3BaseChart.prototype.getYAxisLabel.apply(that, arguments);
    } else {
      labelName = lex.get({
        key: "chart.equipmentagg.tooltip.y",
        args: [that.colKeys[0].capitalizeFirstLetter(), that.series[0].unitSymbol]
      });
    }
    if (that.series[0].showOnY2) {
      returnObj.y2Label.push(labelName);
    } else {
      returnObj.yLabel.push(labelName);
    }
    return returnObj;
  };

  /**
   * This method should be overridden by subclasses to return respective chart types.
   * The default implementation retruns "none".
   * @returns {string}
   */
  AggregationChart.prototype.getSupportedExportTypes = function () {
    return ["aggchart", "csv"];
  };

  /**
   *
   * @returns {boolean}
   */
  AggregationChart.prototype.isMultiBindingSupported = function () {
    return false;
  };

  /**
   *
   * @returns {[string,string]}
   */
  AggregationChart.prototype.getColumnNames = function () {
    var arr = ['Date'],
        that = this;
    var chartModelList = that.chartModelList;
    if (_.isEmpty(that.colKeys)) {
      arr = AnalyticC3BaseChart.prototype.getColumnNames.apply(this);
    } else {
      _.each(that.colKeys, function (key, index) {
        arr.push(lex.get({
          key: "aggregationmultipart.colName." + key,
          args: [chartModelList[index].getUnitDisplay()]
        }));
        arr.push(lex.get({ key: "aggregationmultipart.colName." + key, args: [""] }) + " " + lex.get("chart.interpolationStatus.label"));
      });
    }
    that.columnNames = arr;
    return arr;
  };

  /**
   *
   * @returns {[string,string]}
   */
  AggregationChart.prototype.getColumnKeys = function () {
    var arr = ['date'],
        that = this;
    if (_.isEmpty(that.colKeys)) {
      arr = AnalyticC3BaseChart.prototype.getColumnKeys.apply(this);
    } else {
      arr.push.apply(arr, AnalyticsDataUtils.addInterpolationStatusForKey(that.colKeys));
    }
    that.columnKeys = arr;
    return arr;
  };

  /**
   *
   * @returns {string[]|(string)[]|module:http2.Settings|{data, dataFilter, timeRange, interval, unit, rollup, aggregation, dow, seriesNameBFormat, hisTotEnabled}|*}
   */
  AggregationChart.prototype.getDefaultSettings = function () {
    var defObj = AnalyticC3BaseChart.prototype.getDefaultSettings.apply(this, arguments);
    defObj.chartType = "bar";
    return defObj;
  };

  /**
   *
   * @param data
   * @param timeRange
   * @returns {*}
   */
  AggregationChart.prototype.buildModel = function (data, timeRange) {
    var model = AnalyticC3BaseChart.prototype.buildModel.apply(this, arguments);
    model.setShowOnY2(data.getShowOnY2());
    return model;
  };

  AggregationChart.prototype.getName = function () {
    return "AggregationChart";
  };

  return AggregationChart;
});
