/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/bacnetOws/rc/wb/mgr/commands/GetEventInformationCommand
 */
define(['baja!',
        'lex!bacnetOws',
        'jquery',
        'underscore',
        'dialogs',
        'bajaux/commands/Command',
        'nmodule/webEditors/rc/wb/mgr/mgrUtils',
        'baja!bacnetOws:GetEventInfoConfig,bacnetOws:GetEventInformationJob'], function (
        baja,
        lexs,
        $,
        _,
        dialogs,
        Command,
        mgrUtils) {

  'use strict';

  var getMainTableSelectedSubjects = mgrUtils.getMainTableSelectedSubjects,

      GET_ALARM_SUMMARY_SERVICE_CHOICE = 3,
      GET_EVENT_INFORMATION_SERVICE_CHOICE = 29,

      bacnetOwsLex = lexs[0];

  function getFirstSelectedDevice (mgr) {
    var selection = getMainTableSelectedSubjects(mgr),
        folderType = mgr.getModel().getFolderType();

    function isNotFolder (c) {
      return !(c.getType().is(folderType));
    }

    return _.first(_(selection).filter(isNotFolder));
  }

  /**
   * Convenience for getting a device id from a BacnetDevice instance.
   */
  function getDeviceId (dev) {
    return dev.getConfig().getDeviceObject().getObjectId();
  }

  /**
   * Convenience for getting all the string slots from the job. This is how
   * the event information is passed back.
   */
  function getStrings (job) {
    return job.getSlots().properties().is('baja:String').toValueArray();
  }

  /**
   * Command type to invoke a BGetEventInformationJob on the station and obtain the results
   * from dynamic slots on the job when it completes. The job will determine the services
   * supported by the device - initially trying to use a 'getEventInformation' request,
   * and, if that's not available, then trying a 'getAlarmSummary'. When the job completes
   * the handler in this command will look to see which service was actually used, which
   * will affect how it interprets the results.
   *
   * @class
   * @alias module:nmodule/bacnetOws/rc/wb/mgr/commands/GetEventInformationCommand
   * @extends module:bajaux/commands/Command
   */
  var GetEventInformationCommand = function GetEventInformationCommand (mgr) {
    var that = this;

    Command.call(this, {
      displayName: bacnetOwsLex.get('getEventInfo.label'),
      icon: bacnetOwsLex.get('getEventInfo.icon'),
      description: bacnetOwsLex.get('getEventInfo.description'),
      enabled: false,

      /**
       * Invoke the command - this will cause a bacnetOws:GetEventInformationJob to
       * be submitted.
       *
       * @alias module:nmodule/bacnet/rc/wb/mgr/commands/GetEventInformationCommand#invoke
       * @returns {Promise}
       */
      func: function () {

        return dialogs.showOkCancel({
          title: bacnetOwsLex.get('getEventInfo.label'),
          content: bacnetOwsLex.getSafe('getEventInfo.confirm')
        })
        .ok(function () {
          var config,
              network = mgr.getNetwork(),
              device = getFirstSelectedDevice(mgr);

          try {
            if (device) {

              config = baja.$('bacnetOws:GetEventInfoConfig');
              config.setObjectId(getDeviceId(device));

              return network.submitDeviceManagerJob(config)
                .then(function (ord) {
                  return mgr.setJob(baja.Ord.make({
                    base: baja.Ord.make('station:'),
                    child: ord.relativizeToSession()
                  }));
                });
            }
          }
          catch (err) {
            dialogs.showOk({
              title: bacnetOwsLex.get('getEventInfo.title'),
              content: bacnetOwsLex.getSafe('getEventInfo.fail', String(err))
            });
            throw err;
          }
        })
        .promise();
      }
    });

    mgr.on('jobcomplete', function (job) {
      if (job.getType().is('bacnetOws:GetEventInformationJob')) {
        that.complete(job);
      }
    });
  };

  GetEventInformationCommand.prototype = Object.create(Command.prototype);
  GetEventInformationCommand.prototype.constructor = GetEventInformationCommand;

  GetEventInformationCommand.GET_ALARM_SUMMARY_SERVICE_CHOICE = GET_ALARM_SUMMARY_SERVICE_CHOICE;
  GetEventInformationCommand.GET_EVENT_INFORMATION_SERVICE_CHOICE = GET_EVENT_INFORMATION_SERVICE_CHOICE;

  /**
   * In the event of a failure (e.g. something went wrong sending the request or the device
   * supports neither of the 'getEventInformation' or 'getAlarmSummary' services), the job will
   * place some error text in a dynamic property. This will look for that property and return
   * it, or the empty string if not found.
   *
   * @param {baja.Component} job
   * @returns {string}
   */
  function getFailureCause (job) {
    var cause = job.get('failureCause');
    if (cause === null) { cause = ''; }
    return cause;
  }

  function displayDialogForFailureCause (job) {
    dialogs.showOk({
      title: bacnetOwsLex.get('getEventInfo.title'),
      content: bacnetOwsLex.getSafe('getEventInfo.fail', getFailureCause(job))
    });
  }

  /**
   * Tokenize a single event information string and append it to the results string to
   * be displayed in a dialog.
   *
   * @param {string} results - the string being built containing all the combined event information strings.
   * @param {string} info - an alarm summary response, tokenized by '|' characters.
   * @param {number} index - the index of the event info in the results array.
   * @returns {string} - the results string with the additional info appended.
   */
  function formatEventInfoText (results, info, index) {
    var tokens = info.split('|');
    if (tokens.length < 11) { return 'Cannot parse: ' + info; }

    if (index > 0) { results += '\n'; }
    return results + bacnetOwsLex.get.apply(bacnetOwsLex, [ 'getEventInfo.eventInfoFormat' ].concat(tokens));
  }

  /**
   * Display the results when the job's service choice property is the 'GET_EVENT_INFORMATION'
   * service.
   *
   * @param {baja.Component} job - a GetEventInformationJob instance
   */
  function displayEventInformationResults (job) {
    dialogs.showOk({
      title: bacnetOwsLex.get('getEventInfo.title'),
      content: function (dlg) {
        var events = getStrings(job),
            jq = dlg.content();

        jq.append($('<span>').text(bacnetOwsLex.get('getEventInfo.header') + '\n'))
          .append($('<pre>').text(_(events).reduce(formatEventInfoText, '')));
      }
    });
  }

  /**
   * Tokenize a single alarm summary string and append it to the results string to
   * be displayed in a dialog.
   *
   * @param {string} results - the string being built containing all the combined summary strings.
   * @param {string} summary - an alarm summary response, tokenized by '|' characters.
   * @param {number} index - the index of the alarm summary in the results array.
   * @returns {string} the results string with the additional summary appended.
   */
  function formatAlarmSummary (results, summary, index) {
    var tokens = summary.split('|');
    if (tokens.length < 3) { return 'Cannot parse: ' + summary; }

    if (index > 0) { results += '\n'; }

    return results + bacnetOwsLex.get.apply(bacnetOwsLex, [ 'getASumm.alarmSummaryFormat' ].concat(tokens));
  }

  /**
   * Display the results when the job's service choice property is the 'GET_ALARM_SUMMARY'
   * service. This involves displaying a warning dialog before the results. The results
   * are stored as dynamic string properties on the job.
   *
   * @param {baja.Component} job - a GetEventInformationJob instance
   */
  function displayAlarmSummaryResults (job) {
    dialogs.showOk({
      title: bacnetOwsLex.get('getEventInfo.title'),
      content: function (dlg) {
        var jq = dlg.content();

        jq.append($('<span>').text(bacnetOwsLex.get('getEventInfo.unsupported') + '\n'))
          .append($('<br>'))
          .append($('<span>').text(bacnetOwsLex.get('getEventInfo.alarmSummary') + '\n'))
          .append($('<br>'))
          .append($('<span>').text('(' + bacnetOwsLex.get('getASumm.warning') + ')'));
      }
    })
    .ok(function () {
      var alarms = getStrings(job);

      dialogs.showOk({
        title: bacnetOwsLex.get('getASumm.title'),
        content: function (dlg) {
          var jq = dlg.content();

          jq.append($('<span>').text(bacnetOwsLex.get('getASumm.header') + '\n'))
            .append($('<pre>').text(_(alarms).reduce(formatAlarmSummary, '')));
        }
      });
    });
  }

  /**
   * Called when the submitted job is complete. Will obtain the results from
   * the encoded string properties of the job and display a dialog. This will
   * check the service that was used, depending on the device's supported set.
   *
   * @static
   * @param {baja.Component} job
   */
  GetEventInformationCommand.prototype.complete = function (job) {
    return job.loadSlots()
      .then(function () {
        if ('success' === job.getJobState().getTag()) {
          switch (job.getServiceChoice()) {
            case GET_EVENT_INFORMATION_SERVICE_CHOICE: {
              displayEventInformationResults(job);
              break;
            }
            case GET_ALARM_SUMMARY_SERVICE_CHOICE: {
              displayAlarmSummaryResults(job);
              break;
            }
            default: {
              displayDialogForFailureCause(job);
              break;
            }
          }
        }
        else {
          displayDialogForFailureCause(job);
        }
      });
  };

  return GetEventInformationCommand;
});
