/**
 * @private
 * @module nmodule/micros/rc/MicrosNewCommand
 */
define([
  'baja!',
  'bajaScript/baja/ord/SlotPath',
  'nmodule/webEditors/rc/wb/mgr/commands/NewCommand',
  'nmodule/micros/rc/MicrosUtils',
  'nmodule/micros/rc/MicrosPointManagerModel',
  'bajaux/commands/Command',
  'Promise',
  'jquery',
  'dialogs',
  'underscore',
  'nmodule/js/rc/asyncUtils/asyncUtils',
  'nmodule/webEditors/rc/fe/feDialogs',
  'nmodule/webEditors/rc/wb/PropertySheet',
  'nmodule/webEditors/rc/wb/mgr/BatchComponentEditor',
  'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo',
  'css!nmodule/micros/rc/micros',
    ], function (
   baja,
   SlotPath,
   NewCommand,
   MicrosUtils,
   MicrosPointManagerModel,
   Command,
   Promise,
   $,
   dialogs,
   _,
   asyncUtils,
   feDialogs,
   PropertySheet,
   BatchComponentEditor,
   MgrTypeInfo
    ) {

  'use strict';

  var NEW_COMMAND_DEFAULT_TITLE = 'Add Rooms',
      //HELP_DIALOG_ERROR = 'commands.micros.new.help.error',
      HELP_TEXT_LOCATION = MicrosUtils.microsLex('micros.helptext'),
      UNSET_ROOM_NUMBER = MicrosUtils.microsLex('unset.room.number'),
      DUPLICATE_ROOM_NUMBER = MicrosUtils.microsLex("microsProxyExt.duplicate.room.number");
      //HELP_TEXT_LOCATION = 'module://micros/rc/html/altHelp.html';
      //HELP_TEXT_LOCATION = 'module://micros/help.html';
  // 2 things, in the micros-rt jar, don't seem to be able access help.html from the broswer
  // (can't open it in myModules either)
  // There appears to be a different format


  /**
   * Command for adding new components to a Micros Point Manager view.
   *
   * @private
   * @class
   * @alias module:nmodule/micros/rc/MicrosNewCommand
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   */
  var MicrosNewCommand = function MicrosNewCommand(manager) {
    Command.call(this, {
      module: 'micros',
      lex: 'commands.micros.new',
      // shoud mean that displayName & icon equate to the keys below ...
      //  commands.micros.new.displayName=New
      //  commands.micros.new.icon=module://icons/x16/newFile.png

      enabled:     true,

      func: function () {
        var mgrModel = manager.getModel();

        return manager.getOrdBase()
        .then(function (ordBase) {
           return promptForParams(mgrModel, ordBase)
            .then(function (params) {
              return params && promptForNewComponents(mgrModel, params, ordBase);
            })
            .then(function (comps) {
              return comps && mgrModel.addInstances(comps);
            });
        })
        .catch(feDialogs.error);
      }
    });
  };


  /**
   * To query the user for what type of new components to create, and how
   * many, a PropertySheet will be shown in a dialog. This component contains
   * the necessary fields to load into that PropertySheet.
   *
   * @inner
   * @param {Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>} newTypes an
   * array of 'MgrTypeInfo' instances, for the types that will be presented to the
   * user to choose from.
   * @returns {baja.Component}
   */
  function makeComponent(newTypes) {
    // newTypes isn't used - for micros it's always a BooleanPoint,
    // so we're not showing any dropdown options

    var comp = baja.$('baja:Component', {
          pointPrefix:      'Room',
          roomPrefix:       '',
          startRoomNumber:  baja.Integer.make(-1), // TODO should this be set from the 'unset.room.number' lexicon ?
          count:            baja.Integer.make(1),
          displayNames:     baja.NameMap.make({
            pointPrefix:      MicrosUtils.microsLex('pmc.add.pointPrefix'),
            roomPrefix:       MicrosUtils.microsLex('pmc.add.roomPrefix'),
            startRoomNumber:  MicrosUtils.microsLex('pmc.add.startRn'),
            count:            MicrosUtils.microsLex('pmc.add.count')
          })
        });

    comp.setFacets({
      slot: 'count',
      facets: baja.Facets.make({ min: 1, max: 100 })
    });

    comp.setFacets({
      slot: 'startRoomNumber',
      facets: baja.Facets.make({ min: MicrosNewCommand.getMinimumRoomNumber(), max: Infinity })
    });

    comp.setFlags({
      slot: 'displayNames',
      flags: baja.Flags.HIDDEN
    });

    return comp;
  }

  /**
   * When loading up the 'BatchComponentEditor', a secondary 'MgrModel' will be
   * constructed solely for the components being edited. After editing the
   * components, they will be applied back to the original 'MgrModel'.
   *
   * @inner
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} mgrModel the
   * `MgrModel` to which we're adding components
   * @param {Object} params
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo} params.typeInfo the 'MgrTypeInfo'
   * representing the type of the components we're creating
   * @param {Number} params.count how many new components to create
   * @returns {Promise} promise to be resolved with a new
   * 'MgrModel' containing the new instances to edit
   */
  MicrosNewCommand.makeSubMgrModel = function (mgrModel, params) {

    var origContainer = mgrModel.getComponentSource().getContainer(),
        source = baja.$('baja:Folder'),
        typeInfo = params.typeInfo,
        count = params.count,
        pointPrefix = params.pointPrefix,
        roomPrefix = params.roomPrefix,
        startRoomNumber = params.startRoomNumber,
        newComponents = [],
        offset;

    for (offset = 0; offset < count; offset++) {
      var calculatedRoomNumber,
          completeRoomNumber,
          slotName;

      // The room's number is the starting number
      //   plus any offset (derived from the total number of rooms to add, ie count)
      // - should be treated as numbers
      calculatedRoomNumber = startRoomNumber + offset;

      // The complete room number is the number above with any prefix before it
      //   (default is '', but could be 'G' or '1' or '30' etc
      // - should be treated as Strings
      completeRoomNumber = roomPrefix + calculatedRoomNumber;

      // slotName if the pointPrefix (default = 'Room') plus the complete room number
      //   (eg 'RoomG23')
      // - should be treated as Strings
      slotName = pointPrefix + completeRoomNumber;

      if ( origContainer.has(slotName) ) {
        throw new Error(DUPLICATE_ROOM_NUMBER + ' [' + slotName + ']');
      }

      newComponents.push({
        slotName: SlotPath.escape(slotName),
        roomNumber: completeRoomNumber
      });
    }

    return asyncUtils.mapAsync(newComponents, function (newComponent) {
      return Promise.resolve(mgrModel.newInstance(typeInfo, newComponent.roomNumber))
      .then(function (instance) {
        return source.add({ slot: newComponent.slotName, value: instance });
      });
    })
    .then(function () {
      return new MicrosPointManagerModel({
        componentSource: source,
        component: origContainer,
        columns: mgrModel.getColumns()
      });
    });
  };

  /**
   * Present the user with a prompt asking what type of components to add and
   * how many.
   *
   * @inner
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} mgrModel
   * @param {baja.Component} [ordBase]
   * @returns {Promise} promise to be resolved with an object with
   * the new room properties, or null if the prompt was canceled
   */
  function promptForParams(mgrModel, ordBase) {
    var newTypes = mgrModel.getNewTypes(),
        newType = newTypes[0],  // should only be one and it should be control:BooleanPoint
        comp;

    MgrTypeInfo.markDuplicates(newTypes);
    comp = makeComponent(newTypes);

    return feDialogs.showFor({
      title: MicrosUtils.microsLex('pmc.add.title', NEW_COMMAND_DEFAULT_TITLE ), // 'pmc.add.title' doesn't currently exist
      value: comp,
      type: PropertySheet,
      properties: { showHeader: false, showControls: false, ordBase: ordBase },

      progressCallback: function (msg, arg) {

        switch(msg) {
          //case 'created':     return console.log('editor created', arg);
          //case 'initialized':   return console.log('editor initialized', arg.jq());

          case 'loaded':
            var jq = arg.jq();

            var contentToBeInserted = '<a id="helpLink" class="icon-icons-x16-questionMark" />',
                target = 'td.col-value > input';

            // put the help icon on the first row of the dialog
            jq.find(target).first().after(contentToBeInserted);

            jq.on('click', '#helpLink', function() {
              MicrosNewCommand.getHelpDialog(HELP_TEXT_LOCATION);
              return false;
            });

            // Add a validator to make sure the dialog initialises with the OK button disabled
            // and that startRoomNumber is changed to a valid value
            arg.validators().add(function (val) {
              // when the dialog is first created/validate val (the ComplexDiff) is empty
              // startRoomNumber only gets added as a property when it's changed in the dialog
              if(val.$map && !val.$map.startRoomNumber) {
                var startRoomNumberInput = jq.find('td.slot-startRoomNumber > input');
                // startRoomNumber must be more that the UnsetRoomNumber (which is -1 by default)
                if (startRoomNumberInput && startRoomNumberInput.val() < MicrosNewCommand.getMinimumRoomNumber()) {
                  //console.log(MicrosUtils.microsLex('pmc.startRn.invalid') + startRoomNumberInput.val());
                  throw new Error(MicrosUtils.microsLex('pmc.startRn.invalid') + startRoomNumberInput.val());
                }
              }
              return true;
            });

            return;
            //return console.log('editor loaded', arg.value());

          //case 'invalid':     return console.log('validation error', arg);
          //case 'valid':       return console.log('value is ok');
        }
      }
    })
    .then(function (diff) {

      if (diff) {
        return diff.apply(comp)
        .then(function () {
          return {
            typeInfo:         newType,
            pointPrefix:      comp.get('pointPrefix').valueOf(),
            roomPrefix:       comp.get('roomPrefix').valueOf(),
            startRoomNumber:  comp.get('startRoomNumber').valueOf(),
            count: comp.get('count').valueOf()
          };
        });
      }
    });
  }

  /**
   * After determining what type of components to add and how many, show a
   * 'BatchComponentEditor' allowing the user to make changes to the new
   * instances.
   *
   * @inner
   * @param {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel} mgrModel
   * @param {Object} params
   * @param {String} params.typeSpec
   * @param {Number} params.count
   * @returns {Promise} promise to be resolved with an array of
   * new `Component` instances, or `null` if `BatchComponentEditor` dialog was
   * canceled
   */
  function promptForNewComponents(mgrModel, params, ordBase) {
    return MicrosNewCommand.makeSubMgrModel(mgrModel, params)
    .then(function (subMgrModel) {
      return feDialogs.showFor({
        value: subMgrModel,
        properties: { ordBase: ordBase },
        type: BatchComponentEditor
      });
    })
    .then(function (model) {
      return model && _.map(model.getRows(), function (row) {
        return row.getSubject();
      });
    });
  }


  MicrosNewCommand.prototype = Object.create(NewCommand.prototype);
  MicrosNewCommand.prototype.constructor = MicrosNewCommand;

  MicrosNewCommand.getMinimumRoomNumber = function() {
    var unsetRoomNumber = MicrosNewCommand.getUnsetRoomNumberFromLexicon();
    if (unsetRoomNumber) {
      return unsetRoomNumber+1;
    }
    return 0;
  };

  MicrosNewCommand.getUnsetRoomNumberFromLexicon = function() {
    return parseInt(UNSET_ROOM_NUMBER);
  };

  MicrosNewCommand.getHelpDialog = function (helpTextOrd) {
    return dialogs.showOk(function(dlg, jq) {
      return baja.Ord.make(helpTextOrd).get()
      .then(function(file) {
        return MicrosNewCommand.getFileContents(file);
      })
      .then(function (response) {
        jq.addClass('MicrosUxHelpContainer');
        jq.html(response);
        return Promise.resolve(true);
      })
      .catch(feDialogs.error);
    });
  };

  MicrosNewCommand.getFileContents = function (helpFile) {
    return $.ajax(helpFile.getReadUri());
  };

  return MicrosNewCommand;
});
