/**
 * @private
 * @module nmodule/micros/rc/MicrosDeviceManager
 */
define([
  'baja!',
  'Promise',
  'underscore',
  'nmodule/micros/rc/MicrosUtils',
  'bajaux/commands/Command',
  'nmodule/webEditors/rc/wb/mgr/commands/EditCommand',
  'nmodule/webEditors/rc/wb/mgr/commands/NewCommand',
  'nmodule/driver/rc/wb/mgr/DeviceMgr',
  'nmodule/webEditors/rc/fe/baja/util/compUtils',
  'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo',
  'nmodule/driver/rc/wb/mgr/DeviceMgrModel',
  'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn',
  'nmodule/webEditors/rc/wb/mgr/model/columns/NameMgrColumn',
  'nmodule/webEditors/rc/wb/mgr/model/columns/TypeMgrColumn',
  'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn',
  'nmodule/driver/rc/wb/mgr/columns/DeviceExtsMgrColumn'
    ], function (
  baja,
  Promise,
  _,
  MicrosUtils,
  Command,
  EditCommand,
  NewCommand,
  DeviceMgr,
  compUtils,
  MgrTypeInfo,
  DeviceMgrModel,
  IconMgrColumn,
  NameMgrColumn,
  TypeMgrColumn,
  PropertyMgrColumn,
  DeviceExtsMgrColumn
    ) {

  'use strict';

  var PING_LABEL = 'dm.PingCmd.label',
      PING_ICON = 'dm.PingCmd.icon';

  var TYPES_TO_LOAD = [
    MicrosUtils.typeSpec.DEVICE,
    MicrosUtils.typeSpec.MICROS_DEVICE,
    MicrosUtils.typeSpec.MICROS_NETWORK
  ];



  /**
   * Constructor.
   */
  var MicrosDeviceManager = function MicrosDeviceManager (params)  {
    DeviceMgr.call(this, {
      moduleName: 'micros',
      keyName: 'MicrosDeviceManager',
      // we don't want the NewFolder command to display, so don't specify folderType
      //folderType: MICROS_DEVICE_FOLDER_TYPE,
      subscriptionDepth: 3
    });
  };

  MicrosDeviceManager.prototype = Object.create(DeviceMgr.prototype);
  MicrosDeviceManager.prototype.constructor = MicrosDeviceManager;


  //var makeColumns = function() {
  MicrosDeviceManager.makeColumns = function() {

    var columns = [
      new IconMgrColumn(),
      new NameMgrColumn(),
      new TypeMgrColumn(),
      new DeviceExtsMgrColumn('exts', { component: baja.$(MicrosUtils.typeSpec.MICROS_DEVICE) }),
      new PropertyMgrColumn('status', {
        //type: DEVICE_TYPE//,
        type: baja.lt(MicrosUtils.typeSpec.MICROS_DEVICE)//,
        //flags: MicrosUtils.flags.READONLY
      }),
      new PropertyMgrColumn('enabled', {
        flags: MicrosUtils.flags.UNSEEN,
        displayName: MicrosUtils.microsLex('enabled')
      }),
      new PropertyMgrColumn('address', {
        flags: MicrosUtils.flags.EDITABLE,
        displayName: MicrosUtils.microsLex('address')
      }),
      new PropertyMgrColumn('health', {
        //type: DEVICE_TYPE,
        type: baja.lt(MicrosUtils.typeSpec.MICROS_DEVICE),
        displayName: MicrosUtils.microsLex('health')
      }),
      new PropertyMgrColumn('versionPMS', {displayName: MicrosUtils.microsLex('versionPMS', 'Version PMS')}),
      new PropertyMgrColumn('versionIFC', {displayName: MicrosUtils.microsLex('versionIFC', 'Version IFC')}),
      new PropertyMgrColumn('hotelID', {displayName: MicrosUtils.microsLex('hotelID','Hotel ID')})
    ];

    return columns;
  };


  /**
   * Create a MgrModel instance with the appropriate columns.
   *
   * @param {baja.Component} component,  a 'micros:MicrosNetwork' or 'micros:MicrosDeviceFolder' instance
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/mgr/model/MgrModel>}
   */
  MicrosDeviceManager.prototype.makeModel = function (component) {
    return baja.importTypes({
      // make sure the types we need are available locally
      typeSpecs: TYPES_TO_LOAD
    })
    .then(function (/*types*/) {
      return  MgrTypeInfo.make([ baja.lt(MicrosUtils.typeSpec.MICROS_DEVICE) ]);
    })
    .then(function (newTypes) {
      return new DeviceMgrModel({
        component: component,
        newTypes: newTypes,
        columns: MicrosDeviceManager.makeColumns()
      });
    });
  };

  MicrosDeviceManager.makePingCommand = function(manager) {
    return new Command({
      displayName: MicrosUtils.microsLex(PING_LABEL),
      icon:        MicrosUtils.microsLex(PING_ICON),
      enabled:     true,
      func:        function () {
        manager.getNetwork().pingDevice();
      }
    });
  };

  MicrosDeviceManager.prototype.makeCommands = function() {

    var that = this,
        i,
        newCommandIndex = -1,
        editCommandIndex = -1,
        cmds = DeviceMgr.prototype.makeCommands.apply(that, arguments);

    // The superclass implementation of this method gives us New & Edit.
    // But we want Edit & Ping.
    // On workbench there's also a Tagit button. It's not implemented in the UX manager,
    // but if it is at some point in the future, this implementation should pick it up automatically.

    for (i=0; i < cmds.length; i++) {
      if (cmds[i] instanceof NewCommand) {
        newCommandIndex = i;
        break;
      }
    }

    // remove the NewCommand if we found one
    if (newCommandIndex > -1) {
      cmds.splice(newCommandIndex, 1);
    }

    for (i=0; i < cmds.length; i++) {
      if (cmds[i] instanceof EditCommand) {
        editCommandIndex = i;
        break;
      }
    }

    // insert Ping Command after EditCommand
    //cmds.splice(editCommandIndex+1, 0, pingCommand);
    cmds.splice(editCommandIndex+1, 0, MicrosDeviceManager.makePingCommand(that));


    // This is the simpler implentation creating the commands list from scratch
//    var that = this,
//        cmds = [new EditCommand(that),
//               pingCommand
//        ];

    return cmds;
  };

  /**
   * Return the MicrosNetwork from the source component (which may be the network or may
   * be a device folder several levels deep).
   *
   * @returns {baja.Component}
   */
  MicrosDeviceManager.prototype.getNetwork = function () {
    var that = this,
      model = that.getModel(),
      comp = model.getComponentSource().getContainer();

    return compUtils.closest(comp, baja.lt(MicrosUtils.typeSpec.MICROS_NETWORK));
  };

  return MicrosDeviceManager;
});
