/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/ndriver/rc/util/learnUtil
 */
define([
    'baja!',
    'baja!' +
    'control:NumericWritable,control:NumericPoint,control:BooleanWritable,control:BooleanPoint,' +
    'control:EnumWritable,control:EnumPoint,control:StringWritable,control:StringPoint,' +
    'ndriver:NDiscoveryGroup,ndriver:NDeviceManager',
    'underscore',
    'Promise',
    'jquery',
    'nmodule/ndriver/rc/util/util',
    'nmodule/ndriver/rc/util/columnUtil',
    'nmodule/ndriver/rc/util/rpcUtil',
    'nmodule/webEditors/rc/wb/table/tree/TreeTableModel',
    'nmodule/webEditors/rc/wb/tree/TreeNode',
    'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn'
  ], function (
    baja,
    types,
    _,
    Promise,
    $,
    util,
    columnUtil,
    rpcUtil,
    TreeTableModel,
    TreeNode,
    IconMgrColumn
  ) {

  'use strict';

  var N_DISCOVERY_GROUP = 'ndriver:NDiscoveryGroup',
    I_N_DISCOVERY_ICON = 'ndriver:INDiscoveryIcon',

    DEVICE_ICON = baja.Icon.make([ 'module://icons/x16/device.png' ]),
    DEVICE_FOLDER_ICON = baja.Icon.make([ 'module://icons/x16/deviceFolder.png' ]),
    POINT_FOLDER_ICON = baja.Icon.make([ 'module://icons/x16/pointFolder.png' ]),

    NUMERIC_WRITABLE_ICON = baja.Icon.make([ 'module://icons/x16/control/numericPoint.png' ]),
    BOOLEAN_WRITABLE_ICON = baja.Icon.make([ 'module://icons/x16/control/booleanPoint.png' ]),
    ENUM_WRITABLE_ICON = baja.Icon.make([ 'module://icons/x16/control/enumPoint.png' ]),
    STRING_WRITABLE_ICON = baja.Icon.make([ 'module://icons/x16/control/stringPoint.png' ]),

    NUMERIC_POINT_ICON = baja.Icon.make([ 'module://icons/x16/statusNumeric.png' ]),
    BOOLEAN_POINT_ICON = baja.Icon.make([ 'module://icons/x16/statusBoolean.png' ]),
    ENUM_POINT_ICON = baja.Icon.make([ 'module://icons/x16/statusEnum.png' ]),
    STRING_POINT_ICON = baja.Icon.make([ 'module://icons/x16/statusString.png' ]),

    DEFAULT_POINT_ICON = null;


  var getDiscoveryIcon = function (discovery, isPointManager, isDiscoveryGroup) {
    var defaultTypeSpec, defaultType,
      discoveryType = discovery.getType();

    if (discoveryType.is(I_N_DISCOVERY_ICON)) {
      return rpcUtil.getDiscoveryIcon(discovery);
    }

    if (!isPointManager) {
      return isDiscoveryGroup ? DEVICE_FOLDER_ICON : DEVICE_ICON;
    }

    // at this point we must have a point manager

    if (isDiscoveryGroup) {
      return POINT_FOLDER_ICON;
    }


    // TODO - although this next bit works, it makes a network call for each discovery object.
    //            Might want to rethink how this is done and/or how important the icons are.
    //            (the same applies to the rpcUtil.getDiscoveryIcon call above).

    // For discovered points, try to match the discovered type with one of the ControlPoint types
    // see com.tridium.ndriver.ui.NMgrLearn.java#getPointIconDefault
    return rpcUtil.getValidDatabaseTypes(discovery)
      .then(function (validTypeSpecs) {
        if (!validTypeSpecs || validTypeSpecs.length === 0) {
          return DEFAULT_POINT_ICON;
        }

        defaultTypeSpec = validTypeSpecs[0];

        if (!defaultTypeSpec) {
          return DEFAULT_POINT_ICON;
        }

        defaultType = baja.$(defaultTypeSpec).getType();

        if (defaultType.is('control:NumericWritable')) { return NUMERIC_WRITABLE_ICON; }
        if (defaultType.is('control:NumericPoint')) { return NUMERIC_POINT_ICON; }
        if (defaultType.is('control:BooleanWritable')) { return BOOLEAN_WRITABLE_ICON; }
        if (defaultType.is('control:BooleanPoint')) { return BOOLEAN_POINT_ICON; }
        if (defaultType.is('control:EnumWritable')) { return ENUM_WRITABLE_ICON; }
        if (defaultType.is('control:EnumPoint')) { return ENUM_POINT_ICON; }
        if (defaultType.is('control:StringWritable')) { return STRING_WRITABLE_ICON; }
        if (defaultType.is('control:StringPoint')) { return STRING_POINT_ICON; }

        return DEFAULT_POINT_ICON;
      });
  };


  /**
   * A utility class for dealing with ndriver discovery
   *
   * API Status: **Private**
   * @exports nmodule/ndriver/rc/util/learnUtil
   */
  var exports = {};

  /*
   * Make a tree node for an item in the discovery table.
   *
   * @param  {baja.Complex} discovery - a discovered item.
   * @param  {String} discoveryLeafTypeSpec - the type spec of the discoveryLeaf.
   * @param  {Boolean} isPointManager - whether the manager is a PointManager, (false assumes it's a DeviceManager).
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/tree/TreeNode>}
   */
  exports.makeDiscoveryTableNode = function (discovery, discoveryLeafTypeSpec, isPointManager) {
    var name = discovery.getName(),
      displayName = discovery.getDisplayName(),
      mayHaveKids = _.constant(false),
      kids = [],
      promisesForKids = [],
      isGroup = _.constant(false),
      isDiscoveryGroup = discovery.getType().is(N_DISCOVERY_GROUP); // using NDiscoveryGroup rather than
                                                                    // INDiscoveryGroup as it has a 'description' property

    if (isDiscoveryGroup) {
      displayName = discovery.getDescription();

      mayHaveKids = _.constant(true);
      isGroup = _.constant(true);

      _.each(discovery.getSlots().toValueArray(), function (kid) {
        if (kid.getType().is(discoveryLeafTypeSpec) || baja.hasType(kid, N_DISCOVERY_GROUP)) {
          promisesForKids.push(
            exports.makeDiscoveryTableNode(kid, discoveryLeafTypeSpec, isPointManager)
          );
        }
      });
    }


    return Promise.all(promisesForKids)
      .then(function (resolvedKids) {
        kids = resolvedKids;
        return getDiscoveryIcon(discovery, isPointManager, isDiscoveryGroup);
      })
      .then(function (icon) {
        var node = new TreeNode(name, displayName, kids);
        node.value = _.constant(discovery);
        node.getIcon = _.constant(icon);
        node.mayHaveKids = mayHaveKids;
        node.isGroup = isGroup;

        return node;
      });
  };

  /*
   * Make the model for the discovery table.
   *
   * @param  {String} discoveryLeafTypeSpec - the type spec of the discoveryLeaf.
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */
  exports.makeLearnModel = function (discoveryLeafTypeSpec) {
    return columnUtil.getColumnsFor(
      baja.$(discoveryLeafTypeSpec),
      [ new IconMgrColumn() ], // standard columns
      true                     // 'isDicovery'
    ).then(function (columns) {
      return TreeTableModel.make({
        columns: columns
      });
    });
  };

  return exports;
});
