/**
 * @copyright 2021 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/httpClient/rc/HttpClientChooser
 */
define([
  'baja!',
  'Promise',
  'underscore',
  'nmodule/webEditors/rc/fe/baja/BaseEditor'
], function (
  baja,
  Promise,
  _,
  BaseEditor
) {

  'use strict';

  var clientMap = {};

  /**
   * A field editor for picking from a list of
   * 'httpClient:BIHttpCommClient' components
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/httpClient/rc/HttpClientChooser
   */
  var HttpClientChooser = function HttpClientChooser(params) {
    BaseEditor.call(this, _.extend({
      keyName: 'HttpClientChooser'
    }, params));
  };
  HttpClientChooser.prototype = Object.create(BaseEditor.prototype);
  HttpClientChooser.prototype.constructor = HttpClientChooser;

  /**
   * Initializes this editor.
   *
   * @param {JQuery} dom
   */
  HttpClientChooser.prototype.doInitialize = function (dom) {
    var that = this;

    dom.addClass('HttpClientChooser');

    dom.html(
      '<span><select class="client-selection" /></span>'
    );

    dom.on('change', '.client-selection', function (e) {
      that.setModified(true);
      e.preventDefault();
      return false;
    });
  };

  /**
   * Loads the client selection into the editor
   *
   * @param {String} clientHandle the handle ord of the selected client.
   * @returns {Promise}
   */
  HttpClientChooser.prototype.doLoad = function (clientHandle) {
    return this.$populateSelectOptions(clientHandle);
  };

  /**
   * Reads the select element and return the currently selected value.
   *
   * @returns {String}
   */
  HttpClientChooser.prototype.doRead = function () {

    var selectedClientName = this.$getSelect().val();
    var saveValue = "";
    if (selectedClientName) {
      Object.keys(clientMap).forEach(function (handleOrd) {

        var clientName = clientMap[handleOrd];
        if (clientName === selectedClientName) {
          saveValue = handleOrd;
        }
      });
    }
    return saveValue;
  };

  /**
   * Enables or disables the select dropdown.
   *
   * @param {Boolean} enabled
   */
  HttpClientChooser.prototype.doEnabled = function (enabled) {
    this.$getSelect().prop('disabled', this.isReadonly() || !enabled);
  };

  /**
   * Disables or enables the select dropdown.
   *
   * @param {Boolean} readonly
   */
  HttpClientChooser.prototype.doReadonly = function (readonly) {
    this.$getSelect().prop('disabled', !this.isEnabled() || readonly);
  };

  /**
   * Clean up after this editor
   * @returns {Promise}
   */
  HttpClientChooser.prototype.doDestroy = function () {
    this.jq().removeClass('HttpClientChooser');
  };

  /**
   * Populates the dropdown with the possible clients.
   *
   * @private
   * @param {String} clientHandle the currently selected client handle ord.
   * @returns {Promise}
   */
  HttpClientChooser.prototype.$populateSelectOptions = function (clientHandle) {
    var that = this,
      selectElement = that.$getSelect();

    // clear out the dropdown options
    that.$getSelect().empty();

    // add a blank element to use if the current value no longer exists,
    // this element will be removed it the current value is an option.
    selectElement.append(
      '<option class="ux-option blank-option" value=""></option>'
    );
    // and make it the selected one.
    selectElement.val('');

    return that.$getClients()
      .then(function (clients) {
        clientMap = clients;
        Object.keys(clientMap).forEach(function (handleOrd) {
          var clientDisplayName = clientMap[handleOrd];
          if (clientDisplayName) {
            selectElement.append(
              '<option class="ux-option" value="' + _.escape(clientDisplayName) + '">' + _.escape(baja.SlotPath.unescape(clientDisplayName)) + '</option>'
            );

            if (handleOrd === clientHandle) {
              // if this matches the current value, set it as the selected option
              selectElement.val(clientDisplayName);
              // remove the blank option
              that.jq().find('.blank-option').remove();
            }
          }
        });
      });
  };

  /**
   * Construct a list of available http clients on the station.
   *
   * @private
   * @returns {Promise.<Object>}
   */
  HttpClientChooser.prototype.$getClients = function () {

    return baja.rpc("type:httpClient:ResponseCookieAuth", "findAllClients");
  };


  /**
   * Get the client dropdown select element.
   *
   * @private
   * @returns {JQuery}
   */
  HttpClientChooser.prototype.$getSelect = function () {
    return this.jq().find('.client-selection');
  };

  return (HttpClientChooser);
});
