/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 */

/* eslint-env browser */
define(['bajaux/events', 'jquery', 'Promise', 'underscore', 'nmodule/js/rc/log/Log', 'nmodule/schedule/rc/model/TimeOfDay', 'nmodule/schedule/rc/util/enclosingPoint', 'nmodule/schedule/rc/util/time', 'nmodule/webEditors/rc/util/htmlUtils', 'nmodule/webEditors/rc/wb/mixin/mixinUtils'], function (events, $, Promise, _, Log, TimeOfDay, enclosingPoint, time, htmlUtils, mixinUtils) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin,
      contextMenuOnLongPress = htmlUtils.contextMenuOnLongPress,
      roundToHalfHour = time.roundToHalfHour,
      MILLIS_IN_DAY = time.MILLIS_IN_DAY,
      MILLIS_IN_HALF_HOUR = time.MILLIS_IN_HALF_HOUR,
      MODIFY_EVENT = events.MODIFY_EVENT,
      MIXIN_NAME = 'dragSupport';

  function makeTime(millis) {
    return new TimeOfDay({
      millis: millis
    });
  }

  function getTouches(_ref) {
    var originalEvent = _ref.originalEvent;
    return originalEvent && originalEvent.touches || [];
  }

  function getTouch(event) {
    return getTouches(event)[0] || event;
  }

  function getX(event) {
    return getTouch(event).pageX;
  }

  function getY(event) {
    return getTouch(event).pageY;
  }

  function leftClick(event) {
    return (event.which === 1 || getTouches(event).length === 1) && !event.shiftKey;
  }
  /**
   * @typedef {object} DragOp
   * @property {number} start initial start time, in millis past midnight
   * @property {number} finish initial finish time, in millis past midnight
   * @property {number} startY pageY of the mouse event that started the drag
   * @property {string} operation `create`, `move`, `resizeTop`, or `resizeBottom`
   * @property {module:nmodule/schedule/rc/fe/ScheduleBlockEditor} scheduleBlockEditor
   * the editor we're dragging - undefined for a `create` operation
   */

  /**
   * @param {module:nmodule/schedule/rc/fe/DayEditor} dayEditor
   * @param {module:nmodule/schedule/rc/fe/ScheduleBlockEditor} blockEditor
   * @param {string} operation
   * @param {jQuery.Event} event
   * @returns {DragOp|undefined}
   */


  function startOp(dayEditor, blockEditor, operation, event) {
    var start,
        startY = getY(event),
        finish;

    if (blockEditor) {
      var block = blockEditor.value();
      start = +block.getStart();
      finish = +block.getFinish() || MILLIS_IN_DAY;
    } else if (operation === 'create') {
      var blocksDiv = dayEditor.$getBlocksDisplayElement(),
          pixels = startY - blocksDiv.offset().top,
          startMillis = Math.max(dayEditor.$pixelsToMillis(pixels), 0);
      start = finish = roundToHalfHour(startMillis); //Don't create an event starting the bottom midnight

      if (start === 0 && finish === 0 && dayEditor.$closerToBottom(pixels)) {
        return;
      }
    }

    return {
      start: start,
      finish: finish,
      startY: startY,
      operation: operation,
      scheduleBlockEditor: blockEditor
    };
  }
  /**
   * Given a mouse click, find the divs where the click lies within that div's
   * bounds.
   *
   * @inner
   * @param {jQuery} div only find divs that are children of this parent div
   * @param {jQuery.Event} event the mouse click event
   * @return jQuery the divs that lie under this mouse click
   */


  function getDivsUnderMouse(div, event) {
    var x = getX(event),
        y = getY(event);
    var selector = ':enclosingPoint(' + x + ',' + y + ')';
    return div.find(':not(.overlay)').filter(selector);
  }
  /**
   * Given a set of divs underneath our mouse click, decides which one is
   * determined to be "on top" - top/bottom resizers always win, followed
   * by the schedule block itself, the block divs on the day itself coming
   * in last.
   *
   * @inner
   * @param {jQuery} divs the divs from whom to pick a winner
   */


  function getPriorityDiv(divs) {
    var grabs = divs.filter('.grab'),
        blocks = divs.filter('.ScheduleBlockEditor');

    if (grabs.length && blocks.length) {
      var block = blocks[0];
      return grabs.filter(function (i, grab) {
        return $.contains(block, grab);
      });
    } else if (grabs.length) {
      return grabs;
    } else if (blocks.length) {
      return blocks;
    } else {
      return divs.filter('.block');
    }
  }
  /**
   * Set the selected block. We do the event trigger here rather than
   * directly in `$setSelectedBlock` since `$setSelectedBlock` will be called
   * from `selectionChange` handlers and we don't want infinite loops.
   *
   * @inner
   * @param dayEditor
   * @param scheduleBlockEditor
   */


  function selectBlock(dayEditor, scheduleBlockEditor) {
    dayEditor.$setSelectedBlock(scheduleBlockEditor);
    dayEditor.trigger('selectionChange', scheduleBlockEditor);
  }
  /**
   * When we receive a mousedown on a grabber/resize div/divs, we must decide
   * which grabber "wins."
   *
   * Often two schedule blocks abut each other, or one block is small enough
   * that the top/bottom grabber divs overlap each other. In this case, we
   * examine the total screen range occupied by the two divs and check to see
   * whether the click event occurred closer to the top or to the bottom. The
   * top or bottom grabber, respectively, then wins out over the other.
   *
   * Of course if only one grabber lies under the mouse click, that div is
   * the de facto winner.
   *
   * @inner
   * @param {jQuery} grabs the divs to choose a winner from (these divs should
   * be preselected as those with the <code>grab</code> CSS class)
   * @param {jQuery.Event} event the mousedown event
   * @returns {jQuery} the winning grabber div
   */


  function chooseGrabber(grabs, event) {
    if (grabs.length === 1) {
      return grabs;
    }

    var minGrabTop = Number.MAX_VALUE,
        maxGrabBottom = Number.MIN_VALUE,
        minBlockTop = Number.MAX_VALUE,
        maxBlockBottom = Number.MIN_VALUE,
        highestGrab = null,
        lowestGrab = null;
    grabs.each(function () {
      var grab = $(this),
          grabTop = grab.offset().top,
          grabBottom = grabTop + grab.outerHeight(),
          block = grab.parent(),
          blockTop = block.offset().top,
          blockBottom = blockTop + block.outerHeight();

      if (grabTop < minGrabTop || grabTop === minGrabTop && blockTop < minBlockTop) {
        highestGrab = grab;
      }

      if (grabBottom > maxGrabBottom || grabBottom === maxGrabBottom && blockBottom > maxBlockBottom) {
        lowestGrab = grab;
      }

      minGrabTop = Math.min(grabTop, minGrabTop);
      maxGrabBottom = Math.max(grabBottom, maxGrabBottom);
      minBlockTop = Math.min(blockTop, minBlockTop);
      maxBlockBottom = Math.max(blockBottom, maxBlockBottom);
    });
    return getY(event) < (maxGrabBottom + minGrabTop) / 2 ? highestGrab : lowestGrab;
  }
  /**
   * Applies drag, resize, double click, and other mouse-related functionality
   * to a day editor.
   *
   * This module is documented as an object, but actually exports a function -
   * documented members will be added to the function's target.
   *
   * API Status: **Private**
   * @exports nmodule/schedule/rc/fe/mixin/DayEditorDragSupport
   * @mixin
   *
   * @example
   *   require(['nmodule/schedule/rc/fe/mixin/DayEditorDragSupport'],
   *           function (addDragSupport) {
   *
   *     addDragSupport(myDayEditor);
   *
   *   });
   */


  var exports = {
    /**
     * Create a new schedule block. This will add a new `ScheduleBlock`
     * to the `DayEditor` and build the appropriate
     * `ScheduleBlockEditor`.
     *
     * @private
     * @param {DragOp} op
     * @returns {Promise} promise to be resolved when the new
     * `ScheduleBlockEditor` is built
     */
    $doCreate: function $doCreate(op) {
      var _this = this;

      var builder = this.getBuilder(),
          start = makeTime(op.start),
          finish = makeTime(op.finish || MILLIS_IN_HALF_HOUR);
      return this.createScheduleBlock(start, finish).then(function (scheduleBlock) {
        return _this.addScheduleBlock(scheduleBlock);
      }).then(function () {
        return builder.getKeys();
      }).then(function (keys) {
        var lastKey = _.last(keys);

        return builder.$loadFor(lastKey).then(function () {
          var scheduleBlockEditor = builder.getEditorFor(lastKey);
          op.scheduleBlockEditor = scheduleBlockEditor;
          selectBlock(_this, scheduleBlockEditor);
        });
      });
    },

    /**
     * Perform the logic of resizing the currently selected schedule block.
     * This function will either resize or move the selected block, depending
     * on the current operation.
     *
     * This function will check for collisions and will move the block as far
     * as it can in the desired direction without colliding with another
     * block.
     *
     * Note that target times will always be rounded to the half-hour. If the
     * user wishes to change a schedule per-minute, he or she must edit it
     * by hand.
     *
     * @private
     * @param {Number} delta the amount of time, in milliseconds, we're moving
     * or resizing the currently selected block
     * @param {object} op
     * @returns {Promise|undefined}
     */
    $doTimeChange: function $doTimeChange(delta, op) {
      var that = this,
          operation = op.operation,
          scheduleBlockEditor = op.scheduleBlockEditor,
          opStart = op.start,
          opFinish = op.finish,
          direction = delta > 0 ? 'down' : 'up',
          translator = that.$getTranslator();

      if (operation === 'move') {
        opStart = roundToHalfHour(opStart);
        opFinish = roundToHalfHour(opFinish);
      }

      function doneTrying(delta) {
        return direction === 'down' ? delta < 0 : delta > 0;
      }

      function tryTimeChange(delta) {
        if (doneTrying(delta)) {
          return Promise.resolve();
        }

        var newStart, newFinish;

        switch (operation) {
          case 'move':
            newStart = roundToHalfHour(opStart + delta);
            newFinish = roundToHalfHour(opFinish + delta);
            break;

          case 'resizeBottom':
            newStart = opStart;
            newFinish = Math.min(opFinish + delta, MILLIS_IN_DAY);
            newFinish = Math.max(newFinish, opStart + MILLIS_IN_HALF_HOUR);
            newFinish = roundToHalfHour(newFinish);
            break;

          case 'resizeTop':
            newStart = Math.max(opStart + delta, 0);
            newStart = Math.min(newStart, opFinish - MILLIS_IN_HALF_HOUR);
            newStart = roundToHalfHour(newStart);
            newFinish = opFinish;
            break;
        }

        function tryMove() {
          var startTime = makeTime(newStart),
              finishTime = makeTime(newFinish);

          if (newStart < 0 || newFinish < 0 || !that.canMoveBlockTo(scheduleBlockEditor, startTime, finishTime)) {
            return Promise.reject(new Error('cannot move to this time range'));
          }

          return translator.$toTimeRangeDisplay(startTime, finishTime, that.properties().toValueMap()).then(function (timeRangeDisplay) {
            return scheduleBlockEditor.setTimeRange(startTime, finishTime, timeRangeDisplay);
          });
        }

        return tryMove()["catch"](function () {
          switch (direction) {
            case 'down':
              return tryTimeChange(delta - MILLIS_IN_HALF_HOUR);

            case 'up':
              return tryTimeChange(delta + MILLIS_IN_HALF_HOUR);
          }
        });
      }

      return tryTimeChange(delta);
    },

    /**
     * Get which element is currently under the mouse, priority-selected
     * (grabbers first, then `TimeSchedule` blocks).
     *
     * @private
     * @param {jQuery.Event} e
     * @returns {jQuery}
     */
    $getElementUnderMouse: function $getElementUnderMouse(e) {
      var that = this,
          blocksDiv = that.$getBlocksDisplayElement(),
          underMouse = getDivsUnderMouse(blocksDiv, e);
      return getPriorityDiv(underMouse);
    },

    /**
     * Get the transparent overlay element that intercepts mouse events and
     * delegates them to the proper handlers.
     *
     * @private
     * @returns {jQuery}
     */
    $getOverlayElement: function $getOverlayElement() {
      return this.$getBlocksContainerElement().children('.overlay');
    },
    $editScheduleBlock: function $editScheduleBlock(scheduleBlockEditor) {
      if (scheduleBlockEditor) {
        return this.editScheduleBlock(scheduleBlockEditor);
      }
    },

    /**
     * If a schedule block was double clicked, show a dialog to edit it in
     * more detail.
     *
     * @private
     * @param {jQuery.Event} e
     */
    $handleDblclick: function $handleDblclick(e) {
      var that = this,
          pDiv = that.$getElementUnderMouse(e),
          scheduleBlockEditor = pDiv.data('widget') || pDiv.parent().data('widget');
      return that.$editScheduleBlock(scheduleBlockEditor);
    },

    /**
     * Start a drag operation (move, resize, or create).
     *
     * @private
     * @param {jQuery.Event} e
     */
    $handleMousedown: function $handleMousedown(e) {
      /*
      the retriggered mouse event when clicking away from a context menu causes
      strange behavior, so ignore it.
      TODO: should we even be doing the retrigger in CommandGroupContextMenu?
       */
      if (e.isTrigger && e.which !== 3 && !this.properties().getValue('$receiveTriggeredEvents')) {
        return;
      }

      var that = this,
          pDiv = this.$getElementUnderMouse(e),
          blockEditor,
          operation;

      if (pDiv.hasClass('ScheduleBlockEditor')) {
        blockEditor = pDiv.data('widget');
        operation = 'move';
      } else if (pDiv.hasClass('grab')) {
        pDiv = chooseGrabber(pDiv, e);
        blockEditor = pDiv.parent().data('widget');
        operation = pDiv.hasClass('top') ? 'resizeTop' : 'resizeBottom';
      } else {
        operation = 'create';
      }

      selectBlock(that, blockEditor);

      if (leftClick(e) && !that.isReadonly() && that.isEnabled()) {
        // left click only when enabled and not readonly
        this.$currentOp = startOp(that, blockEditor, operation, e);
      }
    },

    /**
     * Update CSS cursor, and if an operation is underway (i.e. the mouse is
     * down), perform the appropriate work (move, resize, or create).
     *
     * @private
     * @param {jQuery.Event} e
     * returns {Promise|undefined}
     */
    $handleMousemove: function $handleMousemove(e) {
      var that = this,
          op = that.$currentOp,
          pDiv = that.$getElementUnderMouse(e),
          blockEditor = (pDiv.hasClass('grab') ? chooseGrabber(pDiv, e) : pDiv).closest('.ScheduleBlockEditor'),
          valueDisplay = blockEditor.find('.valueDisplay').text(),
          timeDisplay = blockEditor.find('.timeDisplay').text(),
          title = valueDisplay ? "".concat(valueDisplay, " (").concat(timeDisplay, ")") : '',
          cursor = pDiv.hasClass('grab') && !that.isReadonly() && that.isEnabled() ? 'row-resize' : pDiv.hasClass('ScheduleBlockEditor') && !that.isReadonly() && that.isEnabled() ? 'ns-resize' : 'default';
      that.$getOverlayElement().css('cursor', cursor).attr('title', title);

      if (!op) {
        return;
      }

      var deltaMillis = that.$pixelsToMillis(getY(e) - op.startY);

      if (!op.started && Math.abs(deltaMillis) < MILLIS_IN_HALF_HOUR) {
        return; //no change
      }

      op.started = true;

      switch (op.operation) {
        case 'move':
        case 'resizeBottom':
        case 'resizeTop':
          return this.$doTimeChange(deltaMillis, op);

        case 'create':
          if (op.startY < getY(e)) {
            //set new operation sync - otherwise can get multiple mousemove
            //before operation changes from 'create' and therefore multiple
            //blocks created
            op.operation = 'resizeBottom';
            this.$doCreate(op);
          }

      }
    },

    /**
     * Stop any ongoing operation.
     *
     * @private
     */
    $handleMouseup: function $handleMouseup() {
      var op = this.$currentOp;
      this.$getOverlayElement().css('cursor', 'default');

      if (op && op.modified) {
        this.setModified(true);
      }

      delete this.$currentOp;
    },

    /**
     * Converts a pixel distance on a day div into the corresponding time
     * interval. For instance, if `pixels === blocksDiv.height()`, then we're
     * converting the entire length of the day and the result will equal
     * `MILLIS_IN_DAY`.
     *
     * @private
     * @param {Number} pixels the number of pixels on the y axis
     */
    $pixelsToMillis: function $pixelsToMillis(pixels) {
      var blocksDiv = this.$getBlocksDisplayElement();
      return pixels / blocksDiv.outerHeight() * MILLIS_IN_DAY;
    },

    /**
     * Returns true if the pixels are closer to the bottom of the div, this helps determine
     * if the time of day of midnight is for the top or bottom of the div.
     * @private
     * @param {Number} pixels
     * @returns {boolean}
     */
    $closerToBottom: function $closerToBottom(pixels) {
      var blocksDiv = this.$getBlocksDisplayElement();
      return pixels > blocksDiv.outerHeight() / 2;
    },

    /**
     * Arm mouse event handlers on the `DayEditor` element.
     *
     * @private
     * @param {jQuery} dom
     */
    $armHandlers: function $armHandlers(dom) {
      var that = this,
          handlers = that.$handlers = {
        dblclick: function dblclick(e) {
          that.$handleDblclick(e);
        },
        mousedown: function mousedown(e) {
          that.$handleMousedown(e);
        },
        mousemove: function mousemove(e) {
          Promise.resolve(that.$handleMousemove(e))["catch"](Log.error);
        },
        mouseleave: function mouseleave(e) {
          Promise.resolve(that.$handleMousemove(e))["catch"](Log.error);
        },
        mouseup: function mouseup(e) {
          that.$handleMouseup(e);
        },
        touchstart: function touchstart(e) {
          that.$handleMousedown(e);
        },
        touchend: function touchend(e) {
          that.$handleMouseup(e);
        },
        touchmove: function touchmove(e) {
          if (getTouches(e).length === 1) {
            e.preventDefault();
            Promise.resolve(that.$handleMousemove(e))["catch"](Log.error);
          }
        },
        contextmenu: function contextmenu(e) {
          // If the opening of the context menu was from a long touch, we need to delete the
          //  current op, so it is not applied when the menu is closed.
          delete that.$currentOp;
        }
      };

      _.each(that.$handlers, function (handler, event) {
        dom.on(event, '.overlay', handler);
      });

      dom.on('selectstart', false);
      $(document).on('mouseup', handlers.mouseup);
      that.$longPressDelay = 500;
    }
  };

  var addDragSupport = function addDragSupport(target) {
    if (!applyMixin(target, MIXIN_NAME, exports)) {
      return;
    }

    var _doInitialize = target.doInitialize,
        _doDestroy = target.doDestroy;

    target.doInitialize = function (dom) {
      var that = this;
      return Promise.resolve(_doInitialize.apply(that, arguments)).then(function () {
        dom.on(MODIFY_EVENT, '.ScheduleBlockEditor', function () {
          var op = that.$currentOp;

          if (op) {
            op.modified = true;
          }

          return false;
        });
        that.$getBlocksContainerElement().append('<div class="overlay"/>');
        contextMenuOnLongPress(dom, {
          selector: '.overlay'
        });
        that.$armHandlers(dom);
      });
    };

    target.doDestroy = function () {
      $(document).off('mouseup', this.$handlers.mouseup);
      return _doDestroy.apply(this, arguments);
    };
  };

  return addDragSupport;
});
