/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/fe/menu/WeekScheduleContextMenu
 */
define(['bajaux/commands/Command', 'bajaux/commands/CommandGroup', 'jquery', 'Promise', 'nmodule/webEditors/rc/wb/menu/CommandGroupContextMenu'], function (Command, CommandGroup, $, Promise, CommandGroupContextMenu) {
  'use strict'; ////////////////////////////////////////////////////////////////
  // Support commands
  ////////////////////////////////////////////////////////////////

  function applyMFCommand(weekEditor, scheduleBlocks) {
    return new Command({
      module: 'schedule',
      lex: 'commands.applyMF',
      func: function func() {
        return weekEditor.$getApplyMFCommand().invoke(scheduleBlocks);
      }
    });
  }

  function copyDayCommand(menu, selectedDay) {
    return new Command({
      module: 'schedule',
      lex: 'commands.copyDay',
      func: function func() {
        return selectedDay.readScheduleBlocks().then(function (scheduleBlocks) {
          menu.$copyData = scheduleBlocks;
        });
      }
    });
  }

  function pasteDayCommand(selectedDay, copyData) {
    return new Command({
      module: 'schedule',
      lex: 'commands.pasteDay',
      func: function func() {
        return selectedDay.$getApplyCommand().invoke(copyData);
      }
    });
  } ////////////////////////////////////////////////////////////////
  // WeekScheduleContextMenu
  ////////////////////////////////////////////////////////////////

  /**
   * Context menu support for a `WeekEditor`.
   *
   * @class
   * @alias module:nmodule/schedule/rc/fe/menu/WeekScheduleContextMenu
   * @extends module:nmodule/webEditors/rc/wb/menu/CommandGroupContextMenu
   * @param {module:nmodule/schedule/rc/fe/WeekEditor} weekEditor
   */


  var WeekScheduleContextMenu = function WeekScheduleContextMenu(weekEditor) {
    CommandGroupContextMenu.apply(this, arguments);
  };

  WeekScheduleContextMenu.prototype = Object.create(CommandGroupContextMenu.prototype);
  WeekScheduleContextMenu.prototype.constructor = WeekScheduleContextMenu;
  /**
   * Assemble the appropriate `Command`s for a right click on a week editor.
   * Includes one or more of `Delete Event`, `Copy Day`, `Paste Day`,
   * `All Day Event`, `Apply M-F`, `Clear Day`, and/or `Clear Week`, depending
   * on the right-clicked day's configuration and whether copy data is present.
   *
   * @returns {Promise.<module:bajaux/commands/CommandGroup>} resolved to a
   * `bajaux:CommandGroup`
   */

  WeekScheduleContextMenu.prototype.toContextMenu = function (e) {
    var that = this,
        weekEditor = that.$ed,
        selectedDay = $(e.currentTarget).closest('.DayEditor').data('widget'),
        copyData = that.$copyData;

    if (!selectedDay) {
      return Promise.reject(new Error('no day selected'));
    }

    return selectedDay.readScheduleBlocks().then(function (scheduleBlocks) {
      var blockSelected = !!selectedDay.$getSelectedBlock(),
          hasBlocks = !!scheduleBlocks.length,
          commands = [];

      if (selectedDay.isEnabled() && !selectedDay.isReadonly()) {
        if (blockSelected) {
          commands.push(selectedDay.$getDeleteCommand());
        }

        if (copyData) {
          commands.push(pasteDayCommand(selectedDay, copyData));
        }

        commands.push(selectedDay.$getAllDayCommand());

        if (blockSelected) {
          commands.push(selectedDay.$getEditCommand());
        }

        if (hasBlocks) {
          commands.push(applyMFCommand(weekEditor, scheduleBlocks));
          commands.push(copyDayCommand(that, selectedDay));
          commands.push(selectedDay.$getClearCommand());
        }

        commands.push(weekEditor.$getClearWeekCommand());
      } else if (blockSelected) {
        commands.push(selectedDay.$getEditCommand());
      }

      return new CommandGroup({
        commands: commands
      });
    });
  };
  /**
   * Arm a right click listener for the backing `WeekEditor`'s DOM.
   * Context menu will appear only when right-clicking a `DayEditor`.
   */


  WeekScheduleContextMenu.prototype.arm = function () {
    return CommandGroupContextMenu.prototype.arm.call(this, {
      selector: '.DayEditor'
    });
  };

  return WeekScheduleContextMenu;
});
