/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/fe/SpecialEventsEditor
 */
define(['baja!', 'Promise', 'bajaux/events', 'bajaux/Properties', 'bajaux/mixin/responsiveMixIn', 'nmodule/js/rc/log/Log', 'nmodule/schedule/rc/baja/fe/ActiveDatesCalendar', 'nmodule/schedule/rc/baja/fe/BajaDayEditor', 'nmodule/schedule/rc/baja/fe/TimeScheduleEditor', 'nmodule/schedule/rc/baja/util/bindDayEditorToTimeSchedule', 'nmodule/schedule/rc/fe/CalendarEventsEditor', 'nmodule/schedule/rc/fe/templates', 'nmodule/schedule/rc/model/DaySchedule', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/mixin/zoomMixIn'], function (baja, Promise, events, Properties, responsiveMixIn, Log, ActiveDatesCalendar, BajaDayEditor, TimeScheduleEditor, bindDayEditorToTimeSchedule, CalendarEventsEditor, templates, DaySchedule, fe, zoomMixIn) {
  'use strict';

  var scheduleLabels = templates.scheduleLabels;

  var tpl = function tpl(ed) {
    return "\n    <div class=\"row header\">\n      <div class=\"activeDates\"></div>\n    </div>\n    <div class=\"row footer\">\n      <div class=\"calendarEventsList\"></div>\n      <div class=\"specialEventDayEditorContainer\">\n        <div class=\"dayEditorContainer\">\n          <div class=\"dayEditorOuter\">\n            <div class=\"specialEventDayEditor\">\n              ".concat(scheduleLabels(ed.$getTimeLabels()), "\n              <div class=\"dayEditor\"></div>\n            </div>\n          </div>\n        </div>\n        <div class=\"timeSchedule\"></div>\n      </div>\n    </div>");
  };
  /**
   * Editor for a `CalendarSchedule` or the special events tab of a
   * `WeeklySchedule`. Allows maintaining the list of special events and
   * editing the daily schedule for those special events with a `DayEditor`.
   *
   * It supports the following `bajaux` `Properties`:
   *
   * - `effectiveValue`: **required** the initial effective value to use when
   *   adding blocks to the day editor.
   * - `timeLabels`: **required** - an array of the display strings for 3:00 AM
   *  to 9:00 PM in three-hour increments
   *
   * @class
   * @extends module:nmodule/schedule/rc/fe/CalendarEventsEditor
   * @alias module:nmodule/schedule/rc/fe/SpecialEventsEditor
   * @param {Object} params
   * @param {Object} [params.properties]
   * @param {Array.<String>} [params.properties.timeLabels]
   */


  var SpecialEventsEditor = function SpecialEventsEditor(params) {
    var _this = this;

    CalendarEventsEditor.apply(this, arguments);
    this.validators().add(function () {
      return _this.$getTimeScheduleEditor().validate();
    });
    this.on('selectedEventChanged', function (event) {
      var sched = event || _this.value(),
          dayEd = _this.$getDayEditor();

      return Promise.all([_this.$getActiveDatesCalendar().load(sched), !event && Promise.all([dayEd.load(new DaySchedule({
        dayOfWeek: 'monday'
      })), _this.$getTimeScheduleEditor().setReadonly(true), _this.$enableDayEditor(false), _this.$selected = null //TODO: can we make this less stateful
      ])]);
    });
    zoomMixIn(this, {
      zoomTargets: ['.specialEventDayEditor']
    });
    responsiveMixIn(this, {
      'special-events-short': {
        maxHeight: 525
      }
    });
  };

  SpecialEventsEditor.prototype = Object.create(CalendarEventsEditor.prototype);
  SpecialEventsEditor.prototype.constructor = SpecialEventsEditor;

  SpecialEventsEditor.prototype.doChanged = function (name, value) {
    switch (name) {
      case 'facets':
        this.$rebuildTimeScheduleEditor()["catch"](Log.error);
        this.$getDayEditor().properties().addAll(new Properties(value.toObject()));
        break;

      case 'defaultOutput':
        this.$getDayEditor().properties().setValue(name, value);
        break;
    }
  };

  SpecialEventsEditor.prototype.doInitialize = function (dom) {
    var _this2 = this;

    dom.addClass('SpecialEventsEditor').html(tpl(this));
    var props = this.properties(),
        dayEditor = new BajaDayEditor({
      properties: Properties.extend(props, (props.getValue('facets') || baja.Facets.DEFAULT).toObject()),
      enabled: false
    });
    dom.on(events.MODIFY_EVENT, '.DayEditor', function () {
      _this2.setModified(true);

      _this2.$handleModifiedDay()["catch"](Log.error);
    });
    return Promise.all([dayEditor.initialize(dom.find('.dayEditor')), fe.buildFor({
      readonly: true,
      type: ActiveDatesCalendar,
      value: 0,
      //TODO: fe.buildFor chokes on value: null
      dom: dom.find('.activeDates'),
      properties: this.properties().subset(['refBase', 'sourceOrd'])
    }), this.$rebuildTimeScheduleEditor(), CalendarEventsEditor.prototype.doInitialize.call(this, dom.find('.calendarEventsList'))]).then(function () {
      return _this2.$enableDayEditor(false);
    });
  };
  /**
   * Enable / Disable the Day Editor
   *
   * @param {Boolean} enabled - true if the Day Editor should be enabled, false if the
   *  editor should be disabled.
   * @returns {Promise} resolves when the Day Editor is enabled / disabled
   */


  SpecialEventsEditor.prototype.$enableDayEditor = function (enabled) {
    return this.$getDayEditor().setEnabled(enabled);
  };
  /**
   * @returns {boolean}
   * @private
   */


  SpecialEventsEditor.prototype.$includeReference = function () {
    return true;
  };
  /**
   * @param {baja.Component} specialEvents a `schedule:CompositeSchedule`,
   * typically `WeeklySchedule.schedule.specialEvents`.
   * @returns {Promise}
   */


  SpecialEventsEditor.prototype.doLoad = function (specialEvents) {
    return Promise.all([this.$getActiveDatesCalendar().load(specialEvents), CalendarEventsEditor.prototype.doLoad.apply(this, arguments)]);
  };

  SpecialEventsEditor.prototype.doLayout = function () {
    return this.$getActiveDatesCalendar().layout();
  };

  SpecialEventsEditor.prototype.$getActiveDatesCalendar = function () {
    return this.jq().find('.activeDates').data('widget');
  };

  SpecialEventsEditor.prototype.$getDayEditor = function () {
    return this.jq().find('.dayEditor').data('widget');
  };

  SpecialEventsEditor.prototype.$getTimeScheduleEditor = function () {
    return this.jq().find('.timeSchedule').data('widget');
  };

  SpecialEventsEditor.prototype.$rebuildTimeScheduleEditor = function () {
    var _this3 = this;

    var timeScheduleEd = this.$getTimeScheduleEditor(),
        readonly = timeScheduleEd ? timeScheduleEd.isReadonly() : true,
        defaultTimeSchedule = this.$getDefaultTimeSchedule(),
        dom = this.jq();
    return Promise.resolve(timeScheduleEd && timeScheduleEd.destroy()).then(function () {
      return fe.buildFor({
        type: TimeScheduleEditor,
        value: defaultTimeSchedule,
        dom: dom.find('.timeSchedule'),
        properties: (_this3.properties().getValue('facets') || baja.Facets.DEFAULT).toObject(),
        readonly: readonly
      });
    }).then(function () {
      return bindDayEditorToTimeSchedule(function () {
        return _this3.$getDayEditor();
      }, function () {
        return _this3.$getTimeScheduleEditor();
      }, defaultTimeSchedule, _this3.jq());
    });
  };
  /**
   * @override
   * @returns {Promise}
   */


  SpecialEventsEditor.prototype.doRemove = function () {
    if (this.$selected && this.$selected.getSubject().getParent() === null) {
      this.$selected = null;
      return this.$enableDayEditor(false);
    }
  };
  /**
   * Save the day schedule.
   *
   * @private
   * @returns {Promise} resolves when the schedule is saved to the component.
   */


  SpecialEventsEditor.prototype.$doSaveDaySchedule = function () {
    var that = this;

    if (that.$selected) {
      return that.$getDayEditor().read().then(function (readValue) {
        return that.$getTranslator().fromModel(readValue);
      }).then(function (dailySchedule) {
        // Save the 'day' portion of dailySchedule to the loaded component
        var name = that.$selected.getSubject().getName(),
            newDay = dailySchedule.get('day').newCopy();
        return that.value().get(name).set({
          slot: 'day',
          value: newDay
        }).then(function () {
          return dailySchedule;
        });
      });
    }

    return Promise.resolve();
  };
  /**
   * Called when a row is selected.
   *
   * @override
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} selected - row that was selected
   * @returns {Promise} resolves when processing of the selected event completes
   */


  SpecialEventsEditor.prototype.doCalendarEventSelected = function (selected) {
    var that = this,
        dailySchedule = that.value().get(selected.getSubject().getName()); // TODO why is this call to $doSaveDaySchedule necessary?
    // For some reason the final modification to the day schedule isn't being
    // saved.

    return Promise.all([this.$doSaveDaySchedule().then(function () {
      return that.$getTranslator().toModel(dailySchedule);
    }).then(function (daySchedule) {
      that.$selected = selected;
      return that.$getDayEditor().load(daySchedule);
    }).then(function () {
      return that.$enableDayEditor(true);
    })]);
  };

  SpecialEventsEditor.prototype.$handleModifiedDay = function () {
    var _this4 = this;

    return this.$doSaveDaySchedule().then(function () {
      return _this4.emit('selectedEventChanged', _this4.$getSelectedCalendarEvent().getSubject());
    });
  };

  SpecialEventsEditor.prototype.$getTranslator = function () {
    return this.properties().getValue('translator');
  };

  SpecialEventsEditor.prototype.$getDefaultTimeSchedule = function () {
    var timeSchedule = baja.$('schedule:TimeSchedule'),
        effectiveValue = this.$getEffectiveValue(),
        facets = this.$getFacets();
    timeSchedule.add({
      slot: 'effectiveValue',
      value: effectiveValue,
      facets: facets
    });
    return timeSchedule;
  };

  SpecialEventsEditor.prototype.$getEffectiveValue = function () {
    var effectiveValue = this.properties().getValue('effectiveValue');
    return effectiveValue ? effectiveValue.newCopy(true) : effectiveValue;
  };

  SpecialEventsEditor.prototype.$getFacets = function () {
    return this.properties().getValue('facets') || baja.Facets.DEFAULT;
  };

  SpecialEventsEditor.prototype.$getTimeLabels = function () {
    var timeLabels = this.properties().getValue('timeLabels');

    if (!timeLabels) {
      throw new Error('properties.timeLabels is a required parameter for SpecialEventsEditor');
    }

    if (timeLabels.length !== 7) {
      throw new Error('timeLabels length must be 7');
    }

    return timeLabels;
  };
  /**
   * Save abstract schedule onto this special events.
   *
   * @override
   * @param obj
   * @param {String} [obj.slot]
   * @param {component} [obj.value]
   */


  SpecialEventsEditor.prototype.doSaveAbstractSchedule = function (obj) {
    var slot = obj.slot,
        value = obj.value,
        dailySchedule = baja.$('schedule:DailySchedule', {
      days: value
    });
    return CalendarEventsEditor.prototype.doSaveAbstractSchedule.call(this, {
      slot: slot,
      value: dailySchedule
    });
  };
  /**
   * Get the abstract schedule.  Special Events schedule is stored in 'days' slot.
   *
   * @override
   * @param {String} slotName
   */


  SpecialEventsEditor.prototype.getAbstractScheduleForSlot = function (slotName) {
    return this.value().get(slotName).get('days');
  };

  SpecialEventsEditor.prototype.doSave = function () {
    var that = this;
    return that.$doSaveDaySchedule();
  };

  SpecialEventsEditor.prototype.doDestroy = function () {
    this.jq().removeClass('SpecialEventsEditor');
    return this.getChildWidgets().destroyAll();
  };

  return SpecialEventsEditor;
});
