/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/fe/CalendarEventsEditor
 */
define(['baja!', 'lex!schedule', 'jquery', 'Promise', 'underscore', 'bajaux/commands/Command', 'bajaux/mixin/subscriberMixIn', 'bajaux/util/CommandButtonGroup', 'nmodule/js/rc/switchboard/switchboard', 'nmodule/schedule/rc/baja/fe/ReferenceScheduleOrdEditor', 'nmodule/schedule/rc/fe/AbstractScheduleSelector', 'nmodule/schedule/rc/fe/menu/CalendarEventsContextMenu', 'nmodule/schedule/rc/util/scheduleUtils', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/util/ListSelection', 'nmodule/webEditors/rc/util/htmlUtils', 'nmodule/webEditors/rc/wb/commands/RenameCommand', 'nmodule/webEditors/rc/wb/commands/DeleteCommand', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/table/model/Row', 'nmodule/webEditors/rc/wb/table/model/TableModel', 'nmodule/webEditors/rc/wb/table/Table'], function (baja, lexs, $, Promise, _, Command, subscribable, CommandButtonGroup, switchboard, ReferenceScheduleOrdEditor, AbstractScheduleSelector, CalendarEventsContextMenu, scheduleUtils, BaseEditor, fe, feDialogs, ListSelection, htmlUtils, RenameCommand, DeleteCommand, Column, Row, TableModel, Table) {
  'use strict';

  var scheduleLex = lexs[0],
      contextMenuOnLongPress = htmlUtils.contextMenuOnLongPress;

  var tpl = function tpl(ed) {
    return "\n    <div class=\"commands\"></div>\n    <div class=\"calendarEventsTable ux-select-none\"></div>";
  }; ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Calendar Events Table Column
   *
   * @inner
   * @private
   * @class
   */


  var CalendarEventsColumn = function CalendarEventsColumn(name) {
    Column.apply(this, [name, {
      displayName: scheduleLex.get(name)
    }]);
    this.setSortable(false);
  };

  CalendarEventsColumn.prototype = Object.create(Column.prototype);
  CalendarEventsColumn.prototype.constructor = CalendarEventsColumn;
  /**
   * Access the data straight from the row.
   *
   * @param row The row to access the information from.
   * @returns {string} The value to display for the row and column.
   */

  CalendarEventsColumn.prototype.getValueFor = function (row) {
    var name = this.getName();

    if (name === 'summary') {
      // Return result of RPC to get summary
      return row.data('summary');
    } else {
      return unescape(row.getSubject().getName());
    }
  };
  /**
   * Editor for adding, removing, and editing special events from a container
   * component (typically a `WeeklySchedule.schedule.specialEvents`, or a
   * `CalendarSchedule`).
   *
   * When a new event is selected, or the selected event is edited, a
   * `selectedEventChanged` tinyevent will be emitted with the new value of
   * the event. This value may be `null` if events are deselected.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/schedule/rc/fe/CalendarEventsEditor
   */


  var CalendarEventsEditor = function CalendarEventsEditor() {
    BaseEditor.apply(this, arguments);
    switchboard(this, {
      $calendarEventSelected: {
        allow: 'oneAtATime',
        onRepeat: 'queue'
      },
      $doMoveUp: {
        allow: 'oneAtATime',
        onRepeat: 'queue',
        notWhile: '$doMoveDown'
      },
      $doMoveDown: {
        allow: 'oneAtATime',
        onRepeat: 'queue',
        notWhile: '$doMoveUp'
      }
    });
    subscribable(this);
  };

  CalendarEventsEditor.prototype = Object.create(BaseEditor.prototype);
  CalendarEventsEditor.prototype.constructor = CalendarEventsEditor;

  CalendarEventsEditor.prototype.doInitialize = function (dom) {
    var _this = this;

    var that = this;
    that.$buildCommands();
    dom.addClass('CalendarEventsEditor').html(tpl(this));
    var nameColumn = new CalendarEventsColumn('name'),
        summaryColumn = new CalendarEventsColumn('summary'),
        columns = [nameColumn, summaryColumn];
    that.$model = new TableModel({
      columns: columns
    }); //TODO: do we need to fire a click here? See NCCB-31214 - does armRowHandlers in
    //  Table.js need to fire on contextmenu as well

    contextMenuOnLongPress(dom, {
      selector: '.ux-table',
      fireClick: true
    });
    return fe.buildFor({
      dom: $('.commands', dom),
      type: CommandButtonGroup,
      value: that.getCommandGroup(),
      properties: {
        toolbar: true
      },
      formFactor: 'mini'
    }).then(function () {
      new CalendarEventsContextMenu(_this).arm();
    });
  };

  CalendarEventsEditor.prototype.doLoad = function (value) {
    var that = this;
    return that.$removeAllSchedules().then(function () {
      return that.$insertSchedules(value.getSlots().dynamic().is('schedule:AbstractSchedule').toValueArray());
    }).then(function () {
      return fe.buildFor({
        dom: that.jq().find('.calendarEventsTable'),
        type: Table,
        value: that.$getModel(),
        properties: {
          fixedHeaders: true
        }
      });
    }).then(function (table) {
      that.$table = table;
      that.jq().on('dblclick', '.calendarEventsTable tr', function () {
        return that.$editAbstractSchedule();
      });
      var selection = table.$getSelection(); // Only one item at a time can be selected, so each item is "exclusive"

      selection.setExclusiveFilter(function () {
        return true;
      });
      selection.on('changed', function () {
        that.$updateCommands();
        return that.$calendarEventSelected();
      });
    });
  };

  CalendarEventsEditor.prototype.$buildCommands = function () {
    var that = this;
    that.$addCommand = new Command({
      module: 'schedule',
      lex: 'commands.addItem',
      func: function func() {
        return that.$addAbstractSchedule();
      },
      enabled: !that.isReadonly()
    });
    that.$editCommand = new Command({
      module: 'schedule',
      lex: 'commands.editItem',
      enabled: false,
      func: function func() {
        // this.value().get('days') is an AbstractSchedule
        // Just edit it; all of the editors have been created.
        return that.$editAbstractSchedule();
      }
    });
    that.$moveUpCommand = new Command({
      module: 'schedule',
      lex: 'commands.reorder.up',
      enabled: false,
      func: function func() {
        return that.$doMoveUp();
      }
    });
    that.$moveDownCommand = new Command({
      module: 'schedule',
      lex: 'commands.reorder.down',
      enabled: false,
      func: function func() {
        return that.$doMoveDown();
      }
    });
    that.$renameCommand = new Command({
      module: 'schedule',
      lex: 'commands.renameItem',
      enabled: false,
      func: function func() {
        return that.$doRename();
      }
    });
    that.$deleteCommand = new Command({
      module: 'schedule',
      lex: 'commands.deleteItem',
      enabled: false,
      func: function func() {
        return that.$doDelete();
      }
    });
    that.getCommandGroup().add(that.$addCommand);
    that.getCommandGroup().add(that.$editCommand);
    that.getCommandGroup().add(that.$moveUpCommand);
    that.getCommandGroup().add(that.$moveDownCommand);
    that.getCommandGroup().add(that.$renameCommand);
    that.getCommandGroup().add(that.$deleteCommand);
  };

  CalendarEventsEditor.prototype.$insertSchedules = function (schedules) {
    var _this2 = this;

    var batch = new baja.comm.Batch(),
        promises = schedules.map(function (dailySchedule, index) {
      return toLocalizableString(dailySchedule, batch).then(function (displayString) {
        var row = new Row(dailySchedule);
        row.data('summary', displayString);
        row.data('index', index);
        return row;
      });
    });
    batch.commit();
    return Promise.all(promises).then(function (rows) {
      return _this2.$getModel().insertRows(rows, 0);
    }).then(function () {
      return _this2.$reOrderIndices();
    });
  };

  CalendarEventsEditor.prototype.$removeAllSchedules = function () {
    return this.$getModel().removeRows(this.$getModel().getRows());
  };

  CalendarEventsEditor.prototype.$getTable = function () {
    return this.$table;
  };

  CalendarEventsEditor.prototype.$getModel = function () {
    return this.$model;
  };

  CalendarEventsEditor.prototype.doSaveAbstractSchedule = function (_ref) {
    var slot = _ref.slot,
        value = _ref.value;
    //TODO: Component#reorderToTop?
    var sched = this.value(),
        slots = sched.getSlots().dynamic().toArray();
    return sched.add({
      slot: slot,
      value: value
    }).then(function () {
      return sched.reorder([slot].concat(slots));
    }); //send new to top
  };

  CalendarEventsEditor.prototype.$addAbstractSchedule = function () {
    var _this3 = this;

    var schedule = this.value();
    return feDialogs.showFor({
      type: AbstractScheduleSelector,
      value: schedule,
      formFactor: 'compact',
      properties: {
        defaultSlotName: schedule.getUniqueName('Event'),
        includeReference: this.properties().getValue("includeReference"),
        sourceOrd: this.properties().getValue("sourceOrd")
      }
    }).then(function (result) {
      if (!result) {
        return;
      } // Refresh table with the new item


      return _this3.doSaveAbstractSchedule(result).then(function () {
        // Create a table of rows that have already been inserted
        var insertedAlready = {};

        _this3.$getModel().getRows().forEach(function (row) {
          insertedAlready[row.getSubject().getName()] = true;
        }); // Get the list of slots in the edited object


        var newItems = schedule.getSlots().dynamic().is('schedule:AbstractSchedule').toValueArray().filter(function (slot) {
          return !(slot.getName() in insertedAlready);
        });
        return _this3.$insertSchedules(newItems);
      }).then(function () {
        _this3.$getSelection().select(0);

        _this3.setModified(true);
      });
    });
  };
  /**
   * @returns {module:nmodule/webEditors/rc/wb/table/model/Row} Currently selected Rows in the calendarEvents table
   */


  CalendarEventsEditor.prototype.$getSelectedCalendarEvent = function () {
    // Get selected calendar event
    return this.$getSelection().getSelectedElements(this.$getModel().getRows())[0];
  };

  CalendarEventsEditor.prototype.$editAbstractSchedule = function () {
    var that = this,
        selectedRow = this.$getSelectedCalendarEvent(),
        selectedName,
        dailySchedule,
        abstractSchedule,
        showParams;

    if (!selectedRow) {
      return;
    }

    selectedName = selectedRow.getSubject().getName();
    dailySchedule = that.value().get(selectedName);
    abstractSchedule = that.getAbstractScheduleForSlot(selectedName);

    if (baja.hasType(abstractSchedule, 'schedule:ScheduleReference')) {
      showParams = {
        value: abstractSchedule.get('ref'),
        formFactor: 'compact',
        title: unescape(selectedName),
        type: ReferenceScheduleOrdEditor,
        properties: {
          sourceOrd: this.properties().getValue("sourceOrd")
        }
      };
    } else {
      showParams = {
        value: abstractSchedule,
        formFactor: 'compact',
        title: unescape(selectedName),
        properties: {
          includeWeekdays: true
        }
      };
    } //this makes the field editors in the dialog match readonly or !enabled


    showParams.progressCallback = function (msg, arg) {
      switch (msg) {
        case 'initialized':
          return Promise.join(that.isReadonly() && arg.setReadonly(true), !that.isEnabled() && arg.setEnabled(false));
      }
    };

    var copy = abstractSchedule.newCopy();
    return feDialogs.showFor(showParams).then(function (result) {
      if (!result) {
        return;
      }

      if (baja.hasType(result, 'baja:Ord')) {
        abstractSchedule.set({
          slot: 'ref',
          value: result
        });
      }

      if (!abstractSchedule.equivalent(copy)) {
        // Update model with the new displayString
        return toLocalizableString(dailySchedule).then(function (displayString) {
          selectedRow.data('summary', displayString);
          that.setModified(true);
          that.emit('selectedEventChanged', that.$getSelectedCalendarEvent().getSubject());
          return that.$getModel().emit('rowsChanged', [selectedRow]);
        });
      }
    });
  };

  CalendarEventsEditor.prototype.$getAllNames = function () {
    return _.map(this.value().getSlots().dynamic().toArray(), String);
  };

  CalendarEventsEditor.prototype.$getScheduleNames = function () {
    return _.map(this.value().getSlots().dynamic().is('schedule:AbstractSchedule').toArray(), String);
  };

  CalendarEventsEditor.prototype.$doMove = function (direction) {
    var selectedRow = this.$getSelectedCalendarEvent(),
        selectedName,
        swapName,
        names,
        scheduleNames,
        orgIdx,
        idx,
        swapIdx,
        rows;

    if (!selectedRow) {
      return;
    } // In case something is out of order


    this.$reOrderIndices();
    names = this.$getAllNames();
    scheduleNames = this.$getScheduleNames(); // orgIdx is index into "scheduleNames" for the item being moved

    orgIdx = selectedRow.data('index'); // Make sure not moving above the top or below the bottom (this can happen
    // if the command is invoked multiple times quickly, not giving the button
    // a chance to be disabled when it gets to the top or bottom.

    if (orgIdx + direction < 0 || orgIdx + direction >= scheduleNames.length) {
      return;
    } // Find the names of the items being swapped


    selectedName = scheduleNames[orgIdx];
    swapName = scheduleNames[scheduleNames.indexOf(selectedName) + direction]; // idx and swapIdx are the indices into "names" for the items being swapped

    idx = names.indexOf(selectedName);
    swapIdx = names.indexOf(swapName); // Change the order of the slot names

    names[idx] = names[swapIdx];
    names[swapIdx] = selectedName; // Change the indices within the table model

    rows = this.$getModel().getRows();
    rows[orgIdx].data('index', orgIdx + direction);
    rows[orgIdx + direction].data('index', orgIdx); // Sort the model

    this.$getModel().sort(function (row1, row2) {
      return row1.data('index') - row2.data('index');
    });
    this.setModified(true); // Change the item that is selected to be the moved item

    this.$getSelection().select(idx + direction); // Change the order of the slots within the component

    return this.value().reorder(names);
  };

  CalendarEventsEditor.prototype.$doMoveUp = function () {
    return this.$doMove(-1);
  };

  CalendarEventsEditor.prototype.$doMoveDown = function () {
    return this.$doMove(1);
  };

  CalendarEventsEditor.prototype.$doRename = function () {
    var _this4 = this;

    var selectedRow = this.$getSelectedCalendarEvent();

    if (!selectedRow) {
      return;
    }

    var selectedName = selectedRow.getSubject().getName();
    return new RenameCommand(this.value().get(selectedName)).invoke().then(function () {
      _this4.setModified(true);

      _this4.$getModel().emit('rowsChanged', [selectedRow]);
    });
  };
  /**
   * Override this in a derived class to get notified when a slot is removed.
   * @param {String} slot
   */


  CalendarEventsEditor.prototype.doRemove = function (slot) {};

  CalendarEventsEditor.prototype.$doDelete = function () {
    var _this5 = this;

    var selectedRow = this.$getSelectedCalendarEvent();

    if (!selectedRow) {
      return;
    }

    var slot = selectedRow.getSubject().getName();
    var idx = selectedRow.data('index');
    return new DeleteCommand(this.value().get(slot)).invoke().then(function (result) {
      if (result[1] === 'cancel') {
        return;
      }

      _this5.setModified(true);

      return Promise.resolve(_this5.doRemove(slot)).then(function () {
        // Select the next item, or if there is no next, select prev
        var rows = _this5.$getModel().getRows();

        if (rows.length > idx + 1) {
          // Deleting a row, select the next row
          _this5.$getSelection().select(idx + 1);
        } else if (rows.length > 1) {
          // Deleting the last row, select the previous row
          _this5.$getSelection().select(rows.length - 2);
        } else {
          // Nothing is going to be selected, so disable the commands
          // that require a selection.
          _this5.$disableCommandsRequiringSelection();
        }

        return _this5.$getModel().removeRows([selectedRow]);
      }).then(function () {
        return _this5.$reOrderIndices();
      });
    });
  };

  CalendarEventsEditor.prototype.$reOrderIndices = function () {
    // Reorder the row indices
    this.$getModel().getRows().forEach(function (row, index) {
      row.data('index', index);
    });
  };
  /**
   * Override this to change the slot where the abstract schedule for this
   * value is stored.
   *
   * @param {String} slotName
   */


  CalendarEventsEditor.prototype.getAbstractScheduleForSlot = function (slotName) {
    return this.value().get(slotName);
  };

  CalendarEventsEditor.prototype.calendarEventSelectedDone = function () {// This implementation does not do anything; override if a derived class
    // needs to be notified when the calendarEvent selection has resolved
  };

  CalendarEventsEditor.prototype.$disableCommandsRequiringSelection = function () {
    this.$getEditCommand().setEnabled(false);
    this.$getRenameCommand().setEnabled(false);
    this.$getMoveUpCommand().setEnabled(false);
    this.$getMoveDownCommand().setEnabled(false);
    this.$getDeleteCommand().setEnabled(false);
  };
  /**
   * Called when a calendar event in the calendar events table is selected.
   *
   * @returns {Promise}
   */


  CalendarEventsEditor.prototype.$calendarEventSelected = function () {
    var that = this,
        oldSelectedRow = that.$oldSelectedRow,
        selectedRow = that.$getSelectedCalendarEvent();
    that.$oldSelectedRow = selectedRow; // If nothing is selected, disable command buttons

    if (!selectedRow) {
      if (oldSelectedRow) {
        that.emit('selectedEventChanged', null);
      }

      return Promise.resolve(that.$disableCommandsRequiringSelection());
    }

    return Promise.resolve(that.doCalendarEventSelected(selectedRow)).then(function () {
      if (selectedRow !== oldSelectedRow) {
        that.emit('selectedEventChanged', selectedRow.getSubject());
      }

      return that.calendarEventSelectedDone();
    });
  };

  CalendarEventsEditor.prototype.$updateCommands = function () {
    var that = this,
        selection = that.$getSelection();

    if (!that.isReadonly() && that.isEnabled()) {
      that.$getEditCommand().setEnabled(true);
      that.$getRenameCommand().setEnabled(true);
      that.$getDeleteCommand().setEnabled(true);
      that.$getMoveUpCommand().setEnabled(!selection.isSelected(0));
      that.$getMoveDownCommand().setEnabled(!selection.isSelected(that.$getModel().getRows().length - 1));
    } else {
      that.$getEditCommand().setEnabled(true);
    }
  };
  /**
   * Override this to inject additional logic when a calendar event is selected
   * @param {Row} selected - newly selected row
   * @return Promise that resolves when finished handling the selection event
   */


  CalendarEventsEditor.prototype.doCalendarEventSelected = function (selected) {};
  /**
   * @private
   * @returns {module:nmodule/webEditors/rc/util/ListSelection}
   */


  CalendarEventsEditor.prototype.$getSelection = function () {
    return this.$getTable().$getSelection();
  };

  CalendarEventsEditor.prototype.$getEditCommand = function () {
    return this.getCommandGroup().get(1);
  };

  CalendarEventsEditor.prototype.$getMoveUpCommand = function () {
    return this.getCommandGroup().get(2);
  };

  CalendarEventsEditor.prototype.$getMoveDownCommand = function () {
    return this.getCommandGroup().get(3);
  };

  CalendarEventsEditor.prototype.$getRenameCommand = function () {
    return this.getCommandGroup().get(4);
  };

  CalendarEventsEditor.prototype.$getDeleteCommand = function () {
    return this.getCommandGroup().get(5);
  };

  CalendarEventsEditor.prototype.doEnabled = function (enabled) {
    var enabledAndNotReadonly = !this.isReadonly() && enabled;
    this.$addCommand.setEnabled(enabledAndNotReadonly);
    scheduleUtils.configureEditCommandReadonly(this.$editCommand, !enabledAndNotReadonly);
    return this.getChildWidgets().setAllEnabled(enabled);
  };

  CalendarEventsEditor.prototype.doReadonly = function (readonly) {
    var enabledAndNotReadonly = this.isEnabled() && !readonly;
    this.$addCommand.setEnabled(enabledAndNotReadonly);
    scheduleUtils.configureEditCommandReadonly(this.$editCommand, !enabledAndNotReadonly);
    return this.getChildWidgets().setAllReadonly(readonly);
  };

  CalendarEventsEditor.prototype.doDestroy = function () {
    this.jq().removeClass('CalendarEventsEditor');
    return this.getChildWidgets().destroyAll();
  };

  function toLocalizableString(value, batch) {
    return baja.rpc({
      typeSpec: 'schedule:ScheduleRpc',
      methodName: 'toLocalizableString',
      args: [JSON.stringify(baja.bson.encodeValue(value))],
      batch: batch
    });
  }

  function unescape(str) {
    return baja.SlotPath.unescapeFully(str);
  }

  return CalendarEventsEditor;
});
