/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/fe/YearScheduleEditor
 */
define(['baja!', 'lex!schedule', 'bajaux/mixin/batchSaveMixin', 'Promise', 'underscore', 'bajaux/Properties', 'nmodule/webEditors/rc/fe/baja/compat/RelevantStringEditor'], function (baja, lexs, batchSaveMixin, Promise, _, Properties, RelevantStringEditor) {
  'use strict';

  var scheduleLex = lexs[0],
      ANY_YEAR = scheduleLex.get('year.anyYear');
  /**
   * Editor for a `schedule:YearSchedule`. Allow string entry of an arbitrary
   * string; it should parse to a valid year or to `year.anyYear` from the
   * `schedule` lexicon. A `datalist` facet will be constructed to assist the
   * user.
   *
   * @class
   * @alias module:nmodule/schedule/rc/fe/YearScheduleEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/compat/RelevantStringEditor
   */

  var YearScheduleEditor = function YearScheduleEditor(params) {
    var thisYear = this.$thisYear(),
        datalist = [ANY_YEAR].concat(_.range(thisYear, thisYear + 10)).join(';');
    RelevantStringEditor.call(this, _.extend({}, params, {
      properties: Properties.extend({
        datalist: datalist,
        'relevant-force': true,
        'relevant-filter': false
      }, params && params.properties)
    }));
  };

  YearScheduleEditor.prototype = Object.create(RelevantStringEditor.prototype);
  YearScheduleEditor.prototype.constructor = YearScheduleEditor;
  /**
   * WHAT YEAR IS IT
   *
   * @private
   * @returns {Number}
   */

  YearScheduleEditor.prototype.$thisYear = function () {
    return new Date().getFullYear();
  };

  YearScheduleEditor.prototype.doInitialize = function (dom) {
    return Promise.resolve(RelevantStringEditor.prototype.doInitialize.apply(this, arguments)).then(function () {
      dom.find('input').addClass('nofocus');
    });
  };
  /**
   * Load the year. For `-1` will show the "any year" entry from the lexicon.
   *
   * Unusual override of load to force conversion of integer to string.
   */


  YearScheduleEditor.prototype.load = function (year) {
    var str,
        alwaysEffective = year === -1,
        complex = this.getComplex();

    if (baja.hasType(complex, 'schedule:YearSchedule')) {
      if (complex.get('alwaysEffective')) {
        alwaysEffective = true;
      }
    }

    str = alwaysEffective ? ANY_YEAR : String(year);
    return RelevantStringEditor.prototype.load.call(this, str);
  };
  /**
   * Save value to baja:Complex object
   *
   * If readValue === -1, set alwaysEffective = true, otherwise set alwaysEffective = false.
   *
   * @param readValue value read from doRead()
   * @param params additional parameters for saving, such as batch and progressCallback
   *
   * @returns {Promise} resolves when readValue is saved to the Complex object.
   */


  YearScheduleEditor.prototype.saveToComplex = function (readValue, params) {
    var complex = this.getComplex(),
        slot = this.getSlot(),
        progressCallback = params && params.progressCallback,
        batch = params && params.batch,
        proms = [complex.set({
      slot: 'alwaysEffective',
      value: readValue === -1,
      batch: batch
    }), complex.set({
      slot: slot,
      value: readValue,
      batch: batch
    })];

    if (progressCallback) {
      progressCallback(batchSaveMixin.COMMIT_READY);
    }

    return Promise.all(proms);
  };
  /**
   * If the "any year" option is selected, reads `-1`, otherwise parse the
   * entered year as an integer.
   *
   * @returns {Promise} resolved with the value from the editor
   */


  YearScheduleEditor.prototype.doRead = function () {
    return Promise.resolve(RelevantStringEditor.prototype.doRead.apply(this, arguments)).then(function (str) {
      if (str === ANY_YEAR) {
        return -1;
      } else {
        var year = str === ANY_YEAR ? -1 : parseInt(str, 10);

        if (isNaN(year)) {
          throw new Error('invalid year ' + str);
        }

        return year;
      }
    });
  };

  return YearScheduleEditor;
});
