/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/schedule/rc/baja/fe/TimeScheduleTableModel
 */
define(['baja!', 'baja!schedule:TimeSchedule', 'Promise', 'underscore', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/table/model/TableModel'], function (baja, types, Promise, _, Column, TableModel) {
  'use strict'; ////////////////////////////////////////////////////////////////
  // TimeColumn
  ////////////////////////////////////////////////////////////////

  /**
   * Each row will display the `TimeSchedule`'s start time with the user-defined
   * time format.
   * @inner
   * @class
   * @extends module:nmodule/webEditors/rc/wb/table/model/Column
   */

  var TimeColumn = function TriggerStartColumn() {
    Column.call(this, 'time');
  };

  TimeColumn.prototype = Object.create(Column.prototype);
  TimeColumn.prototype.constructor = TimeColumn;

  TimeColumn.prototype.getValueFor = function (row) {
    return getTime(row);
  };

  TimeColumn.prototype.buildCell = function (row, dom) {
    return getTime(row).toTimeString().then(function (str) {
      return dom.text(str);
    });
  }; ////////////////////////////////////////////////////////////////
  // TimeScheduleTableModel
  ////////////////////////////////////////////////////////////////

  /**
   * Table model for holding, adding to, and removing from a list of
   * `TimeSchedule`s, typically as configured * on a `TriggerSchedule`.
   *
   * Use this table model by calling `addTimes` to add new trigger times, and
   * `removeRows` as usual to remove rows. Both will write through immediately
   * to the backing component.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/wb/table/Table
   * @alias module:nmodule/schedule/rc/baja/fe/TimeScheduleTableModel
   * @param {baja.Component} container component containing any number of child
   * `TimeSchedule`s - typically the `times` slot of a `TriggerSchedule`.
   */


  var TimeScheduleTableModel = function TimeListTableModel(container) {
    TableModel.call(this, {
      columns: [new TimeColumn()],
      rows: getTimeSchedules(container)
    });
    this.$comp = container;
    this.sort(byTime); // allow row removals to write through to backing component.

    this.on('rowsRemoved', function (rows) {
      return Promise.all(rows.map(function (row) {
        return container.remove(row.getSubject().getName());
      }));
    });
  };

  TimeScheduleTableModel.prototype = Object.create(TableModel.prototype);
  TimeScheduleTableModel.prototype.constructor = TimeScheduleTableModel;
  /**
   * Add a list of times to the table model, converting each added time to a
   * new `TimeSchedule` slot on the container component. If the time already
   * exists, it will not be added.
   *
   * @param {Array.<baja.Time>} times
   * @returns {Promise.<boolean>} promise to be resolved with `true` if any
   * slots were actually added
   */

  TimeScheduleTableModel.prototype.addTimes = function (times) {
    var _this = this;

    var rows = this.getRows(),
        toAdd = times.filter(function (time) {
      return !contains(rows, time);
    });

    if (!toAdd.length) {
      return Promise.resolve(false);
    }

    var comp = this.$comp;
    return Promise.all(toAdd.map(function (time) {
      return comp.add({
        slot: 'time?',
        value: baja.$('schedule:TimeSchedule', {
          start: time,
          finish: toFinish(time)
        })
      });
    })).then(function (props) {
      return _this.insertRows(props.map(function (p) {
        return comp.get(p);
      }));
    }).then(function () {
      return _this.sort(byTime);
    }).then(function () {
      return true;
    });
  }; ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Array sort function - sort `TimeSchedule`s by time of day.
   */


  function byTime(r1, r2) {
    return getTime(r1).getTimeOfDayMillis() - getTime(r2).getTimeOfDayMillis();
  }
  /**
   * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row>} rows
   * @param {baja.Time} time
   * @returns {boolean} true if the time is already represented in table rows
   */


  function contains(rows, time) {
    return !!_.find(rows, function (row) {
      return getTime(row).equals(time);
    });
  }
  /**
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @returns {baja.Time} start time for the row's `TimeSchedule`
   */


  function getTime(row) {
    return row.getSubject().get('start');
  }
  /**
   * @param {baja.Component} comp
   * @returns {Array.<baja.Component>} all child `TimeSchedule`s
   */


  function getTimeSchedules(comp) {
    return comp.getSlots().is('schedule:TimeSchedule').toValueArray();
  }
  /**
   * @param {baja.Time} start
   * @returns {baja.Time} finish time is configured to be 20 seconds after start
   */


  function toFinish(start) {
    var startMillis = start.getTimeOfDayMillis(),
        finishMillis = startMillis + 20 * baja.RelTime.MILLIS_IN_SECOND;
    return baja.Time.make(finishMillis);
  }

  return TimeScheduleTableModel;
});
