/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Aaron Laniewicz
 */

/**
 * API Status: **Private**
 * @since Niagara 4.3
 * @module nmodule/control/rc/fe/DailyTriggerModeEditor
 */
define([
  'baja!',
  'baja!control:DailyTriggerMode',
  'bajaux/events',
  'hbs!nmodule/control/rc/fe/templates/DailyTriggerModeEditor',
  'lex!control',
  'nmodule/control/rc/baja/DailyTriggerMode',
  'nmodule/webEditors/rc/fe/CompositeEditor',
  'nmodule/webEditors/rc/fe/config/CompositeBuilder',
  'Promise'
], function (
  baja,
  types,
  events,
  template,
  lexs,
  DailyTriggerMode,
  CompositeEditor,
  CompositeBuilder,
  Promise
) {

  'use strict';

  var controlLex = lexs[0],
      daysOfWeekLexEntry = controlLex.get({
        key: 'trigger.daysOfWeek',
        def: 'Days Of Week'
      }),
      randomizationLexEntry = controlLex.get({
        key: 'trigger.randomization',
        def: 'Randomization'
      }),
      timeOfDayLexEntry = controlLex.get({
        key: 'trigger.timeOfDay',
        def: 'Time Of Day'
      });

  /**
   * Field editor for working with `control:DailyTriggerMode` instances.
   *
   * @class
   * @alias module:nmodule/control/rc/fe/DailyTriggerModeEditor
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   */
  var DailyTriggerModeEditor = function () {
    CompositeEditor.apply(this, arguments);
  };
  DailyTriggerModeEditor.prototype = Object.create(CompositeEditor.prototype);
  DailyTriggerModeEditor.prototype.constructor = DailyTriggerModeEditor;


  /**
   * Initialize the `DailyTriggerModeEditor`.
   *
   * @param {jQuery} dom
   * @returns {Promise}
   */
  DailyTriggerModeEditor.prototype.doInitialize = function (dom) {

    dom.html(template({
      daysOfWeekLabel: daysOfWeekLexEntry,
      randomizationLabel: randomizationLexEntry,
      timeOfDayLabel: timeOfDayLexEntry
    }));

    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };


  /**
   * The builder for `DailyTriggerModeEditor` will create the following
   * editors for:
   *
   *   timeOfDay:     `TimeEditor`
   *   randomization: `RelTimeEditor`
   *   daysOfWeek:    `DaysOfWeekBitsEditor`
   *
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  DailyTriggerModeEditor.prototype.makeBuilder = function () {
    var that = this,
        builder = new CompositeBuilder();

    builder.getKeys = function () {
      return ['daysOfWeek', 'randomization', 'timeOfDay'];
    };

    builder.getDomFor = function (key) {
      return that.jq().find('.key-' + key);
    };

    builder.getValueFor = function (key) {
      var dtm = this.getDataSource();

      switch (key) {
        case 'daysOfWeek':
          return dtm.getDaysOfWeek();
        case 'randomization':
          return dtm.getRandomization();
        case 'timeOfDay':
          return dtm.getTimeOfDay();
      }
    };

    builder.getConfigFor = function (key) {
      return { formFactor: 'mini' };
    };

    return builder;
  };

  /**
   * Load the `timeOfDay`, `randomization`, and `daysOfWeek` of the
   * `DailyTriggerMode` into the appropriate child editors.
   *
   * @param {module:nmodule/control/rc/baja/DailyTriggerMode} intervalTriggerMode
   * @returns {Promise}
   */
  DailyTriggerModeEditor.prototype.doLoad = function (intervalTriggerMode) {
    return CompositeEditor.prototype.doLoad.apply(this, arguments);
  };

  /**
   * Read a new `DailyTriggerMode` with current values of `timeOfDay`,
   * `randomization`, and `daysOfWeek`.
   *
   * @returns {Promise}
   */
  DailyTriggerModeEditor.prototype.doRead = function () {
    return this.getBuilder().readAll()
      .spread(function (
        daysOfWeek,
        randomization,
        timeOfDay
      ) {
        return DailyTriggerMode.make({
          daysOfWeek: daysOfWeek,
          randomization: randomization,
          timeOfDay: timeOfDay
        });
      });
  };

  DailyTriggerModeEditor.prototype.$getDaysOfWeekEditor = function () {
    return this.getBuilder().getEditorFor('daysOfWeek');
  };

  DailyTriggerModeEditor.prototype.$getDaysOfWeekLabel = function () {
    return this.jq().find('.daysOfWeekLabel').text();
  };

  DailyTriggerModeEditor.prototype.$getRandomizationEditor = function () {
    return this.getBuilder().getEditorFor('randomization');
  };

  DailyTriggerModeEditor.prototype.$getRandomizationLabel = function () {
    return this.jq().find('.randomizationLabel').text();
  };

  DailyTriggerModeEditor.prototype.$getTimeOfDayEditor = function () {
    return this.getBuilder().getEditorFor('timeOfDay');
  };

  DailyTriggerModeEditor.prototype.$getTimeOfDayLabel = function () {
    return this.jq().find('.timeOfDayLabel').text();
  };
  
  return DailyTriggerModeEditor;
});
