/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Aaron Laniewicz
 */

/**
 * API Status: **Private**
 * @since Niagara 4.3
 * @module nmodule/control/rc/baja/IntervalTriggerMode
 */
define([
  'baja!',
  'bajaScript/baja/obj/objUtil',
  'lex!control'
], function (
  baja,
  objUtil,
  lexs
) {

  'use strict';

  var DaysOfWeekBits = baja.DaysOfWeekBits,
      RelTime = baja.RelTime,
      Simple = baja.Simple,
      Time = baja.Time,

      bajaDef = baja.def,
      callSuper = baja.callSuper,
      objectify = baja.objectify,
      subclass = baja.subclass,

      cacheDecode = objUtil.cacheDecode,
      cacheEncode = objUtil.cacheEncode,

      controlLex = lexs[0],
      intervalDisplayNameLex = controlLex.get({
        key: 'trigger.interval',
        def: 'Interval'
      });

  /**
   * Represents a `control:IntervalTriggerMode` in BajaScript.
   *
   * When creating a `Simple`, always use the `make()` method instead of
   * creating a new object.
   *
   * @class
   * @extends baja.Simple
   * @alias module:nmodule/control/rc/baja/IntervalTriggerMode
   */
  var IntervalTriggerMode = function IntervalTriggerMode() {
    callSuper(IntervalTriggerMode, this, arguments);
  };
  subclass(IntervalTriggerMode, Simple);

  /**
   * Make a new `IntervalTriggerMode` instance.
   *
   * @param {Object} [obj] - Object Literal for the method's arguments.
   *
   * @param {Boolean} [obj.rangeEnabled] - whether or not the timeOfDay range is
   * enabled.
   *
   * @param {baja.Time} [obj.startTime] - the start time of the active range.
   *
   * @param {baja.Time} [obj.endTime] - the end time of the active range.
   *
   * @param {baja.RelTime} [obj.interval] - the amount of time between triggers.
   *
   * @param {baja.DaysOfWeekBits} [obj.daysOfWeek] - the days of week that the
   * trigger will fire.
   *
   * @returns {module:nmodule/control/rc/baja/IntervalTriggerMode}
   */
  IntervalTriggerMode.make = function (obj) {
    var itm = new IntervalTriggerMode();

    obj = objectify(obj);

    itm.$rangeEnabled = bajaDef(
      obj.rangeEnabled,
      !!(obj.startTime || obj.endTime)
    );
    itm.$startTime = bajaDef(obj.startTime, Time.make(0));
    itm.$endTime = bajaDef(obj.endTime, Time.make({
      hour: 23,
      min: 59,
      sec: 59,
      ms: 999
    }));
    itm.$interval = bajaDef(obj.interval, RelTime.make({ minutes: 1 }));
    itm.$daysOfWeek = bajaDef(obj.daysOfWeek, DaysOfWeekBits.DEFAULT);

    return itm;
  };

  /**
   * Make a new `IntervalTriggerMode` instance.
   *
   * @see module:nmodule/control/rc/baja/IntervalTriggerMode.make
   */
  IntervalTriggerMode.prototype.make = function () {
    return IntervalTriggerMode.make.apply(IntervalTriggerMode, arguments);
  };

  /**
   * Decode a `IntervalTriggerMode` from a `String`.
   *
   * @param {String} str - an encoded `IntervalTriggerMode`.
   *
   * @returns {module:nmodule/control/rc/baja/IntervalTriggerMode}
   */
  IntervalTriggerMode.prototype.decodeFromString = cacheDecode(function (str) {
    var tokens = str.split(';');

    return IntervalTriggerMode.make({
      rangeEnabled: Boolean.DEFAULT.decodeFromString(tokens[0]),
      startTime: Time.DEFAULT.decodeFromString(tokens[1]),
      endTime: Time.DEFAULT.decodeFromString(tokens[2]),
      interval: RelTime.DEFAULT.decodeFromString(tokens[3]),
      daysOfWeek: DaysOfWeekBits.DEFAULT.decodeFromString(tokens[4])
    });
  });

  /**
   * Encode the `IntervalTriggerMode` (itself) to a `String`.
   *
   * @returns {String}
   */
  IntervalTriggerMode.prototype.encodeToString = cacheEncode(function () {
    var that = this;

    return that.$rangeEnabled.encodeToString() + ';' +
      that.$startTime.encodeToString() + ';' +
      that.$endTime.encodeToString() + ';' +
      that.$interval.encodeToString() + ';' +
      that.$daysOfWeek.encodeToString();
  });

  /**
   * Default `IntervalTriggerMode` instance.
   *
   * @type {module:nmodule/control/rc/baja/IntervalTriggerMode}
   */
  IntervalTriggerMode.DEFAULT = IntervalTriggerMode.make();

  /**
   * Return the `String` representation of the `IntervalTriggerMode` (itself).
   *
   * @returns {String}
   */
  IntervalTriggerMode.prototype.toString = function () {
    return String(this.$interval) + ' ' + String(this.$daysOfWeek);
  };

  /**
   * Get the display name for this trigger from lexicon.
   *
   * @returns {String}
   */
  IntervalTriggerMode.prototype.getDisplayName = function () {
    return intervalDisplayNameLex;
  };

  /**
   * Is the time of day range enabled?
   *
   * @returns {Boolean}
   */
  IntervalTriggerMode.prototype.isRangeEnabled = function () {
    return this.$rangeEnabled;
  };


  /**
   * Get the start time of the active range.
   *
   * @returns {baja.Time}
   */
  IntervalTriggerMode.prototype.getStartTime = function () {
    return this.$startTime;
  };

  /**
   * Get the end time of the active range.
   *
   * @returns {baja.Time}
   */
  IntervalTriggerMode.prototype.getEndTime = function () {
    return this.$endTime;
  };

  /**
   * Get the amount of time between triggers.
   *
   * @returns {baja.RelTime}
   */
  IntervalTriggerMode.prototype.getInterval = function () {
    return this.$interval;
  };

  /**
   * Get the days of week that the trigger will fire.
   *
   * @returns {baja.DaysOfWeekBits}
   */
  IntervalTriggerMode.prototype.getDaysOfWeek = function () {
    return this.$daysOfWeek;
  };

  return IntervalTriggerMode;
});
