/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/fe/RemoteStationEditor
 */
define(['baja!',
  'baja!niagaraDriver:NiagaraStation',
  'lex!alarm',
  'bajaux/events',
  'jquery',
  'Promise',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'nmodule/webEditors/rc/fe/baja/util/spaceUtils',
  'css!nmodule/alarm/rc/fe/alarmEditors'], function (baja,
                                                     types,
                                                     lexs,
                                                     events,
                                                     $,
                                                     Promise,
                                                     fe,
                                                     BaseEditor,
                                                     spaceUtils) {

  'use strict';

  var DESTROY_EVENT      = events.DESTROY_EVENT,
      DISABLE_EVENT      = events.DISABLE_EVENT,
      ENABLE_EVENT       = events.ENABLE_EVENT,
      INITIALIZE_EVENT   = events.INITIALIZE_EVENT,
      LOAD_EVENT         = events.LOAD_EVENT,
      MODIFY_EVENT       = events.MODIFY_EVENT,
      READONLY_EVENT     = events.READONLY_EVENT,
      WRITABLE_EVENT     = events.WRITABLE_EVENT,

      resolveService     = spaceUtils.resolveService,

      alarmLex           = lexs[0],
      SELECT_STATION_MSG = alarmLex.get('stationRecipient.selectStation');

  function retrieveStationNames(names, parent) {
    return parent.loadSlots()
      .then(function () {
        var promises = [];
        parent.getSlots().properties().each(function (prop) {
          if (prop.getType().is('niagaraDriver:NiagaraStation')) {
            names.push(prop.getName());
          } else if (prop.getType().is('baja:Folder')) {
            promises.push(retrieveStationNames(names, parent.get(prop)));
          }
        });
        return Promise.all(promises);
      });
  }

  /**
   * Editor for selecting a remote station on a `StationRecipient`.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/alarm/rc/fe/RemoteStationEditor
   */
  var RemoteStationEditor = function RemoteStationEditor() {
    BaseEditor.apply(this, arguments);
  };
  RemoteStationEditor.prototype = Object.create(BaseEditor.prototype);
  RemoteStationEditor.prototype.constructor = RemoteStationEditor;

  /**
   * Get the editor holding the station name.
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/StringEditor}
   */
  RemoteStationEditor.prototype.$getStringEditor = function () {
    return this.jq().children('.type-baja-String').data('widget');
  };

  RemoteStationEditor.prototype.$getNiagaraNetwork = function () {
    return this.getOrdBase()
      .then(function (base) {
        return resolveService('niagaraDriver:NiagaraNetwork', { base: base });
      });
  };

  /**
   * Resolve all the station names contained directly underneath the station's
   * `NiagaraNetwork`, or as members of child folders.
   *
   * @private
   * @returns {Promise} promise to be resolved with an array of station names
   */
  RemoteStationEditor.prototype.$getRemoteStationNames = function () {
    var names = [];
    return this.$getNiagaraNetwork()
      .then(function (network) {
        return retrieveStationNames(names, network);
      })
      .catch(function () {
        //if the network could not be found, just resolve empty array
      })
      .then(function () {
        names.sort(function (a, b) {
          return a.toLowerCase().localeCompare(b.toLowerCase());
        });
        return names;
      });
  };

  /**
   * Initializes a string editor with the datalist set to the list of station
   * names retrieved from the station.
   *
   * @param {JQuery} dom
   */
  RemoteStationEditor.prototype.doInitialize = function (dom) {
    var that = this;

    dom.on([
      DESTROY_EVENT,
      DISABLE_EVENT, ENABLE_EVENT,
      INITIALIZE_EVENT, LOAD_EVENT,
      READONLY_EVENT, WRITABLE_EVENT
    ].join(' '), '.editor', false);

    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });

    return that.$getRemoteStationNames()
      .then(function (names) {
        return fe.buildFor({
          dom: $('<span/>').appendTo(dom),
          properties: {
            datalist: names.join(';'),
            placeholder: SELECT_STATION_MSG
          },
          value: '',
          formFactor: 'mini'
        });
      });
  };

  /**
   * Loads in the selected station name. If no station name is loaded the
   * "select station" message will be shown.
   *
   * @param {string} stationName the station name to load
   */
  RemoteStationEditor.prototype.doLoad = function (stationName) {
    return this.$getStringEditor().load(stationName);
  };

  /**
   * Reads the currently entered station name. If the "select station" message
   * is still entered, an empty string will be resolved.
   *
   * @returns {Promise}
   */
  RemoteStationEditor.prototype.doRead = function () {
    return this.$getStringEditor().read()
      .then(function (stationName) {
        return stationName === SELECT_STATION_MSG ? '' : stationName;
      });
  };

  /**
   * Destroy all child editors.
   *
   * @returns {Promise}
   */
  RemoteStationEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  RemoteStationEditor.prototype.doReadonly = function (readonly) {
    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  RemoteStationEditor.prototype.doEnabled = function (enabled) {
    return this.getChildEditors().setAllEnabled(enabled);
  };

  return RemoteStationEditor;
});
