/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/fe/AlarmInstructionsEditor
 */
define(['baja!',
  'lex!alarm',
  'bajaux/Properties',
  'bajaux/Widget',
  'bajaux/commands/Command',
  'bajaux/mixin/responsiveMixIn',
  'bajaux/util/CommandButtonGroup',
  'underscore',
  'nmodule/alarm/rc/baja/AlarmInstructions',
  'nmodule/webEditors/rc/fe/CompositeEditor',
  'nmodule/webEditors/rc/fe/feDialogs',
  'nmodule/webEditors/rc/fe/array/ArrayCompositeEditor',
  'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor',
  'nmodule/webEditors/rc/fe/baja/util/spaceUtils',
  'nmodule/webEditors/rc/fe/baja/util/subscriberUtils',
  'nmodule/webEditors/rc/fe/config/CompositeBuilder',
  'css!nmodule/alarm/rc/fe/alarmEditors'], function (baja,
                                                     lexs,
                                                     Properties,
                                                     Widget,
                                                     Command,
                                                     responsiveMixIn,
                                                     CommandButtonGroup,
                                                     _,
                                                     AlarmInstructions,
                                                     CompositeEditor,
                                                     feDialogs,
                                                     ArrayCompositeEditor,
                                                     DisplayOnlyEditor,
                                                     spaceUtils,
                                                     subscriberUtils,
                                                     CompositeBuilder) {

  'use strict';

  var alarmLex        = lexs[0],
      resolveService  = spaceUtils.resolveService,
      leaseWhenNeeded = subscriberUtils.leaseWhenNeeded,
      getAlarmService = leaseWhenNeeded(_.partial(resolveService,
        'alarm:AlarmService'));

  ////////////////////////////////////////////////////////////////
  // Custom Commands
  ////////////////////////////////////////////////////////////////

  /**
   * Command to add a new alarm instruction.
   *
   * @inner
   * @class
   * @extends {module:bajaux/commands/Command}
   * @param {module:nmodule/alarm/rc/fe/AlarmInstructionsEditor} ed
   */
  var AddCommand = function AddCommand(ed) {
    Command.call(this, {
      module: 'alarm',
      lex: 'commands.addMsg',
      func: function () {
        return ed.$getMasterInstructions()
          .then(function (master) {
            return feDialogs.showFor({
              properties: {
                // if we're already editing the master instruction list,
                // it would be weird to show the datalist.
                datalist: master.equals(ed.value()) ? null :
                  master.getInstructions().join(';')
              },
              value: '',
              formFactor: 'mini'
            });
          })
          .then(function (str) {
            if (!str) {
              return;
            }

            return ed.read()
              .then(function (ai) {
                return ed.$getArrayEditor().load(
                  ai.getInstructions().concat([baja.Format.make(str)]));
              });
          });
      }
    });
  };
  AddCommand.prototype = Object.create(Command.prototype);
  AddCommand.prototype.constructor = AddCommand;

  ////////////////////////////////////////////////////////////////
  // AlarmInstructionsEditor
  ////////////////////////////////////////////////////////////////

  /**
   * Editor for handling `alarm:AlarmInstructions` values.
   *
   * When form factor is `mini`, this will be a display-only editor showing
   * the number of instructions configured.
   *
   * When form factor is `compact`, this will be an editor allowing you to edit
   * all the individual instructions, add and remove them, and change their
   * order.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/alarm/rc/fe/AlarmInstructionsEditor
   */
  var AlarmInstructionsEditor = function AlarmInstructionsEditor() {
    var that = this;

    CompositeEditor.apply(that, arguments);

    if (that.getFormFactor() === Widget.formfactor.mini) {
      DisplayOnlyEditor.$mixin(that);
    }

    var getEd = function () {
      return that.$getArrayEditor();
    };

    that.getCommandGroup().add(
      new AddCommand(that),
      new ArrayCompositeEditor.RemoveCommand(getEd),
      new ArrayCompositeEditor.MoveUpCommand(getEd),
      new ArrayCompositeEditor.MoveDownCommand(getEd)
    );

    responsiveMixIn(this, {
      'alarm-instructions-sm': {
        maxWidth: 525
      }
    });
  };
  AlarmInstructionsEditor.prototype = Object.create(CompositeEditor.prototype);
  AlarmInstructionsEditor.prototype.constructor = AlarmInstructionsEditor;

  /**
   * Configures the composite builder for the loaded `AlarmInstructions`
   * instance.
   *
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  AlarmInstructionsEditor.prototype.makeBuilder = function () {
    var that    = this,
        builder = new CompositeBuilder();

    builder.getKeys = function () {
      return ['array', 'commands'];
    };

    builder.getValueFor = function (key) {
      switch (key) {
        case 'array':
          return this.getDataSource().getInstructions();
        case 'commands':
          return that.getCommandGroup();
      }
    };

    builder.getConfigFor = function (key) {
      switch (key) {
        case 'array':
          return {
            type: ArrayCompositeEditor,
            properties: new Properties({
              numbers: { show: true, countFromZero: false }
            })
          };
        case 'commands':
          return { type: CommandButtonGroup };
      }
    };

    builder.getDomFor = function (key) {
      return that.jq().children('.js-' + key);
    };

    return builder;
  };

  /**
   * Create the DOM structure for housing the ArrayCompositeEditor and
   * the CommandButtonGroup.
   *
   * @param {jQuery} dom
   * @returns {Promise}
   */
  AlarmInstructionsEditor.prototype.doInitialize = function (dom) {
    dom.html('<div class="js-array"/><div class="editor js-commands"/>');
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   * Read a new `alarm:AlarmInstructions` instance from the currently entered
   * instructions.
   *
   * @returns {Promise.<module:nmodule/alarm/rc/baja/AlarmInstructions>}
   */
  AlarmInstructionsEditor.prototype.doRead = function () {
    return this.getBuilder().getEditorFor('array').read()
      .then(function (array) {
        return AlarmInstructions.make(array);
      });
  };

  /**
   * When in mini/display-only mode, just display the number of instructions
   * configured.
   *
   * @param {module:nmodule/alarm/rc/baja/AlarmInstructions} alarmInstructions
   * @returns {String}
   */
  AlarmInstructionsEditor.prototype.valueToString = function (alarmInstructions) {
    return alarmInstructions.size() + ' ' + alarmLex.get('alarm.instructions');
  };

  /**
   * Resolve the `alarm:AlarmService` from the station.
   *
   * @private
   * @returns {Promise.<baja.Component>} promise to be resolved with the `AlarmService`
   */
  AlarmInstructionsEditor.prototype.$getAlarmService = function () {
    return this.getOrdBase()
      .then(function (ordBase) {
        return getAlarmService(ordBase);
      });
  };

  /**
   * Get the master alarm instruction list from the station.
   *
   * @private
   * @returns {Promise.<module:nmodule/alarm/rc/baja/AlarmInstructions>} promise
   * to be resolved with the master `alarm:AlarmInstructions` instance
   * from the `AlarmService`, or with `AlarmInstructions.DEFAULT` if the alarm
   * service could not be found
   */
  AlarmInstructionsEditor.prototype.$getMasterInstructions = function () {
    return this.$getAlarmService()
      .then(function (alarmService) {
        return alarmService.get('masterAlarmInstructions');
      })
      .catch(function () {
        return AlarmInstructions.DEFAULT;
      });
  };

  /**
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/array/ArrayCompositeEditor}
   */
  AlarmInstructionsEditor.prototype.$getArrayEditor = function () {
    return this.getBuilder().getEditorFor('array');
  };

  /**
   * @private
   * @returns {module:bajaux/commands/Command}
   */
  AlarmInstructionsEditor.prototype.$getAddCommand = function () {
    return this.getCommandGroup().get(0);
  };

  return AlarmInstructionsEditor;
});
