/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/db/editors/AlarmTablePaginationModel
 */
define([ 'baja!',
  'baja!alarm:AlarmFilterSet,alarm:AlarmRecord,bql:DynamicTimeRange',
  'Promise',
  'underscore',
  'nmodule/alarm/rc/Alarm',
  'nmodule/alarm/rc/console/table/columns/InfoColumn',
  'nmodule/alarm/rc/db/AlarmPropertyColumn',
  'nmodule/alarm/rc/db/AlarmFacetColumn',
  'nmodule/alarm/rc/util/alarmUtils',
  'nmodule/webEditors/rc/wb/table/model/TableModel',
  'nmodule/webEditors/rc/wb/table/pagination/PaginationModel' ], function (
  baja,
  types,
  Promise,
  _,
  Alarm,
  InfoColumn,
  AlarmPropertyColumn,
  AlarmFacetColumn,
  alarmUtils,
  TableModel,
  PaginationModel) {

  'use strict';

  var getAlarmDataDisplayName = alarmUtils.getAlarmDataDisplayName;

  var visibleColumns = [
    'info',
    'timestamp',
    'sourceState',
    'ackState',
    'source',
    'alarmClass',
    'priority',
    'msgText'
  ];

  /**
   * @class
   * @alias module:nmodule/alarm/rc/db/editors/AlarmTablePaginationModel
   * @extends module:nmodule/webEditors/rc/wb/table/pagination/PaginationModel
   * @param {object} params
   * @param {object} params.displayNamesMap
   * @param {baja.TimeZoneDatabase} params.timeZoneDatabase
   * @param {baja.Simple} [params.timeRange]
   * @param {baja.Complex} [params.filterSet]
   * @param {boolean} [params.sortDescending]
   * @param {string} [params.sortColumn]
   */
  var AlarmTablePaginationModel = function AlarmTablePaginationModel(params) {
    params = params || {};
    PaginationModel.apply(this, arguments);
    this.$displayNamesMap = params.displayNamesMap;
    this.$timeZoneDatabase = params.timeZoneDatabase;
    this.setTimeRange(params.timeRange || baja.$('bql:DynamicTimeRange', 'today'));
    this.setFilterSet(params.filterSet || baja.$('alarm:AlarmFilterSet'));
    this.setSortDescending(!!params.sortDescending);
    this.setSortColumn(params.sortColumn || 'timestamp');
  };
  AlarmTablePaginationModel.prototype = Object.create(PaginationModel.prototype);
  AlarmTablePaginationModel.prototype.constructor = AlarmTablePaginationModel;

  /**
   * @param {number} rowStart
   * @param {number} rowEnd
   * @returns {Promise<module:nmodule/webEditors/rc/wb/table/model/TableModel>}
   */
  AlarmTablePaginationModel.prototype.makeTableModel = function (rowStart, rowEnd) {
    var that = this,
      limit = rowEnd - rowStart;

    return Promise.all([
      Alarm.retrieveAlarms({
        timeRange: this.$timeRange,
        limit: limit,
        offset: rowStart,
        filterSet: this.$filterSet,
        sortDescending: this.$sortDescending,
        sortColumn: this.$sortColumn,
        alarmClassDisplayNames: this.$displayNamesMap,
        timeZoneDatabase: this.$timeZoneDatabase,
        chunkSize: 1000
      }),
      this.$getColumns()
    ])
      .spread(function (obj, columns) {
        that.setRowCount(obj.count);
        return new TableModel({ columns: columns, rows: obj.alarms });
      });
  };

  /**
   * @param {baja.Simple} timeRange a `bql:DynamicTimeRange` to query for
   */
  AlarmTablePaginationModel.prototype.setTimeRange = function (timeRange) {
    this.$timeRange = timeRange;
  };

  /**
   * @param {baja.Complex} filterSet a `bql:FilterSet` to filter results
   */
  AlarmTablePaginationModel.prototype.setFilterSet = function (filterSet) {
    this.$filterSet = filterSet;
  };

  /**
   * @param {boolean} sortDescending
   */
  AlarmTablePaginationModel.prototype.setSortDescending = function (sortDescending) {
    this.$sortDescending = sortDescending;
  };

  /**
   * @param {string} sortColumn column to sort by
   */
  AlarmTablePaginationModel.prototype.setSortColumn = function (sortColumn) {
    this.$sortColumn = sortColumn;
  };

  /**
   * @private
   * @returns {Promise.<object>}
   */
  AlarmTablePaginationModel.prototype.$getDisplayNames = function () {
    return Alarm.getDisplayNamesMap();
  };

  /**
   * @private
   * @returns {Promise.<Array.<module:nmodule/webEditors/rc/wb/table/model/Column>>}
   */
  AlarmTablePaginationModel.prototype.$getColumns = _.once(function () {
    return makeColumns();
  });

  function makeColumns() {
    var facetKeys = [ 'msgText' ],
      record = baja.$('alarm:AlarmRecord'),
      columnNames = record.getSlots().toArray().map(String),
      columns = [ new InfoColumn() ]
        .concat(columnNames.map(function (name) {
          return makePropertyColumn(record, name);
        }))
        .concat(facetKeys.map(makeFacetColumn));

    columns.forEach(function (column) {
      column.setUnseen(!_.contains(visibleColumns, column.getName()));
    });

    return columns;
  }

  /**
   * @returns {module:nmodule/alarm/rc/db/AlarmPropertyColumn}
   */
  function makePropertyColumn(record, slotName) {
    return new AlarmPropertyColumn(record.getSlot(slotName), {
      displayName: record.getDisplayName(slotName)
    });
  }

  /**
   * @returns {module:nmodule/alarm/rc/db/AlarmFacetColumn}
   */
  function makeFacetColumn(key) {
    return new AlarmFacetColumn(key, { displayName: getAlarmDataDisplayName(key) });
  }

  return AlarmTablePaginationModel;
});
