/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/support/SupportFactory
 */
define([
  'nmodule/alarm/rc/console/support/AlarmConsoleSupport',
  'nmodule/alarm/rc/console/support/AlarmConsoleAudioSupport',
  'nmodule/alarm/rc/console/baja/BajaAlarmConsoleSupport',
  'nmodule/alarm/rc/console/baja/BajaAlarmConsoleAudioSupport'], function (AlarmConsoleSupport,
                                                                           AlarmConsoleAudioSupport,
                                                                           BajaAlarmConsoleSupport,
                                                                           BajaAlarmConsoleAudioSupport) {

  'use strict';


  /**
   * SupportFactory allows registration and retrieval of AlarmConsoleSupport implementations
   *
   * @class
   * @alias module:nmodule/alarm/rc/console/support/SupportFactory
   * @see {@link module:nmodule/alarm/rc/console/support/AlarmConsoleSupport}
   */
  var SupportFactory = function () {
    this.$supportClasses = {};
    this.$supportAudioClasses = {};
    this.registerSupport('baja', BajaAlarmConsoleSupport);
    this.registerAudioSupport('baja', BajaAlarmConsoleAudioSupport);
  };

  /**
   * Registers an AlarmConsoleSupport subclass
   *
   * @param name - name to register the support class with
   * @param {Function} supportClass - a subclass of AlarmConsoleSupport
   * @return {Boolean} true if the class was registered, false otherwise
   */
  SupportFactory.prototype.registerSupport = function (name, supportClass) {
    if (supportClass.prototype instanceof AlarmConsoleSupport) {
      this.$supportClasses[name] = supportClass;
      return true;
    }
    return false;
  };

  /**
   * Registers an AlarmConsoleAudioSupport subclass
   *
   * @param name - name to register the support class with
   * @param {Function} supportClass - a subclass of AlarmConsoleAudioSupport
   * @return {Boolean} true if the class was registered, false otherwise
   */
  SupportFactory.prototype.registerAudioSupport = function (name, supportClass) {
    if (supportClass.prototype instanceof AlarmConsoleAudioSupport) {
      this.$supportAudioClasses[name] = supportClass;
      return true;
    }
    return false;
  };

  /**
   * Returns an instance of the registered support class for a given name. If no registered instance
   * is present, defaults to an instance of AlarmConsoleSupport
   *
   * @param name - name key for which class is registered
   * @return {module:nmodule/alarm/rc/console/support/AlarmConsoleSupport}
   */
  SupportFactory.prototype.getSupport = function (name) {
    var SupportClass = this.$supportClasses[name];
    return (SupportClass && new SupportClass()) || new AlarmConsoleSupport();
  };

  /**
   * Returns an instance of the registered support class for a given name. If no registered instance
   * is present, defaults to an instance of AlarmConsoleSupport
   *
   * @param name - name key for which class is registered
   * @return {module:nmodule/alarm/rc/console/support/AlarmConsoleSupport}
   */
  SupportFactory.prototype.getAudioSupport = function (name) {
    var SupportClass = this.$supportAudioClasses[name];
    return (SupportClass && new SupportClass()) || new AlarmConsoleAudioSupport();
  };

  /**
   * Returns an instance of the registered support class for a given name. If no registered instance
   * is present, defaults to an instance of AlarmConsoleSupport
   *
   * @static
   * @param name - name key for which class is registered
   * @return {module:nmodule/alarm/rc/console/support/AlarmConsoleSupport}
   */
  SupportFactory.getSupport = function (name) {
    return new SupportFactory().getSupport(name);
  };

  /**
   * Returns an instance of the registered support class for a given name. If no registered instance
   * is present, defaults to an instance of AlarmConsoleSupport
   *
   * @static
   * @param name - name key for which class is registered
   * @return {module:nmodule/alarm/rc/console/support/AlarmConsoleAudioSupport}
   */
  SupportFactory.getAudioSupport = function (name) {
    return new SupportFactory().getAudioSupport(name);
  };

  return SupportFactory;
});
