/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/fe/OptionsEditor
 */
define([
  'baja!',
  'lex!alarm',
  'Promise',
  'bajaux/commands/Command',
  'bajaux/commands/CommandGroup',
  'bajaux/util/CommandButtonGroup',
  'nmodule/webEditors/rc/fe/CompositeEditor',
  'nmodule/webEditors/rc/fe/config/CompositeBuilder',
  'nmodule/webEditors/rc/wb/PropertySheet',
  'nmodule/alarm/rc/console/audio/AlarmAudioHandler',
  'nmodule/alarm/rc/console/audio/AlarmAudioQueue'
], function (baja,
             lexs,
             Promise,
             Command,
             CommandGroup,
             CommandButtonGroup,
             CompositeEditor,
             CompositeBuilder,
             PropertySheet,
             AlarmAudioHandler,
             AlarmAudioQueue) {

  'use strict';

  var alarmLex = lexs[0];

  /**
   * Editor for editing options related to the bajaux Alarm Console.
   *
   * This editor requires an instance of
   * {@link nmodule/alarm/rc/console/AlarmConsoleViewModel} to work
   * properly. Pass this dependency to the constructor as a part of its
   * {@link bajaux/Properties} with the key `viewModel`.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/CompositeEditor
   * @alias module:nmodule/alarm/rc/console/fe/OptionsEditor
   */
  var OptionsEditor = function OptionsEditor() {
    CompositeEditor.apply(this, arguments);
    this.$viewModel = this.properties().getValue('viewModel');
  };
  OptionsEditor.prototype = Object.create(CompositeEditor.prototype);
  OptionsEditor.prototype.constructor = OptionsEditor;

  /**
   * Takes a subset of a plain alarm console options object and turns it into a
   * baja:Component for easy editing of multiple properties at once.
   *
   * @private
   * @param {Object} options alarm console options
   * @returns {baja.Component}
   */
  OptionsEditor.prototype.$makeOptionsComponent = function (options) {
    var viewModel        = this.$viewModel,
        optionsComponent = baja.$('baja:Component'),
        viewOptions      = viewModel.$isSingleSourceModel() ? options.singleSourceView : options.multiSourceView;

    optionsComponent.add({
      slot: 'notesRequiredOnAck',
      cx: {displayName: alarmLex.get('alarm.console.commands.options.notesRequiredOnAck')},
      value: viewOptions.notesRequiredOnAck
    });

    if (viewModel.$isSingleSourceModel()) {
      optionsComponent.add({
        slot: 'pageSize',
        cx: {displayName: alarmLex.get('alarm.console.commands.options.pageSize')},
        value: baja.Integer.make(viewOptions.pageSize),
        facets: baja.Facets.make({min: 1})
      });
    } else {
      optionsComponent.add({
        slot: 'soundDelay',
        cx: {displayName: alarmLex.get('alarm.console.commands.options.soundDelay')},
        value: baja.RelTime.make(viewOptions.soundDelay || AlarmAudioQueue.DEFAULT_ALARM_DELAY),
        facets: baja.Facets.make({min: baja.RelTime.make(1000)})
      });
      optionsComponent.add({
        slot: 'soundFile',
        cx: {displayName: alarmLex.get('alarm.console.commands.options.soundFile')},
        value: viewOptions.soundFile || AlarmAudioHandler.DEFAULT_ALARM_SOUND
      });
      optionsComponent.add({
        slot: 'disableRowHighlight',
        cx: {displayName: alarmLex.get('alarm.console.commands.options.disableRowHighlight')},
        value: !!(viewOptions.disableRowHighlight || false)
      });
    }

    return optionsComponent;
  };

  /**
   * @returns {module:nmodule/webEditors/rc/fe/config/CompositeBuilder}
   */
  OptionsEditor.prototype.makeBuilder = function () {
    var that    = this,
        builder = new CompositeBuilder();

    builder.getKeys = function () {
      return ['options'];
    };

    builder.getValueFor = function (key) {
      switch (key) {
        case 'options':
          return that.$makeOptionsComponent(this.getDataSource());
      }
    };

    builder.getConfigFor = function (key) {
      switch (key) {
        case 'options':
          return {
            formFactor: 'max',
            properties: {showHeader: false},
            type: PropertySheet
          };
      }
    };

    builder.getDomFor = function (key) {
      return that.jq().children('.js-' + key);
    };

    return builder;
  };

  /**
   * @param {jQuery} dom
   */
  OptionsEditor.prototype.doInitialize = function (dom) {
    dom.addClass('OptionsEditor');
    dom.html('<div class="js-options"/>');
    return CompositeEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   * @returns {Promise.<baja.Component>}
   */
  OptionsEditor.prototype.doRead = function () {
    return this.getBuilder().getEditorFor('options').value();
  };

  /**
   * Save the value from the options editor and set the alarm console
   * options on the server side using the userData API.
   *
   * @returns {Promise}
   */
  OptionsEditor.prototype.doSave = function (value) {
    var that      = this,
        viewModel = that.$viewModel,
        support   = viewModel.$getSupport();

    return CompositeEditor.prototype.doSave.apply(that, arguments)
      .then(function () {
        if (viewModel.$isSingleSourceModel()) {
          return support.setAlarmConsoleOptions({
            notesRequiredOnAck: value.get('notesRequiredOnAck'),
            pageSize: value.get('pageSize').valueOf()
          }, true);
        }
        else {
          return support.setAlarmConsoleOptions({
            notesRequiredOnAck: value.get('notesRequiredOnAck'),
            soundDelay: value.get('soundDelay').getMillis(),
            soundFile: String(value.get('soundFile')),
            disableRowHighlight: value.get('disableRowHighlight').getBoolean()
          }, false);
        }
      });
  };

  return OptionsEditor;
});
