/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Jeremy Narron
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/commands/DetailsCommand
 */
define([
  'lex!alarm,bajaui',
  'jquery',
  'dialogs',
  'Promise',
  'bajaux/commands/Command',
  'nmodule/alarm/rc/console/commands/AcknowledgeCommand',
  'nmodule/alarm/rc/console/commands/HyperlinkCommand',
  'nmodule/alarm/rc/console/commands/NotesCommand',
  'nmodule/alarm/rc/console/commands/SequenceCommand',
  'nmodule/alarm/rc/console/fe/DetailsEditor',
  'nmodule/js/rc/log/Log',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn'
], function (lexs,
             $,
             dialogs,
             Promise,
             Command,
             AcknowledgeCommand,
             HyperlinkCommand,
             NotesCommand,
             SequenceCommand,
             DetailsEditor,
             Log,
             fe,
             alarmConsoleCommandMixIn) {
  'use strict';

  var lex       = lexs[0],
      bajauiLex = lexs[1];

  function updateCommands(dialog, subject) {
    if (subject._ackState === 'acked') {
      dialog.disableButton('acknowledge');
    }
    if (!(subject.alarmData && subject.alarmData.hyperlinkOrd)) {
      dialog.disableButton('hyperlink');
    }
  }

  function details(cmd) {
    var ed,
        support      = cmd.$support,
        ackCmd       = cmd.$ackCmd,
        hyperlinkCmd = cmd.$hyperlinkCmd,
        notesCmd     = cmd.$notesCmd,
        sequenceCmd  = cmd.$sequenceCmd;

    return sequenceCmd.init().then(function (subject) {
      if (!subject) {
        return;
      }
      return dialogs.show({
        content: function (dialog, dom) {
          return fe.buildFor({
            dom: $('<div/>').appendTo(dom),
            value: subject,
            type: DetailsEditor,
            formFactor: 'compact',
            properties: {support: support}
          }).then(function (e) {
            ed = e;
            updateCommands(dialog, ed.value());
          });
        },
        buttons: [
          {
            name: 'previous',
            displayName: bajauiLex.get('commands.back.label'),
            handler: function (dialog) {
              dialog.enableButton('next');
              sequenceCmd.previous().then(function (subject) {
                if (subject) {
                  return ed.load(subject);
                }
                else {
                  dialog.disableButton('previous');
                }
              }).catch(Log.error);
              return false;
            }
          },
          {
            name: 'next',
            displayName: bajauiLex.get('commands.forward.label'),
            handler: function (dialog) {
              dialog.enableButton('previous');
              sequenceCmd.next().then(function (subject) {
                if (subject) {
                  return ed.load(subject);
                }
                else {
                  dialog.disableButton('next');
                }
              }).catch(Log.error);
              return false;
            }
          },
          {
            name: 'acknowledge',
            displayName: lex.get('commands.acknowledge.displayName'),
            handler: function (dialog) {
              ackCmd.invoke({subject: ed.value()}).then(function (failures) {
                if (!failures || failures.length === 0) {
                  dialog.disableButton('acknowledge');
                }
              }).catch(Log.error);
              return false;
            }
          },
          {
            name: 'hyperlink',
            displayName: lex.get('commands.hyperlink.displayName'),
            handler: function () {
              hyperlinkCmd.invoke({subject: ed.value()})
                .catch(Log.error);
              return false;
            }
          },
          {
            name: 'notes',
            displayName: lex.get('commands.notes.displayName'),
            handler: function () {
              cmd.$support.getNotes({uuid: ed.value().uuid})
                .then(function (notes) {
                  var subject = ed.value();
                  subject.alarmData.notes = notes;
                  return notesCmd.invoke({callThrough: true, subject: subject})
                    .catch(Log.error);
                });
              return false;
            }
          },
          {
            name: 'close',
            displayName: lex.get('command.closeNoteDialog.label'),
            handler: function () {
              return Promise.all([sequenceCmd.reset(), ed && ed.destroy()]);
            },
            esc: true
          }
        ]
      }).promise();
    });
  }

  /**
   * Shows the currently selected alarm's details in a pop up dialog box.
   *
   * @constructor
   * @param {module:nmodule/alarm/rc/console/AlarmConsole} view The alarm console view.
   * @extends {module:bajaux/commands/Command}
   * @mixes {module:nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn}
   */
  var DetailsCommand = function (view) {
    var that = this;
    Command.apply(that, [{
      module: 'alarm',
      lex: 'alarm.console.commands.details',
      func: function () {
        return details(that);
      }
    }]);

    alarmConsoleCommandMixIn(that);

    that.$view = view;
    that.$viewModel = view.$getViewModel();
    that.$support = that.$viewModel.$getSupport();
    that.$ackCmd = new AcknowledgeCommand(view, {details: true});
    that.$hyperlinkCmd = new HyperlinkCommand(view, {details: true});
    that.$notesCmd = new NotesCommand(view, {details: true});
    that.$sequenceCmd = new SequenceCommand(view);
    that.$showInMenu = true;
  };

  DetailsCommand.prototype = Object.create(Command.prototype);
  DetailsCommand.prototype.constructor = DetailsCommand;

  DetailsCommand.prototype.$getAcknowledgeCommand = function () {
    return this.$ackCmd;
  };

  DetailsCommand.prototype.$getHyperlinkCommand = function () {
    return this.$hyperlinkCmd;
  };

  DetailsCommand.prototype.$getNotesCommand = function () {
    return this.$notesCmd;
  };

  DetailsCommand.prototype.$getSequenceCommand = function () {
    return this.$sequenceCmd;
  };

  DetailsCommand.prototype.selectionChanged = function () {
    this.setEnabled(this.$view.getSelectedAlarmRecords().length === 1);
  };

  return DetailsCommand;
});
