/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/commands/AcknowledgeCommand
 */
define([
  'baja!',
  'lex!alarm',
  'Promise',
  'underscore',
  'dialogs',
  'bajaux/commands/Command',
  'nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn'
], function (baja,
             lexs,
             Promise,
             _,
             dialogs,
             Command,
             alarmConsoleCommandMixIn) {
  'use strict';

  var lex              = lexs[0],
      ERROR_MSG_PREFIX = lex.get('alarm.console.acknowledge.insufficient.permissions');

  function getSelectedSubjects(table) {
    return _.invoke(table.getSelectedRows(), 'getSubject');
  }

  function makeAckParams(view, params) {
    var ackParams    = {},
        table        = view.$getAlarmTable(),
        singleSource = view.isSingleSourceView(),
        mostRecent   = params && params.mostRecent,
        details      = params && params.details,
        subject      = details && params.subject;

    if (subject) {
      ackParams.ids = [subject.uuid];
    }
    else if (singleSource) {
      ackParams.ids = _.pluck(getSelectedSubjects(table), 'uuid');
    }
    else {
      ackParams.srcs = _.pluck(getSelectedSubjects(table), 'source');
    }
    if (mostRecent) {
      ackParams.mostRecent = mostRecent;
    }
    return ackParams;
  }

  function reportError(failureList) {
    if (_.isArray(failureList) && failureList.length) {
      var errorMsg = ERROR_MSG_PREFIX + "<br/><br/>" + failureList.join("<br/>");
      dialogs.showOk(errorMsg);
    }
    return failureList;
  }

  function acknowledge(view, params) {
    var viewModel = view.$getViewModel(),
        support   = viewModel.$getSupport(),
        singleSrc = viewModel.$isSingleSourceModel(),
        options   = support.getAlarmConsoleOptions(),
        notesCmd  = view.$notesCmd,
        ackParams = makeAckParams(view, params),
        notesOnAck;

    options = singleSrc ? options.singleSourceView : options.multiSourceView;
    notesOnAck = options && options.notesRequiredOnAck;

    if (_.isEmpty(ackParams.srcs) && _.isEmpty(ackParams.ids)) {
      return Promise.resolve();
    }

    if (notesOnAck) {
      return notesCmd.invoke({callThrough: false})
        .then(function (notes) {
          if (notes) {
            ackParams.notes = notes;
            return Promise.resolve(dialogs.showLoading(0,
              viewModel.ackAlarms(ackParams).then(reportError)));
          }
        });
    }

    else {
      return Promise.resolve(dialogs.showLoading(0,
          viewModel.ackAlarms(ackParams).then(reportError)));
    }
  }

  /**
   * Acknowledges all selected alarms or just the most recent.
   *
   * @constructor
   * @param {module:nmodule/alarm/rc/console/AlarmConsole} view The alarm console view.
   * @param {Object} params
   * @param {Boolean} params.mostRecent If true then only the most recent alarm is
   * acknowledged.
   * @extends {module:bajaux/commands/Command}
   * @mixes {module:nmodule/alarm/rc/console/commands/alarmConsoleCommandMixIn}
   */
  var AcknowledgeCommand = function (view, params) {
    Command.apply(this, [{
      module: 'alarm',
      lex: params && params.mostRecent ? 'commands.acknowledgeRecent' : 'commands.acknowledge',
      func: function (ackParams) {
        return acknowledge(view, _.extend({}, params, ackParams));
      }
    }]);

    alarmConsoleCommandMixIn(this);

    this.$view = view;
    this.$mostRecent = params && params.mostRecent;
  };

  AcknowledgeCommand.prototype = Object.create(Command.prototype);
  AcknowledgeCommand.prototype.constructor = AcknowledgeCommand;

  AcknowledgeCommand.prototype.selectionChanged = function () {
    this.setEnabled(!this.$view.isSelectionEmpty() && (!this.$mostRecent || !this.$view.isSingleSourceView()));
  };

  AcknowledgeCommand.prototype.isShownInMenu = function () {
    return !this.$mostRecent || !this.$view.isSingleSourceView();
  };

  AcknowledgeCommand.prototype.isShownInCommandBar = function () {
    return !this.$mostRecent;
  };

  return AcknowledgeCommand;
});
