/**
 * @copyright 2017 Tridium, Inc. All Rights Reserved.
 * @author Danesh Kamal
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/alarm/rc/console/baja/BajaAlarmConsoleSupport
 */
define([
  'baja!',
  'baja!bql:DynamicTimeRange,baja:FrozenEnum,' +
  'alarm:AlarmRecord,alarm:SourceState,alarm:AckState,' +
  'bql:FilterEntry,alarm:AlarmFilterSet,' +
  'bql:BooleanFilter,bql:EnumFilter,bql:StringFilter,bql:IntegerFilter,bql:FacetsFilter',
  'lex!alarm',
  'jquery',
  'underscore',
  'moment',
  'Promise',
  'nmodule/webEditors/rc/fe/fe',
  'nmodule/js/rc/tinyevents/tinyevents',
  'nmodule/webEditors/rc/wb/table/model/TableModel',
  'nmodule/webEditors/rc/wb/table/model/Column',
  'nmodule/webEditors/rc/wb/table/model/Row',
  'nmodule/webEditors/rc/util/textUtils',
  'nmodule/alarm/rc/console/support/AlarmConsoleSupport',
  'nmodule/alarm/rc/console/baja/BajaDataChannel',
  'nmodule/alarm/rc/console/table/columns/AckStateColumn',
  'nmodule/alarm/rc/console/table/columns/AlarmDataColumn',
  'nmodule/alarm/rc/console/table/columns/CustomAlarmDataColumn',
  'nmodule/alarm/rc/console/table/columns/DefaultColumn',
  'nmodule/alarm/rc/console/table/columns/InfoColumn',
  'nmodule/alarm/rc/console/table/columns/SourceColumn',
  'nmodule/alarm/rc/console/table/columns/TimestampColumn',
  'nmodule/alarm/rc/console/table/columns/SelectColumn',
  'nmodule/webEditors/rc/servlets/registry'
], function (baja,
             types,
             lexs,
             $,
             _,
             moment,
             Promise,
             fe,
             tinyevents,
             TableModel,
             Column,
             Row,
             textUtils,
             AlarmConsoleSupport,
             BajaDataChannel,
             AckStateColumn,
             AlarmDataColumn,
             CustomAlarmDataColumn,
             DefaultColumn,
             InfoColumn,
             SourceColumn,
             TimestampColumn,
             SelectColumn,
             registry) {
  'use strict';

  var RECIPIENT_UPDATE_EVENT     = 'recipientUpdate',
      SINGLE_SOURCE_UPDATE_EVENT = 'singleSourceUpdate',
      OPTIONS_CHANGED_EVENT      = 'options',
      lex                        = lexs[0],
      alarmImg                   = lex.get('alarm.console.columns.icon.alarm'),
      redAlarmImg                = lex.get('alarm.console.columns.icon.alarmRed'),
      greenAlarmImg              = lex.get('alarm.console.columns.icon.alarmGreen'),
      orangeAlarmImg             = lex.get('alarm.console.columns.icon.alarmOrange'),
      whiteAlarmImg              = lex.get('alarm.console.columns.icon.alarmWhite'),
      notesImg                   = lex.get('alarm.console.columns.icon.note'),
      videoImg                   = lex.get('alarm.console.columns.icon.video'),
      hyperlinkImg               = lex.get('alarm.console.columns.icon.hyperlink'),
      columnNames                = [
        'info',
        'timestamp',
        'source',
        'sourceState',
        'priority',
        'ackState',
        'alarmClass',
        'uuid',
        'ackRequired',
        'normalTime',
        'ackTime',
        'user',
        'alarmData',
        'alarmTransition',
        'lastUpdate'
      ],
      imgMap                     = {
        'a': alarmImg,
        'r': redAlarmImg,
        'g': greenAlarmImg,
        'o': orangeAlarmImg,
        'w': whiteAlarmImg,
        'n': notesImg,
        'v': videoImg,
        'h': hyperlinkImg
      };

  function makeColumns(support) {
    var columns = [ new SelectColumn() ];

    _(columnNames).each(function (name) {
      var column;

      switch (name) {
        case 'ackState':
          column = new AckStateColumn();
          break;
        case 'alarmData':
          column = new AlarmDataColumn();
          break;
        case 'info':
          column = new InfoColumn();
          break;
        case 'source':
          column = new SourceColumn();
          break;
        case 'timestamp':
        case 'normalTime':
        case 'ackTime':
        case 'lastUpdate':
          column = new TimestampColumn(support, name);
          column.setHidable(name !== 'timestamp');
          break;
        default:
          column = new DefaultColumn(name);
      }

      columns.push(column);
    });

    return columns;
  }

  function preloadIcons() {
    var img = new Image();
    _(imgMap).each(function (imgSrc) {
      img.src = imgSrc;
    });
  }

  function loadFilterSet(filterSet) {
    var alarmRecord   = baja.$('alarm:AlarmRecord'),
        props         = alarmRecord.getSlots().properties().frozen().toArray(),
        excludedProps = ['timestamp', 'uuid'],
        propNames;

    props = _.reject(props, function (prop) {
      return _.contains(excludedProps, prop.getName());
    });

    propNames = _.map(props, function (p) {
      return p.getName();
    });

    return Promise.map(props, function (prop) {
      var name        = prop.getName(),
          displayName = alarmRecord.getDisplayName(prop),
          value       = alarmRecord.get(prop),
          typeSpec    = value.getType().getTypeSpec();

      return registry.getAgentOnInfo(typeSpec, {is: 'bql:IBqlFilter'}).then(function (agents) {
        if (agents.length) {
          var filterType  = agents[0].type,
              filter      = baja.$(filterType),
              filterEntry = baja.$('bql:FilterEntry', {label: displayName, filter: filter});

          if (filterType === 'bql:EnumFilter' && value.getType().is('baja:FrozenEnum')) {
            filter.setEnumType(value.asDynamic());
          }

          filterSet.add({slot: name, value: filterEntry});
        }
      });
    }).then(function () {
      return filterSet.reorder(propNames);
    }).return(filterSet);
  }

  function toSummaryParams(support, params) {
    var source    = params && params.source || 'null',
        timeRange = params && params.timeRange || support.getDefaultTimeRange(),
        filterSet = params && params.filterSet || support.getDefaultFilterSet(),
        pageSize  = support.getDefaultPageSize(),
        pageIndex = support.getDefaultPageIndex(),
        offset    = params ? typeof params.offset === 'number' ? params.offset : (pageIndex - 1) * pageSize : 0,
        limit     = params ? typeof params.limit === 'number' ? params.limit : pageSize : pageSize,
        restart   = params && params.restart,
        column    = params && params.column || 'priority',
        sortDesc  = params && params.sortDesc || false;

    if (!_.contains(columnNames, column)) {
      column = "alarmData." + column;
    }

    return {
      source: source,
      timeRange: timeRange,
      filterSet: filterSet,
      offset: offset,
      limit: limit,
      restart: restart,
      column: column,
      sortDesc: sortDesc
    };
  }

  /**
   * BajaAlarmConsoleSupport is a baja-specific implementation of the
   * AlarmConsoleSupport interface.
   *
   * @class
   * @alias module:nmodule/alarm/rc/console/baja/BajaAlarmConsoleSupport
   * @extends module:nmodule/alarm/rc/console/support/AlarmConsoleSupport
   */
  var BajaAlarmConsoleSupport = function () {
    tinyevents(this);
    preloadIcons();
  };

  BajaAlarmConsoleSupport.prototype = Object.create(AlarmConsoleSupport.prototype);
  BajaAlarmConsoleSupport.prototype.constructor = BajaAlarmConsoleSupport;

  /////////////////////////////////////////////////////////////////////////////////
  //Utilities
  /////////////////////////////////////////////////////////////////////////////////
  /**
   * @param {Number} millis
   * @param {baja.TimeZone} timeZone
   * @returns {String}
   */
  BajaAlarmConsoleSupport.prototype.createDateString = function (millis, timeZone) {
    return BajaDataChannel.createDateString(millis, timeZone);
  };

  /////////////////////////////////////////////////////////////////////////////////
  //Lifecycle
  /////////////////////////////////////////////////////////////////////////////////
  /**
   * Loads an alarm:ConsoleRecipient instance.
   *
   * @param {baja.Component} recipient - an instance of alarm:ConsoleRecipient
   * @param {Object} params Parameters to load the alarm console with.
   * @return {Promise} Promise resolved when support instance has loaded recipient
   */
  BajaAlarmConsoleSupport.prototype.load = function (recipient, params) {
    var that = this;
    that.$dataChannel = new BajaDataChannel(recipient);
    that.$dataChannel.on(SINGLE_SOURCE_UPDATE_EVENT, function (summary) {
      that.emit(SINGLE_SOURCE_UPDATE_EVENT, summary);
    });
    that.$dataChannel.on(RECIPIENT_UPDATE_EVENT, function (summary, update) {
      that.emit(RECIPIENT_UPDATE_EVENT, summary, update);
    });
    that.$dataChannel.on(OPTIONS_CHANGED_EVENT, function (options) {
      that.emit(OPTIONS_CHANGED_EVENT, options);
    });
    return that.$dataChannel.init(params);
  };

  /**
   * Terminates the alarm worker and destroys the data channel.
   *
   * @return {Promise} Promise resolved when support instance has been destroyed
   */
  BajaAlarmConsoleSupport.prototype.destroy = function () {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.destroy());
  };

  /////////////////////////////////////////////////////////////////////////////////
  //View Model Support
  /////////////////////////////////////////////////////////////////////////////////

  /**
   * Returns a default instance of bql:DynamicTimeRange.
   *
   * @return {baja.Component} Instance of bql:DynamicTimeRange
   */
  BajaAlarmConsoleSupport.prototype.getDefaultTimeRange = function () {
    return baja.$('bql:DynamicTimeRange');
  };

  /**
   * Returns a default instance of alarm:AlarmFilterSet.
   *
   * @return {baja.Component} Instance of alarm:AlarmFilterSet
   */
  BajaAlarmConsoleSupport.prototype.getDefaultFilterSet = function () {
    return baja.$('alarm:AlarmFilterSet');
  };

  /**
   * Loads a filter set with the filter entries for a default alarm record
   *
   * @param {baja.Component} filterSet an instance of alarm:AlarmFilterSet
   * @return {Promise} Promise resolved when filter has been loaded with filter entries
   */
  BajaAlarmConsoleSupport.prototype.loadDefaultFilterSet = function (filterSet) {
    return loadFilterSet(filterSet);
  };

  /**
   * Returns the default table model for the alarm table.
   *
   * @return {module:nmodule/webEditors/rc/wb/table/model/TableModel}
   */
  BajaAlarmConsoleSupport.prototype.getDefaultTableModel = function () {
    return new TableModel({rows: [], columns: makeColumns(this)});
  };

  /**
   * Returns the default one-based page index for the alarm table.
   *
   * @return {Number}
   */
  BajaAlarmConsoleSupport.prototype.getDefaultPageIndex = function () {
    return 1;
  };

  /**
   * Returns the default page size for the alarm table.
   *
   * @return {Number}
   */
  BajaAlarmConsoleSupport.prototype.getDefaultPageSize = function () {
    var options = this.getAlarmConsoleOptions();
    return options && options.singleSourceView && options.singleSourceView.pageSize || Number.MAX_VALUE;
  };

  /**
   * Creates a Column instance for custom alarm data.
   *
   * @param {String} name - column name
   * @return {module:nmodule/alarm/rc/console/table/columns/CustomAlarmDataColumn}
   */
  BajaAlarmConsoleSupport.prototype.createAlarmDataColumn = function (name) {
    return new CustomAlarmDataColumn(name);
  };

  /**
   * Translates a string encoding of a baja.Ord to the corresponding URI string
   *
   * @param {String} ordStr the String encoding of a baja.Ord to translate
   * @return {String} the URI corresponding to the translated baja.Ord string or null if the ord scheme
   * is not supported
   */
  BajaAlarmConsoleSupport.prototype.toUri = function (ordStr) {
    var ord        = baja.Ord.make(ordStr),
        list       = ord.parse(),
        query      = list.size() && list.get(0),
        scheme     = query && query.getScheme(),
        httpScheme = ordStr && ordStr.indexOf('http') === 0;

    return scheme instanceof baja.UnknownScheme && !httpScheme ? null : ord.toUri();
  };

  /////////////////////////////////////////////////////////////////////////////////
  //Data Channel
  /////////////////////////////////////////////////////////////////////////////////

  /**
   * Gets the default alarm console options.
   *
   * @return {Object}
   */
  BajaAlarmConsoleSupport.prototype.getDefaultAlarmConsoleOptions = function () {
    return this.$dataChannel.getDefaultAlarmConsoleOptions();
  };

  /**
   * Gets the current alarm console options.
   *
   * @return {Object}
   */
  BajaAlarmConsoleSupport.prototype.getAlarmConsoleOptions = function () {
    return this.$dataChannel && this.$dataChannel.getAlarmConsoleOptions();
  };

  /**
   * Updates the current alarm console options with a key/value pair. This
   * function updates the options on the client and also persists them to the
   * server.
   *
   * @param {String} key
   * @param {*} value
   * @param {Boolean} [singleSource] If true set the options for the single source
   * view rather than the multi source view. If this is undefined, the value will
   * be applied for all options.
   * @return {Promise}
   */
  BajaAlarmConsoleSupport.prototype.setAlarmConsoleOption = function (key, value, singleSource) {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.setAlarmConsoleOption(key, value, singleSource));
  };

  /**
   * Updates the current alarm console options with a group of key/value pairs.
   * This function updates the options on the client and also persists them to
   * the server.
   *
   * @param {Object} options
   * @param {Boolean} singleSource if true set the options for the single source
   * view rather than the multi source view
   * @return {Promise}
   */
  BajaAlarmConsoleSupport.prototype.setAlarmConsoleOptions = function (options, singleSource) {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.setAlarmConsoleOptions(options, singleSource));
  };

  /**
   * @param {Boolean} singleSource if true get the options for the single source
   * view rather than the multi source view
   * @return {Promise}
   */
  BajaAlarmConsoleSupport.prototype.resetAlarmTableSettings = function (singleSource) {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.resetAlarmTableSettings(singleSource));
  };

  /**
   * Gets an array of common user defined alarm data keys for an alarm record.
   * Note that this array contains only the public static final String fields
   * in alarm:AlarmRecord.
   *
   * @return {Promise.<Array.<String>>} Promise resolve with a String array of alarm data fields
   */
  BajaAlarmConsoleSupport.prototype.getAlarmFields = function () {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.getAlarmFields());
  };

  /**
   * Loads the alarm summary from the data channel for the specified params.
   *
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise} Promise resolved when the alarm summary has been retrieved
   */
  BajaAlarmConsoleSupport.prototype.loadAlarmSummary = function (params) {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.loadAlarmSummary(toSummaryParams(this, params)));
  };

  /**
   * Acknowledges alarms by source or uuid with optional notes.
   * @see {@link module:nmodule/alarm/rc/util/alarmUtils.ackAlarms}
   * for details regarding the expected parameter format.
   *
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise.<Array.<String>>} Promise resolved with a String array of alarm classes for which
   * alarms failed to be acknowledged
   */
  BajaAlarmConsoleSupport.prototype.ackAlarms = function (params) {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.ackAlarms(params));
  };

  /**
   * Force clears alarms by source or uuid.
   * @see {@link module:nmodule/alarm/rc/util/alarmUtils.forceClearAlarms}
   * for details regarding the expected parameter format.
   *
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise.<Array.<String>>} Promise resolved with a String array of alarm classes for which
   * alarms failed to be force cleared
   */
  BajaAlarmConsoleSupport.prototype.forceClearAlarms = function (params) {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.forceClearAlarms(params));
  };

  /**
   * Adds notes to alarms by source or uuid.
   * @see {@link module:nmodule/alarm/rc/util/alarmUtils.addNoteToAlarms}
   * for details regarding the expected parameter format.
   *
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise.<Array.<String>>} Promise resolved with a String array of alarm classes for which
   * alarms failed to be force cleared
   */
  BajaAlarmConsoleSupport.prototype.addNoteToAlarms = function (params) {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.addNoteToAlarms(params));
  };

  /**
   * Gets the alarm summary for a single source for the specified params.
   *
   * @see {@link module:baja/alarm/AlarmSpace} for details regarding the expected parameter format.
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise.<Object>} Promise resolved with an object literal containing the summary details
   */
  BajaAlarmConsoleSupport.prototype.getSingleSourceSummary = function (params) {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.getSingleSourceSummary(toSummaryParams(this, params)));
  };

  /**
   * Gets the notes for an alarm record.
   *
   * @see {@link module:baja/alarm/AlarmSpace} for details regarding the expected parameter format.
   * @param {Object} params - the object literal containing the method's arguments
   * @return {Promise.<String>} Promise resolved with a notes String
   */
  BajaAlarmConsoleSupport.prototype.getNotes = function (params) {
    return Promise.resolve(this.$dataChannel && this.$dataChannel.getNotes(params));
  };

  /**
   * Returns the underlying BajaDataChannel instance.
   *
   * @private
   * @return {module:nmodule/alarm/rc/console/baja/BajaDataChannel}
   */
  BajaAlarmConsoleSupport.prototype.$getDataChannel = function () {
    return this.$dataChannel;
  };

  return BajaAlarmConsoleSupport;
});
