/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */
define(['baja!', 'baja!tagdictionary:TagGroupInfo,baja:Relation', 'Promise', 'underscore', 'nmodule/tagdictionary/rc/taginfo/BaseTagInfoMixin', 'nmodule/tagdictionary/rc/taginfo/TagGroupInfoMixin', 'nmodule/tagdictionary/rc/util/rpcUtil', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn', 'nmodule/tagdictionary/rc/column/TagIdColumn', 'nmodule/tagdictionary/rc/column/TagNameColumn', 'nmodule/tagdictionary/rc/column/TagValueColumn', 'nmodule/tagdictionary/rc/column/TagTypeColumn', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/tree/TreeNode', 'bajaScript/baja/tag/Relation', 'log!nmodule.tagdictionary.rc.util.taggingUtil', 'hbs!nmodule/tagdictionary/rc/template/ManagerWrapper', 'hbs!nmodule/tagdictionary/rc/template/MgrHorizontalSplitPane', 'hbs!nmodule/tagdictionary/rc/template/TagMgrVerticalSplitPane', 'lex!tagdictionary,webEditors,baja'], function (baja, types, Promise, _, BaseTagInfoMixin, TagGroupInfoMixin, rpcUtil, compUtils, mixinUtils, IconMgrColumn, TagIdColumn, TagNameColumn, TagValueColumn, TagTypeColumn, Column, TreeNode, Relation, log, tplManagerWrapper, tplMgrStandardContent, tplTagMgrVerticalSplit, lexs) {
  'use strict';

  var tagdictionaryLex = lexs[0],
      webEditorsLex = lexs[1],
      bajaLex = lexs[2],
      logInfo = log.info.bind(log),
      hasMixin = mixinUtils.hasMixin,
      MGR_COMMAND = 'MGR_COMMAND',
      DICTIONARY_ICON = baja.Icon.make(['module://icons/x16/book.png', 'module://icons/x16/badges/tag.png']),
      SEARCH_ICON = baja.Icon.make(['module://icons/x16/find.png']),
      FILTER_ICON = baja.Icon.make(['module://icons/x16/filter.png']),
      TAG_ICON = baja.Icon.make(['module://tagdictionary/rc/icons/tagMarkerFlipped.png']),
      IMPLIED_TAG_ICON = baja.Icon.make(['module://tagdictionary/rc/icons/tagMarkerFlipped.png', 'module://icons/x16/badges/lock.png']),
      TAG_GROUP_ICON = baja.Icon.make(['module://tagdictionary/rc/icons/tagMarkerFlippedPair.png']),
      IMPLIED_TAG_GROUP_ICON = baja.Icon.make(['module://tagdictionary/rc/icons/tagMarkerFlippedPair.png', 'module://icons/x16/badges/lock.png']),
      TAG_FOLDER_ICON = baja.Icon.make(['module://icons/x16/folder.png', 'module://icons/x16/badges/tag.png']),
      // not using workbench:TagFilterEnum, to avoid introducing a dependency on workbench-wb,
  // also we may want a different set of filter options for the HTML ui.
  TAG_FILTER_ENUM = baja.DynamicEnum.make({
    ordinal: 0,
    range: baja.EnumRange.make({
      options: baja.Facets.make({
        lexicon: 'tagdictionary'
      }),
      tags: ['showAll', 'validOnly', 'bestOnly'],
      ordinals: [0, 1, 2]
    })
  }),
      TAG_FILTER_SHOW_ALL = TAG_FILTER_ENUM.get('showAll'),
      TAG_FILTER_VALID_ONLY = TAG_FILTER_ENUM.get('validOnly'),
      TAG_FILTER_BEST_ONLY = TAG_FILTER_ENUM.get('bestOnly'),
      NIAGARA_DICTIONARY = 'n',
      // the Niagara dictionary's namespace
  DEFAULT_TAG_TABLE_FILTER_PARAMS = {
    showDirect: true,
    showImplied: false
  },
      DEFAULT_DISCOVERY_TABLE_FILTER_PARAMS = {
    dictionary: NIAGARA_DICTIONARY,
    textFilter: '',
    showAllValidBest: TAG_FILTER_VALID_ONLY.getTag()
  }; // see BEditTagDialog.java for the workbench equivalents

  var RELATION_SLOT = 'r?',
      TAG_GROUP_RELATION_ID = new baja.Id('n:tagGroup'),
      TAG_GROUP_FACETS = baja.Facets.make({
    'tg__': true
  });
  var STANDARD_TAG_COLUMNS = [new IconMgrColumn(), new TagIdColumn({
    displayName: tagdictionaryLex.getSafe('tagManager.tagId'),
    flags: Column.flags.EDITABLE | Column.flags.READONLY
  }), new TagNameColumn({
    displayName: tagdictionaryLex.getSafe('tagManager.tagName'),
    flags: Column.flags.EDITABLE | Column.flags.READONLY
  }), new TagValueColumn({
    displayName: webEditorsLex.getSafe('value'),
    flags: Column.flags.EDITABLE
  }), new TagTypeColumn({
    displayName: tagdictionaryLex.getSafe('tagManager.tagValueType'),
    flags: Column.flags.UNSEEN
  })];
  var MARKER_TYPE_NAME = baja.Marker.DEFAULT.getType().getTypeName(),
      IMPLIED_ROW_STYLE_CLASS = 'TagManager-implied',
      MARKER_TEXT_STYLE_CLASS = 'marker-text',
      MARKER_TAG_ICON_STYLE_CLASS = 'icon-icons-x16-tag',
      CONFIG_DISPLAY_NAME = bajaLex.getSafe('nav.station');
  /**
   * A utility class for shared functions related to tagging
   *
   * API Status: **Private**
   * @exports nmodule/tagdictionary/rc/util/tagginUtil
   */

  var exports = {};
  exports.DICTIONARY_ICON = DICTIONARY_ICON;
  exports.SEARCH_ICON = SEARCH_ICON;
  exports.FILTER_ICON = FILTER_ICON;
  exports.TAG_ICON = TAG_ICON;
  exports.IMPLIED_TAG_ICON = IMPLIED_TAG_ICON;
  exports.TAG_GROUP_ICON = TAG_GROUP_ICON;
  exports.IMPLIED_TAG_GROUP_ICON = IMPLIED_TAG_GROUP_ICON;
  exports.TAG_FOLDER_ICON = TAG_FOLDER_ICON;
  exports.TAG_FILTER_ENUM = TAG_FILTER_ENUM;
  exports.TAG_FILTER_SHOW_ALL = TAG_FILTER_SHOW_ALL;
  exports.TAG_FILTER_VALID_ONLY = TAG_FILTER_VALID_ONLY;
  exports.TAG_FILTER_BEST_ONLY = TAG_FILTER_BEST_ONLY;
  exports.DEFAULT_TAG_TABLE_FILTER_PARAMS = DEFAULT_TAG_TABLE_FILTER_PARAMS;
  exports.DEFAULT_DISCOVERY_TABLE_FILTER_PARAMS = DEFAULT_DISCOVERY_TABLE_FILTER_PARAMS;
  exports.STANDARD_TAG_COLUMNS = STANDARD_TAG_COLUMNS;
  exports.MARKER_TYPE_NAME = MARKER_TYPE_NAME;
  exports.IMPLIED_ROW_STYLE_CLASS = IMPLIED_ROW_STYLE_CLASS;
  exports.MARKER_TEXT_STYLE_CLASS = MARKER_TEXT_STYLE_CLASS;
  exports.MARKER_TAG_ICON_STYLE_CLASS = MARKER_TAG_ICON_STYLE_CLASS;
  /**
   * Whether passed object is a tag group Relation.
   * To be a tag group Relation, the following should be true:
   * 1. the object is a Relation (BRelation.js or Relation.js)
   * 2. the Relation's id is 'n:tagGroup'
   * 3. the Relation is outbound
   *
   * @param {Object} obj
   * @returns {Boolean}
   */

  exports.isTagGroupRelation = function (obj) {
    if (!obj) {
      return false;
    }

    if (baja.hasType(obj, 'baja:Relation')) {
      // -> BRelation.js (bajaScript/baja/tag/BRelation)
      if (TAG_GROUP_RELATION_ID.getQName() === obj.getRelationId()) {
        // BRelation.js has #getInbound
        return !obj.getInbound();
      }
    } else if (obj instanceof Relation) {
      // -> Relation.js (bajaScript/baja/tag/Relation)
      return TAG_GROUP_RELATION_ID.equals(obj.getId()); // There is no #getInbound on Relation.js, so if we have the correct tagGroup Id, assume it's outbound anyway.
    }

    return false;
  };
  /**
   *  Whether the tag is editable on the target component.
   *
   * @param {baja.Component} target
   * @param {String} tagIdStr, a tag Id as a string.
   * @returns {Boolean}
   */


  exports.tagIsEditable = function (target, tagIdStr) {
    var slotPath = baja.SlotPath.escape(tagIdStr),
        componentFlags = target.getSlot(slotPath).getFlags(),
        tagIsReadOnly = componentFlags & baja.Flags.READONLY,
        tagIsHidden = componentFlags & baja.Flags.HIDDEN;

    if (exports.isTagGroupRelation(target.get(tagIdStr))) {
      // The slot for a tag group is always read only, so we can't really
      // disallow editing on the basis of the READONLY flag
      return !tagIsHidden;
    } // direct tags with the READONLY or HIDDEN flags cannot be edited


    return !(tagIsReadOnly || tagIsHidden);
  };
  /**
   *  Whether the tag on the target component should be viewable in the manager.
   *
   * @param {baja.Component} target
   * @param {String} tagIdStr, a tag Id as a string.
   * @returns {Boolean}
   */


  exports.tagIsViewable = function (target, tagIdStr) {
    var componentFlags = target.getSlot(baja.SlotPath.escape(tagIdStr)).getFlags(),
        // direct tags with the HIDDEN flag should not be shown in the Tag Manager
    tagIsHidden = componentFlags & baja.Flags.HIDDEN;
    return !tagIsHidden;
  };
  /**
   * add the tag represented by tagInfo to the target component.
   *
   * @param {baja.Component} target
   * @param {baja.Component} tagToAdd, an instance of baja.Component
   *                             (usually tagdictionary:TagInfo or tagdictionary:TagGroupInfo)
   *                             with BaseTagInfoMixin mixed in.
   * @param {baja.comm.Batch} batch
   * @returns {Promise}
   */


  exports.addTagToTarget = function (target, tagToAdd, batch) {
    var tagValue = tagToAdd.get('tagValue'); // tagValue property is added in TagUxManager#newInstanceFromDiscoverySubject
    // and its associated column sets/edits the value

    if (tagToAdd.$isTagGroup || tagToAdd.getType().is('tagdictionary:TagGroupInfo')) {
      // This is a BRelation.js object, not a Relation.js object,
      // see tag.js where baja.Relation is set to Relation.js, and then BRelation.js
      // TODO BRelation.js could be changed to provide the a constructor, ie so the line below works:
      // const relation =  new baja.Relation(TAG_GROUP_RELATION_ID, tagValue);
      var relation = baja.$('baja:Relation', {
        relationId: TAG_GROUP_RELATION_ID.getQName(),
        sourceOrd: tagValue
      }); // for tagGroups, check whether a relation with the same sourceOrd exists on the target

      var groupExists = false;
      target.getSlots().properties().is("baja:Relation").toValueArray().forEach(function (relation) {
        if (tagValue.equals(relation.getSourceOrd())) {
          groupExists = true;
          logInfo(tagdictionaryLex.get({
            key: 'tag.ignore.duplicate.group',
            args: [tagToAdd.getTagId().getQName(), target.getDisplayName()]
          }));
        }
      });
      return groupExists ? Promise.resolve() : target.add({
        slot: RELATION_SLOT,
        value: relation,
        flags: baja.Flags.READONLY,
        facets: TAG_GROUP_FACETS,
        batch: batch
      });
    }

    var tagName = tagToAdd.getTagId().getQName(),
        tagSlotName = baja.SlotPath.escape(tagName);

    if (target.has(tagSlotName)) {
      if (tagToAdd.isAdHocTag()) {
        logInfo(tagdictionaryLex.get({
          key: 'tag.adhoc.no.overwrite',
          args: [tagName, target.getDisplayName()]
        }));
        return Promise.resolve();
      }

      if (!exports.tagIsEditable(target, tagName)) {
        logInfo(tagdictionaryLex.get({
          key: 'tag.not.editable',
          args: [tagName, target.getDisplayName()]
        }));
        return Promise.resolve();
      } // for editable tags, allow a duplicate to be overwritten, there may be an updated value.


      return target.set({
        slot: tagSlotName,
        value: tagValue,
        batch: batch
      });
    } else {
      return target.add({
        slot: tagSlotName,
        value: tagValue,
        flags: baja.Flags.METADATA,
        batch: batch
      });
    }
  };
  /**
   * Constructs an Id for the tag group.
   *
   * @param {baja.Component} tagGroupInfoComp a 'tagdictionary:TagGroupInfo' object
   * @returns {baja.Id}
   */


  exports.constructTagGroupId = function (tagGroupInfoComp) {
    var tagGroupName = baja.SlotPath.unescape(tagGroupInfoComp.getName());

    if (tagGroupName.indexOf(':') > 0) {
      // fully qualified tag group name
      return new baja.Id(tagGroupName);
    } // TODO this is looking for the parent TagDictionary, but will only work if the parent
    //      is an instance of BTagDictionary, and not if the tag dictionary is only
    //      a BComponent that implements the TagDictionary interface.


    var dictionary = compUtils.closest(tagGroupInfoComp, 'tagdictionary:TagDictionary'),
        namespace = dictionary ? dictionary.getNamespace() + ':' : '';
    return new baja.Id(namespace + tagGroupName);
  };
  /**
   * The 'MGR_COMMAND' mixin has the option to not show commands in
   * the bar at the bottom of the view. This function will filter out
   * the ones that aren't required.
   *
   * @see module:nmodule/webEditors/rc/wb/mgr/Manager#doInitialize
   *
   * @param {module:bajaux/commands/Command} command The command to add.
   * @returns {Boolean} true, if required in the action bar
   */


  exports.filterCommands = function (command) {
    return hasMixin(command, MGR_COMMAND) ? command.isShownInActionBar() : true;
  };

  var standardMgrContent = function standardMgrContent() {
    return tplMgrStandardContent({
      databaseTitle: tagdictionaryLex.get('tagManager.showing'),
      discoveryTitle: tagdictionaryLex.get('tagManager.availableTags')
    });
  };
  /**
   * returns the html for the Tagging Dialog
   *
   * @returns {String}
   */


  exports.htmlForTaggingDialog = function () {
    // This returns the same HTML as is generated by MgrLearnTableSupport#mgrLearnHtml
    // except that it has different titles for the panes.
    // It is split up into 2 templates to allow this manager to add in a second split pane;
    return tplManagerWrapper({
      managerContent: standardMgrContent()
    });
  };
  /**
   * returns the html for the Tagging View
   *
   * @returns {String}
   */


  exports.htmlForTaggingView = function () {
    var verticalSplit = tplTagMgrVerticalSplit({
      selectedComponentsTitle: tagdictionaryLex.getSafe('tagManager.titlePane.selectedComponents'),
      rightPaneContent: standardMgrContent()
    });
    return tplManagerWrapper({
      managerContent: verticalSplit
    });
  };
  /**
   * returns the TagDictionaryService
   *
   * @returns {Promise.<baja.Component|undefined>} Promise that resolves to the tag dictionary service,
   * or undefined if not found.
   */


  exports.getTagDictionaryService = function () {
    return rpcUtil.getTagDictionaryServiceOrd().then(function (ordStr) {
      return ordStr ? baja.Ord.make(ordStr).get() : undefined;
    });
  };
  /*
   * Make a tree node for a tag.
   *
   * @param  {baja.Tag|baja.Component} tag,
   *             a baja.Tag when populating the database table from #tags() on the component, or
   *             a baja.Component with <module:nmodule/tagdictionary/rc/taginfo/BaseTagInfoMixin>
   *             mixed in when called during the process of adding a tag to the discovery table.
   * @param  {Boolean} implied
   * @returns {module:nmodule/webEditors/rc/wb/tree/TreeNode}
   */


  exports.makeTagTableNode = function (tag, implied) {
    var treeNodeName; // For a component with BaseTagInfoMixin mixed-in, or an implementation of BaseTagInfoMixin.

    if (tag.$tagInfoMixedIn || tag instanceof BaseTagInfoMixin) {
      treeNodeName = tag.getTagId().getQName();
    } else {
      // For Tag.js, Relation.js, etc
      treeNodeName = tag.getId().getQName();
    }

    var node = new TreeNode(treeNodeName, treeNodeName, []);
    node.value = _.constant(tag);
    node.getIcon = implied ? _.constant(IMPLIED_TAG_ICON) : _.constant(TAG_ICON);
    node.mayHaveKids = _.constant(false);
    node.isGroup = _.constant(false); // ie not a folder for tags/tagGroups

    node.isImplied = _.constant(!!implied);
    node.isTagGroup = _.constant(false);
    return node;
  };
  /*
   * Make a tree node for an item in the tag discovery table.
   *
   * @param  {baja.Component} tag,
   *             a baja.Component with <module:nmodule/tagdictionary/rc/taginfo/BaseTagInfoMixin>
   *             mixed in when called during the process of adding a tag to the discovery table.
   * @returns {module:nmodule/webEditors/rc/wb/tree/TreeNode}
   */


  exports.makeTagGroupLearnTableNode = function (tagGroup) {
    var node = exports.makeTagTableNode(tagGroup); // for the purposes of the tag learn table, a tag group node is exactly the same as a tag node
    // apart from the icon, and the tags in the discovery table aren't implied

    node.getIcon = _.constant(TAG_GROUP_ICON);
    node.isTagGroup = _.constant(true);
    return node;
  };
  /*
   * Make a tree node for an item in the tag table.
   *
   * @param  {baja.Id} tagGroupId
   * @param  {baja.Relation} relation
   * @param  {Boolean} implied
   * @returns {module:nmodule/webEditors/rc/wb/tree/TreeNode}
   */


  exports.makeTagGroupTableNode = function (tagGroupId, relation, implied) {
    var node = new TreeNode(tagGroupId.getQName(), tagGroupId.getQName(), []); // add this function to the relation so we don't have to keep looking up the tag group name

    relation.$getTagGroupId = _.constant(tagGroupId);
    node.value = _.constant(relation);
    node.getIcon = implied ? _.constant(IMPLIED_TAG_GROUP_ICON) : _.constant(TAG_GROUP_ICON);
    node.mayHaveKids = _.constant(false);
    node.isGroup = _.constant(false); // ie not a folder for tags/tagGroups

    node.isImplied = _.constant(!!implied);
    node.isTagGroup = _.constant(true);
    return node;
  };
  /*
   * Make a cell content for a baja.Marker.
   *
   * @returns {String} column html for a baja.Marker (with icon & name)
   */


  exports.makeMarkerCellContent = function () {
    return '<span class="' + MARKER_TAG_ICON_STYLE_CLASS + '"/><span class="' + MARKER_TEXT_STYLE_CLASS + '">' + MARKER_TYPE_NAME + '</span>';
  };
  /*
   * Work out a name to use when a component doesn't have one.
   * (intended for use where #getName & #getDisplayName return null)
   *
   * @param  {baja.Component} column html for a baja.Marker (with icon & name)
   * @returns {String} a name, or an empty String if can't be done.
   */


  exports.workOutName = function (component) {
    if (component.getType().is('baja:Station')) {
      return CONFIG_DISPLAY_NAME;
    }

    return '';
  };
  /**
   * Tests the validity of the supplied tag id.
   *
   * A valid tag ID consists of a namespace and tag name separated by a colon (':').
   * The namespace and tag name must start with an alpha (a-z, A-Z) character
   * and contain only alphanumeric characters (a-z, A-Z, 0-9) or an underscore ('_').
   * They may not contain whitespace.
   * (cf. workbench#add.tag.error.invalid.id)
   *
   * @param {String} tagIdStr
   * @returns {Boolean}
   */


  exports.isValidTagId = function (tagIdStr) {
    // (loosely) based upon TagUtil.java#isValidTagId
    function valid(str) {
      return str.length > 0 && // must not be blank
      baja.SlotPath.isValidName(str) && // must be a valid slot name
      str.indexOf('$') < 0; // no special characters
    } // (all the other should have been filtered out by SlotPath#isValidName


    var split = tagIdStr.split(':');

    if (split.length !== 2) {
      return false;
    }

    var namespace = split[0],
        tagName = split[1];
    return valid(namespace) && valid(tagName);
  };

  return exports;
});
