/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/NameMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/MgrColumn', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/table/tree/TreeTableModel', 'nmodule/webEditors/rc/wb/table/tree/TreeNodeRow', 'nmodule/webEditors/rc/wb/tree/TreeNode', 'nmodule/tagdictionary/rc/util/taggingUtil', 'lex!tagdictionary'], function (baja, Promise, _, IconMgrColumn, NameMgrColumn, PropertyMgrColumn, MgrColumn, Column, TreeTableModel, TreeNodeRow, TreeNode, taggingUtil, lexs) {
  'use strict';

  var tagdictionaryLex = lexs[0]; // A standard NameMgrColumn with #getValueFor overridden to cope with the Config node

  var COMPONENT_NAME_COL = new NameMgrColumn(),
      superTagNameGetValueFor = COMPONENT_NAME_COL.getValueFor;

  COMPONENT_NAME_COL.getValueFor = function (row) {
    return superTagNameGetValueFor(row) || taggingUtil.workOutName(row.getSubject());
  };

  var LOCATION_COL = new PropertyMgrColumn('location', {
    displayName: tagdictionaryLex.getSafe('tagManager.locationCol.displayName')
  });

  LOCATION_COL.getValueFor = function (row) {
    return row.getSubject().getSlotPath();
  };
  /*
   *  Hidden column for 'tagValue', used by the BatchComponentEditor,
   *  but not shown in the SelectedComponents Table, largely because
   *  a component does not have a tagValue property.
   */


  var HIDDEN_TAG_VALUE_COL = new PropertyMgrColumn('tagValue', {
    flags: Column.flags.EDITABLE | Column.flags.UNSEEN
  });
  /*
   * baja.Component does not have a 'tagValue', so provide a default.
   */

  HIDDEN_TAG_VALUE_COL.$getDefaultValue = _.constant('');
  /*
   * Prevent the column from appearing in the column selection dropdown
   */

  HIDDEN_TAG_VALUE_COL.isHidable = _.constant(false);
  var COLUMNS = [new IconMgrColumn(), COMPONENT_NAME_COL, LOCATION_COL, HIDDEN_TAG_VALUE_COL];
  /**
   * API Status: **Private**
   * @exports nmodule/tagdictionary/rc/model/SelectedComponentsModel
   */

  var exports = {};
  /**
   * Create the tag manager model
   *
   * @param {module:nmodule/tagdictionary/rc/TagUxManager} manager
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */

  exports.make = function (manager) {
    var mgr = manager;
    return TreeTableModel.make({
      columns: COLUMNS
    }).then(function (model) {
      /**
       * Create a new `Row` instance with a `TreeNode` as the subject.
       * (Used when dragging from Available Tags onto the Selected Components)
       *
       * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} subject
       * @returns {module:nmodule/webEditors/rc/wb/table/tree/TreeNodeRow}
       */
      model.makeRow = function (subject) {
        var implied = false; // not implied, given that we are actually adding it.

        return new TreeNodeRow(subject.$isTagGroup ? taggingUtil.makeTagGroupTableNode(subject.getTagId(), subject, implied) : taggingUtil.makeTagTableNode(subject, implied));
      };

      model.$superInsertRows = _.bind(model.insertRows, model);
      /**
       * Add new rows to the Selected Components model, ignoring duplicates.
       *
       * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row|*>} toInsert
       * @param {Number} [index] index to insert the rows; will append to the
       * end if omitted
       * @returns {Promise} promise to be resolved if the insert is successful
       */

      model.insertRows = function (toInsert, index) {
        var withoutDuplicates = toInsert;
        var modelRows = this.getRows(); // if there are already rows in the model, check for duplicates

        if (modelRows.length > 0) {
          withoutDuplicates = _.filter(toInsert, function (insertRow) {
            var modelRowComp;
            var insertComp = insertRow.getSubject();

            for (var i = 0; i < modelRows.length; i++) {
              modelRowComp = modelRows[i].getSubject();

              if (insertComp.equals(modelRowComp)) {
                return false;
              }
            }

            return true;
          });
        }

        return withoutDuplicates.length ? model.$superInsertRows(withoutDuplicates, index) : Promise.resolve();
      };
      /**
       * Create a new row using the component as the subject.
       * @param {baja.Component} subject
       * @returns {module:nmodule/webEditors/rc/wb/table/tree/TreeNodeRow}
       */


      model.makeComponentRow = function (subject) {
        var name = subject.getName(),
            displayName = subject.getDisplayName();

        if (!name) {
          name = taggingUtil.workOutName(subject);
        }

        if (!displayName) {
          displayName = taggingUtil.workOutName(subject);
        }

        var node = new TreeNode(name, displayName, []);
        node.value = _.constant(subject);
        node.getIcon = _.constant(subject.getIcon());
        node.mayHaveKids = _.constant(false);
        return new TreeNodeRow(node);
      };
      /**
       * Override the model's addInstances function.
       * Adss the tags to all the selected components in the model
       *
       * @param {Array.<baja.Component>} tagsToAdd, an array of baja.Component objects that have BaseTagInfoMixin mixed in.
       * @param {Array.<String>} [names] not used here.
       * @returns {Promise} promise to be resolved when the tags are added.
       */


      model.addInstances = function (tagsToAdd, names) {
        var promises = [],
            batch = new baja.comm.Batch(),
            selectedRows = mgr.getSelectedComponentsTable().getSelectedRows();
        var selectedComponent;
        selectedRows.forEach(function (selectedRow) {
          selectedComponent = selectedRow.getSubject();
          tagsToAdd.forEach(function (tagInstance) {
            promises.push(taggingUtil.addTagToTarget(selectedComponent, tagInstance, batch));
          });
        });
        return batch.commit(Promise.all(promises));
      };
      /**
       * Always returns null, indicating not a duplicate.
       * This model assumes that #addInstances, (ultimately taggingUtil#addTagToTarget)
       * will deal with any duplicates appropriately.
       *
       * @param {Array.<baja.Component>} newTags, an array of baja.Component objects that have BaseTagInfoMixin mixed in.
       * @returns {String|null} the first duplicate name, or null if there were no duplicates found.
       */


      model.findDuplicateTagName = function (newTags) {
        return null;
      };

      return model;
    });
  };

  return exports;
});
