/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/wb/table/tree/TreeTableModel', 'nmodule/webEditors/rc/wb/table/tree/TreeNodeRow', 'nmodule/tagdictionary/rc/util/taggingUtil', 'log!nmodule.tagdictionary.rc.model.ComponentTagModel'], function (baja, Promise, _, TreeTableModel, TreeNodeRow, taggingUtil, log) {
  'use strict';

  var IMPLIED_ROW_STYLE_CLASS = taggingUtil.IMPLIED_ROW_STYLE_CLASS,
      logFine = log.fine.bind(log);
  /**
   * API Status: **Private**
   * @exports nmodule/tagdictionary/rc/model/ComponentTagModel
   */

  var exports = {};
  /**
   * Create the tag manager model
   *
   * @param {module:nmodule/tagdictionary/rc/TagUxManager} mgr
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */

  exports.make = function (mgr) {
    return TreeTableModel.make({
      columns: taggingUtil.STANDARD_TAG_COLUMNS
    }).then(function (model) {
      /**
       * Create a new `Row` instance with a `TreeNode` as the subject.
       * @param {module:nmodule/webEditors/rc/wb/tree/TreeNode} subject
       * @returns {module:nmodule/webEditors/rc/wb/table/tree/TreeNodeRow}
       */
      model.makeRow = function (subject) {
        var implied = false; // not implied, given that we are actually adding it.

        return new TreeNodeRow(subject.$isTagGroup ? taggingUtil.makeTagGroupTableNode(subject.getTagId(), subject, implied) : taggingUtil.makeTagTableNode(subject, implied));
      };
      /**
       * Override the model's addInstances function.
       *
       * @param {Array.<baja.Component>} instances the tags to add
       * @param {Array.<String>} [names] the desired slot names for the instances.
       * If omitted, default names derived from the instance Types will be used.
       * @returns {Promise} promise to be resolved when the instances are added.
       */


      model.addInstances = function (instances, names) {
        var promises = [],
            batch = new baja.comm.Batch();
        instances.forEach(function (instance) {
          promises.push( // update the component whose tags are shown in the database table
          taggingUtil.addTagToTarget(mgr.value(), instance, batch));
        });
        return batch.commit();
      };
      /**
       * Search through the array of tags to be added, looking for any duplicates
       * between the tags names on the component in this model and the existing ones.
       * If we find a duplicate, return its name to show in an error message.
       *
       * @param {Array.<baja.Component>} newTags, an array of baja.Component objects that has Mixin mixed in.
       * @returns {String|null} the first duplicate name, or null if there were no duplicates found.
       */


      model.findDuplicateTagName = function (newTags) {
        var existingTags = getExistingInstances(this.getRows());

        var existingNames = getTagIds(existingTags),
            newNames = getNewTagIds(newTags),
            duplicates = _.intersection(existingNames, newNames);

        return duplicates.length ? _.first(duplicates) : null;
      };

      return model;
    });
  };
  /**
   * Populate the manager's main table from the current tags and tag groups on the managed component
   *
   * @param {nmodule/tagdictionary/rc/TagUxManager} mgr
   * @returns {Promise}
   */


  exports.populateMainTable = function (mgr) {
    var component = mgr.value(),
        model = mgr.getModel();
    return model.clearRows().then(function () {
      // clear down the filter-out list too.
      mgr.$filteredOutTags = [];
      return component ? exports.$populateWithTagsAndTagGroups(mgr) : Promise.resolve();
    });
  };
  /**
   * Populate the manager's main table from the current tags and tag groups on the managed component
   *
   * @param {nmodule/tagdictionary/rc/TagUxManager} mgr
   * @returns {Promise}
   */


  exports.$populateWithTagsAndTagGroups = function (mgr) {
    var component = mgr.value(),
        model = mgr.getModel(),
        filterParams = mgr.$tagTablefilterParams || {},
        showDirect = _.has(filterParams, 'showDirect') ? filterParams.showDirect : true,
        showImplied = _.has(filterParams, 'showImplied') ? filterParams.showImplied : false,
        nodes = [];
    return component.relations().then(function (relations) {
      var directRelations = [],
          impliedRelations = [];

      if (relations instanceof baja.SmartRelations) {
        directRelations = relations.getDirectRelations().getAll();
        impliedRelations = relations.getImpliedRelations().getAll();
      } else if (relations instanceof baja.ComponentRelations) {
        // no implied relations, only direct
        directRelations = relations.getAll();
      }

      function processRelations(relations, implied) {
        var proms = [];
        relations.forEach(function (relation) {
          // directRelations is a ComponentRelations object and #getAll returns Array<module:baja/tag/BRelation>
          // impliedRelations is a RelationSet and #getAll returns Array<module:baja/tag/Relation>
          if (relation.getId().getQName() === 'n:tagGroup') {
            var prom = relation.getEndpointOrd().get({
              base: baja.station
            }).then(function (tagGroupInfo) {
              var row = new TreeNodeRow(taggingUtil.makeTagGroupTableNode(taggingUtil.constructTagGroupId(tagGroupInfo), relation, implied));

              if (implied && !showImplied || !implied && !showDirect) {
                mgr.$filteredOutTags.push(row);
              } else {
                nodes.push(row);
              }
            })["catch"](function (err) {
              // #getEndpointOrd()#get() can throw a BoxError/PermissionException if the user
              // does not have read permissions on the TagGroupInfo endpoint under the TagDictionary.
              // Currently the BoxError gets logged anyway, and here this isn't necessarily a problem,
              // therefore fine logging seems appropriate.
              logFine(err);
            });
            proms.push(prom);
          }
        });
        return proms;
      }

      return Promise.all(processRelations(directRelations, false).concat(processRelations(impliedRelations, true)));
    }).then(function () {
      return component.tags(); // retrieve direct and implied tags
    }).then(function (tags) {
      var directTags = [],
          impliedTags = [];

      if (tags instanceof baja.SmartTags) {
        directTags = tags.getDirectTags().getAll();
        impliedTags = tags.getImpliedTags().getAll();
      } else if (tags instanceof baja.ComponentTags) {
        // no implied tags, only direct
        directTags = tags.getAll();
      }

      directTags.forEach(function (tag) {
        if (!taggingUtil.tagIsViewable(component, tag.getId().getQName())) {
          return;
        } // when 'showDirect' is true, add the direct tags to the database table (nodes).
        // when 'showDirect' is false, store them on the manager (mgr.$filteredOutTags),
        //             where they are used when calculating #isExisting.


        addTagNode(showDirect ? nodes : mgr.$filteredOutTags, tag, false);
      });
      impliedTags.forEach(function (tag) {
        // when 'showImplied' is true, add the implied tags to the database table (nodes).
        // when 'showImplied' is false, store them on the manager (mgr.$filteredOutTags),
        //             where they are used when calculating #isExisting.
        addTagNode(showImplied ? nodes : mgr.$filteredOutTags, tag, true);
      });
    }).then(function () {
      return model.insertRows(nodes, 0);
    });
  };
  /**
   * Update manager row styling - add style class for 'implied'
   *                            - add icon to value column for Markers
   * @private
   * @param row
   * @param dom
   * @returns {JQuery} dom element with updated styling
   */


  exports.finishTableRow = function (row, dom) {
    var node = row.getTreeNode(),
        subject = row.getSubject();
    dom.toggleClass(IMPLIED_ROW_STYLE_CLASS, node.isImplied()); // The marker icon is added for baja.Markers or implied tag groups

    if (subject.getValue && subject.getValue().getType().is('baja:Marker') || node.isTagGroup() && node.isImplied()) {
      dom.find('.js-col-tagValue').html(taggingUtil.makeMarkerCellContent());
    }

    return dom;
  };

  function getExistingInstances(dbModelRows) {
    // it's ok to add a tag that duplicates an implied one, so exclude implied tags here.
    var withoutImplied = _.filter(dbModelRows, function (row) {
      return !row.getTreeNode().isImplied();
    });

    return _.invoke(withoutImplied, 'getSubject');
  }

  function getTagIds(tags) {
    var names = [];
    tags.forEach(function (tag) {
      if (tag instanceof baja.Tag) {
        names.push(tag.getId().getQName());
      } else if (tag instanceof baja.Relation) {
        var tagGroupId = _.result(tag, '$getTagGroupId');

        if (tagGroupId) {
          names.push(tagGroupId.getQName());
        }
      }
    });
    return names;
  }

  function getNewTagIds(newTagComps) {
    var names = [];
    newTagComps.forEach(function (newTagComp) {
      names.push(newTagComp.getTagId().getQName());
    });
    return names;
  }
  /*
   * Add a TreeNodeRow for the tag to the nodes array if it doesn't already exisit
   *
   * @param  {Array.<module:nmodule/nmodule/webEditors/rc/wb/table/tree/TreeNodeRow>} nodes
   * @param  {module:baja/tag/Tag} tag
   * @param  {Boolean} implied
   */


  var addTagNode = function addTagNode(nodes, tag, implied) {
    var exists = _.find(nodes, function (node) {
      return node.getTreeNode().getName() === tag.getId().getQName();
    });

    if (exists) {
      return;
    }

    nodes.push(new TreeNodeRow(taggingUtil.makeTagTableNode(tag, implied)));
  };

  return exports;
});
