/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/tagdictionary/rc/commands/EditTagCommand
 */
define(['baja!', 'underscore', 'jquery', 'lex!tagdictionary,webEditors', 'log!nmodule.tagdictionary.rc.commands.EditTagCommand', 'nmodule/webEditors/rc/wb/mgr/model/columns/IconMgrColumn', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn', 'nmodule/webEditors/rc/wb/mgr/model/MgrColumn', 'bajaux/commands/Command', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/wb/mgr/BatchComponentEditor', 'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor', 'nmodule/webEditors/rc/wb/mgr/model/MgrModel', 'nmodule/webEditors/rc/wb/table/model/source/ArrayComponentSource', 'nmodule/tagdictionary/rc/util/taggingUtil'], function (baja, _, $, lexs, log, IconMgrColumn, Column, PropertyMgrColumn, MgrColumn, Command, feDialogs, fe, BatchComponentEditor, DisplayOnlyEditor, MgrModel, ArrayComponentSource, taggingUtil) {
  'use strict';

  var tagdictionaryLex = lexs[0],
      webEditorsLex = lexs[1],
      logInfo = log.info.bind(log);

  var TAG_ID_COLUMN = function TAG_ID_COLUMN(params) {
    MgrColumn.call(this, 'tagId', _.extend({
      displayName: tagdictionaryLex.getSafe('tagManager.tagId')
    }, params));
  };

  TAG_ID_COLUMN.prototype = Object.create(MgrColumn.prototype);
  TAG_ID_COLUMN.prototype.constructor = TAG_ID_COLUMN;

  TAG_ID_COLUMN.prototype.getValueFor = function (row) {
    return row.getSubject().get('tagId');
  };
  /**
   * A column for managing a tag's value in the BatchComponentEditor
   *
   * @extends module:nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn
   * @param {Object} [params]
   * @param {baja.Component} [params.managedComponent] the component whose tags are being managed
   */


  var TAG_VALUE_COLUMN = function TAG_VALUE_COLUMN(params) {
    PropertyMgrColumn.call(this, 'tagValue', _.extend({
      displayName: webEditorsLex.getSafe('value')
    }, params));
    this.$manager = params && params.manager;
  };

  TAG_VALUE_COLUMN.prototype = Object.create(PropertyMgrColumn.prototype);
  TAG_VALUE_COLUMN.prototype.constructor = TAG_VALUE_COLUMN;

  TAG_VALUE_COLUMN.prototype.getValueFor = function (row) {
    return row.getSubject().get('tagValue');
  };
  /**
   * Sets the new tag value.
   *
   * @param {baja.Value} value
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise} promise to be resolved when the value has
   * been set on the backing `Component`
   */


  TAG_VALUE_COLUMN.prototype.commit = function (value, row, params) {
    var manager = this.$manager,
        tagIdStr = row.getSubject().get('tagId'),
        tagSlotName = baja.SlotPath.escape(tagIdStr),
        progressCallback = params && params.progressCallback;
    var targets = [];

    if (progressCallback) {
      progressCallback(MgrColumn.COMMIT_READY);
    }

    function getSelectedComponentsSelectedSubjects(mgr) {
      return _.invoke(mgr.getSelectedComponentsTable().getSelectedRows(), 'getSubject');
    }

    if (manager.hasMixIn('SELECTED_COMPONENTS_TABLE')) {
      targets = getSelectedComponentsSelectedSubjects(manager);
    } else if (manager.value()) {
      targets = [manager.value()];
    }

    return Promise.all(_.map(targets, function (target) {
      if (target.getPermissions().hasAdminWrite() && target.has(tagSlotName)) {
        if (taggingUtil.tagIsEditable(target, tagIdStr)) {
          return target.set({
            slot: tagSlotName,
            value: value,
            batch: params.batch
          });
        } else {
          logInfo('Not making changes to uneditable tag: ' + tagIdStr + ' on component: ' + target.getDisplayName());
        }
      } // At this point the tag itself is not editable,
      // or doesn't exist on this particular target,
      // or the target itself is not editable.


      return Promise.resolve();
    }));
  };
  /**
   * Set the editor to readonly for an implied Tag or a TagGroup
   *
   * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row>} rows the selected rows
   * @returns {Object}
   */


  TAG_VALUE_COLUMN.prototype.getConfigFor = function (rows) {
    var config = MgrColumn.prototype.getConfigFor.apply(this, arguments); // at this point, expecting that the value will not be editable for multiple rows,
    // so not dealing with that here.

    var subject = rows[0].getSubject(),
        target = this.$manager.value(),
        tagIdStr = subject.get('tagId'); // make the editor readonly for an implied Tag, a TagGroup or
    // a non-editable direct tag (eg with the READONLY flag)

    if (subject.isImplied() || subject.isTagGroup() || !taggingUtil.tagIsEditable(target, tagIdStr)) {
      config = _.extend({
        readonly: true
      }, config);
    } // specify DisplayOnlyEditor for a Marker tag


    if (subject.isMarker()) {
      config = _.extend({
        type: DisplayOnlyEditor
      }, config);
    }

    return config;
  };
  /**
   * We can't set multiple tags to the same tag value. So if editing more than
   * one row, we need the editor to be a DisplayOnlyEditor.
   * For one row, a DisplayOnlyEditor might be suitable (eg. for baja.Marker),
   * but it might not, so we return false and force the editor to re-found.
   *
   * @param {module:nmodule/webEditors/rc/fe/baja/BaseEditor} editor
   * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row>} rows
   * @returns {boolean} true when: there's more than one row and it's a DisplayOnlyEditor
   */


  TAG_VALUE_COLUMN.prototype.isEditorSuitable = function (editor, rows) {
    if (!editor || !editor.jq() || rows.length === 1) {
      return false;
    } // more than one row at this point ...


    return editor instanceof DisplayOnlyEditor;
  };

  var TAG_ICON_COLUMN = function TAG_ICON_COLUMN(name, params) {
    IconMgrColumn.apply(this, [name, params]);
  };

  TAG_ICON_COLUMN.prototype = Object.create(IconMgrColumn.prototype);
  TAG_ICON_COLUMN.prototype.constructor = TAG_ICON_COLUMN;

  TAG_ICON_COLUMN.prototype.getValueFor = function (row) {
    return row.getSubject().getIcon();
  };

  function makeSubMgrModel(manager) {
    var selectedRows = manager.getMainTable().getSelectedRows(),
        comps = _.map(selectedRows, function (row) {
      function getTagId(rw) {
        return rw.getTreeNode().isTagGroup() ? rw.getSubject().$getTagGroupId().getQName() : rw.getSubject().getId().getQName();
      }

      function getTagValue(rw) {
        if (rw.getTreeNode().isTagGroup()) {
          return rw.getSubject().getSourceOrdDisplay();
        }

        if (rw.getSubject().getValue().getType().is('baja:Marker')) {
          return rw.getSubject().getValue().getType().getTypeName();
        }

        return rw.getSubject().getValue();
      }

      function isMarker(rw) {
        return rw.getSubject().getValue && rw.getSubject().getValue().getType().is('baja:Marker');
      } // selectedRows is a list of Tag.js objects, so we need to create
      // baja.Component representations for the the BatchComponentEditor


      var tagComp = baja.$('baja:Component', {
        tagName: row.getSubject().getId().getName(),
        tagId: getTagId(row),
        tagValue: getTagValue(row)
      });
      tagComp.isImplied = _.constant(row.getTreeNode().isImplied());
      tagComp.isTagGroup = _.constant(row.getTreeNode().isTagGroup());
      tagComp.isMarker = _.constant(isMarker(row));
      tagComp.getIcon = _.constant(row.getTreeNode().getIcon());
      return tagComp;
    });

    return new MgrModel({
      componentSource: new ArrayComponentSource(comps),
      columns: EditTagCommand.makeColumns(manager)
    });
  }
  /**
   * Command for editing existing components in a Manager view.
   *
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/webEditors/rc/wb/mgr/commands/EditTagCommand
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   */


  var EditTagCommand = function EditTagCommand(manager) {
    Command.call(this, {
      module: 'tagdictionary',
      lex: 'commands.tagManager.editTag',
      enabled: false,

      /**
       * Takes the components that are currently selected in the manager's
       * main table, and allows for editing them using a `BatchComponentEditor`.
       * @alias module:nmodule/tagdictionary/rc/commands/EditTagCommand#invoke
       * @returns {Promise}
       */
      func: function func() {
        return feDialogs.showFor({
          value: makeSubMgrModel(manager),
          type: BatchComponentEditor
        });
      }
    });
  };

  EditTagCommand.prototype = Object.create(Command.prototype);
  EditTagCommand.prototype.constructor = EditTagCommand;

  EditTagCommand.makeColumns = function (manager) {
    return [new TAG_ICON_COLUMN(), new TAG_ID_COLUMN({
      flags: Column.flags.EDITABLE | Column.flags.READONLY
    }), new TAG_VALUE_COLUMN({
      flags: Column.flags.EDITABLE,
      manager: manager
    })];
  };

  return EditTagCommand;
});
