/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 * @author Andy Sutton
 */

/**
 * API Status: **Private**
 * @module nmodule/tagdictionary/rc/commands/DeleteTagCommand
 */
define(['baja!', 'Promise', 'underscore', 'jquery', 'dialogs', 'bajaux/commands/Command', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/BooleanEditor', 'nmodule/tagdictionary/rc/util/taggingUtil', 'lex!tagdictionary,webEditors', 'log!nmodule.tagdictionary.rc.commands.DeleteTagCommand', 'css!nmodule/tagdictionary/rc/tagdictionary'], function (baja, Promise, _, $, dialogs, Command, feDialogs, fe, BooleanEditor, taggingUtil, lexs, log) {
  'use strict';

  var tagdictionaryLex = lexs[0],
      webEditorsLex = lexs[1],
      logInfo = log.info.bind(log);
  /**
   * Command for editing existing components in a Manager view.
   *
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/tagdictionary/rc/commands/DeleteTagCommand
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} manager
   */

  var DeleteTagCommand = function DeleteTagCommand(manager) {
    var that = this;
    Command.call(that, {
      module: 'tagdictionary',
      lex: 'commands.tagManager.deleteTag',
      enabled: false,

      /**
       * Takes the tags that are currently selected in the manager's
       * main table, and removes them from the managed component.
       * @returns {Promise}
       */
      func: function func() {
        if (selectionHasImpliedTags(manager)) {
          return dialogs.showOk({
            title: tagdictionaryLex.get('cannot.delete.tags'),
            content: tagdictionaryLex.getSafe('cannot.delete.tags.implied')
          });
        }

        var deleteFromAllSelected;
        return dialogs.showOkCancel({
          title: webEditorsLex.get('commands.delete.confirm.title'),
          content: function content(dlg, jq) {
            var confirmDeleteText = tagdictionaryLex.getSafe('commands.tagManager.deleteTag.confirm.content'),
                booleanEditorLabelText = tagdictionaryLex.getSafe('commands.tagManager.deleteTag.checkbox.label');

            if (!manager.hasMixIn('SELECTED_COMPONENTS_TABLE')) {
              jq.html(confirmDeleteText);
              return;
            }

            jq.empty();
            jq.append('<div>' + confirmDeleteText + '</div><br/>');
            return fe.buildFor({
              dom: $('<div/>').appendTo(jq),
              type: BooleanEditor,
              value: true,
              properties: {
                trueText: booleanEditorLabelText,
                falseText: booleanEditorLabelText
              }
            }).then(function (ed) {
              deleteFromAllSelected = ed;
              ed.jq().addClass('DeleteFromAllComponentsBoolean');
            });
          }
        }).ok(function () {
          return Promise.resolve(manager.hasMixIn('SELECTED_COMPONENTS_TABLE') && deleteFromAllSelected.read()).then(function (useComponentSelection) {
            var componentSelection;

            function getSelectedComponentsSelectedSubjects(manager) {
              return _.invoke(manager.getSelectedComponentsTable().getSelectedRows(), 'getSubject');
            }

            if (useComponentSelection) {
              componentSelection = getSelectedComponentsSelectedSubjects(manager);

              if (!componentSelection.length) {
                return;
              }
            } else {
              if (!manager.value()) {
                return;
              }

              componentSelection = [manager.value()];
            }

            if (!permissionToDelete(componentSelection)) {
              return dialogs.showOk({
                title: tagdictionaryLex.get('cannot.delete.tags'),
                content: tagdictionaryLex.getSafe('cannot.delete.tags.permission')
              });
            }

            var slots = _.map(manager.getMainTable().getSelectedRows(), function (row) {
              var rowSubject = row.getSubject();

              if (rowSubject instanceof baja.Tag) {
                return baja.SlotPath.escape(row.getSubject().getId().getQName());
              } else {
                // if the subject isn't a baja.Tag, it's a baja.Relation, so use its name
                return row.getSubject().getName();
              }
            });

            return removeAll(componentSelection, slots)["catch"](feDialogs.error);
          });
        }).promise();
      }
    });
  };

  function permissionToDelete(componentList) {
    var hasPermission = true;

    _.each(componentList, function (component) {
      hasPermission &= component.getPermissions().hasAdminWrite();
    });

    return hasPermission;
  }

  function selectionHasImpliedTags(manager) {
    return _.find(manager.getMainTable().getSelectedRows(), function (row) {
      return row.getTreeNode().isImplied();
    });
  }
  /**
   * Remove all the given slots from the all the components in the list,
   * batching into one network call.
   *
   * @inner
   * @param {Array.<baja.Component>} componentList
   * @param {Array.<baja.Slot|String>} slots
   * @returns {Promise}
   */


  function removeAll(componentList, slots) {
    var batch = new baja.comm.Batch(),
        removes = [];
    var tagName;

    _.forEach(componentList, function (component) {
      _.forEach(slots, function (slot) {
        if (component.has(slot)) {
          tagName = component.getSlot(slot).$getDisplayName();

          if (taggingUtil.tagIsEditable(component, tagName)) {
            removes.push(component.remove({
              slot: slot,
              batch: batch
            }));
          } else {
            logInfo(tagdictionaryLex.get({
              key: 'tag.not.deletable',
              args: [tagName, component.getDisplayName()]
            }));
          }
        }
      });
    });

    batch.commit();
    return Promise.all(removes);
  }

  DeleteTagCommand.prototype = Object.create(Command.prototype);
  DeleteTagCommand.prototype.constructor = DeleteTagCommand;
  return DeleteTagCommand;
});
