/*jshint browser: true */

/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/**
 * @private
 * @module nmodule/cloudBackup/rc/BackupManager
 */
define(['baja!',
        'baja!cloudBackup:CloudBackupJob,cloudBackup:CloudCreateBackupJob,cloudBackup:CloudDeleteBackupJob,cloudBackup:RandomizedTimeTrigger',
        'lex!',
        'jquery',
        'underscore',
        'Promise',
        'bajaux/Widget',
        'bajaux/mixin/subscriberMixIn',
        'bajaux/commands/Command',
        'bajaux/events',
        'nmodule/webEditors/rc/wb/table/Table',
        'nmodule/webEditors/rc/wb/table/model/TableModel',
        'nmodule/webEditors/rc/wb/table/model/Column',
        'nmodule/webEditors/rc/wb/table/model/Row',
        'nmodule/webEditors/rc/util/ListSelection',
        'bajaux/util/CommandButtonGroup',
        'nmodule/webEditors/rc/fe/fe',
        'dialogs',
        'lex!cloudBackup',
        'moment',
        'nmodule/cloudBackup/rc/backupUtils',
        'nmodule/cloudBackup/rc/filesize.min',
        'nmodule/cloudBackup/rc/moment-timezone-with-data.min',
        'hbs!nmodule/cloudBackup/rc/BackupManagerTemplate',
        'hbs!nmodule/cloudBackup/rc/BackupManagerDownloadTemplate',
        'hbs!nmodule/cloudBackup/rc/BackupManagerDownloadDecryptedTemplate',
        'hbs!nmodule/cloudBackup/rc/BackupManagerCreateBackupTemplate',
        'css!nmodule/cloudBackup/rc/BackupManagerStyle'], function (
        baja, 
        types,
        lexjs, 
        $,
        _,
        Promise,
        Widget,
        subscriberMixIn,
        Command,
        events,
        Table,
        TableModel,
        Column,
        Row,
        ListSelection,
        CommandButtonGroup,
        fe,
        dialogs,
        lexicons,
        moment,
        backupUtils,
        filesize,
        momentTimezone,
        backupMgrTemplate,
        downloadTemplate,
        downloadDecryptedTemplate,
        createBackupTemplate) {

  'use strict';

  var lex = lexicons[0];

  /**
   * Convert some back up info json to a row.
   *
   * @inner
   * @private
   *
   * @param {Object} An object that contains back up information.
   * @returns {Row} A row instance.
   */
  function backupInfoJsonToRows(obj) {
    var row = new Row(obj);

    _.each(obj, function (val, key) {
      row.data(key, val);
    });

    return row; 
  }

    /**
     * Extract total size of backups from Json array.
     *
     * @inner
     * @private
     *
     * @param {Array} An object that contains back up information, as a Json array
     * @returns {Number} total size in Bytes, may exceed 32-bit max int.
     */
    function getTotalSize(arr) {
      var size = arr.map(function(e) {
        return e.hasOwnProperty("size") && parseFloat(e.size);
      }).reduce(function (x, y) { return x + y; }, 0);
      return size;
    }

  /**
   * Update the summary information.
   *
   * @inner
   * @private
   * 
   * @param {module:nmodule/cloudBackup/rc/BackupManager} backupMgr The back up manager instance.
   * @param {Array} backupList A list of backup information.
   * @param {Object} limitsPolicy The backup limits policy.
   */
  function updateSummary(backupMgr, backupList, limitsPolicy) {
    var summary = backupMgr.$getSummaryElement(),
        policyType = limitsPolicy.policyType,
        totalSize = getTotalSize(backupList),
        used = filesize(totalSize, { base: 10 }),
        remaining,
        warning = false,
        text = "";

    if (policyType === "count") {
      if (limitsPolicy.limit - backupList.length < 0) {
        warning = true;
      }

      text = lex.get("backupSummaryCount", backupList.length, limitsPolicy.limit, used);
    }
    else if (policyType === "size") {
      remaining = Math.max(limitsPolicy.limit - totalSize, 0);

      if (remaining <= 0) {
        warning = true;
      }

      text = lex.get("backupSummarySize", backupList.length, used, filesize(limitsPolicy.limit, { base: 10 }));
    }
    else {      
      text = lex.get("backupSummary", backupList.length, used);
    }

    summary.toggleClass("BackupManager-warning", warning);
    summary.text(text);
  }

  /**
   * Return the back up information for the selected rows.
   *
   * @inner
   * @private
   * 
   * @param {module:nmodule/cloudBackup/rc/BackupManager} backupMgr The back up manager instance.
   * @return {Array<Object>}} An array of backup information.
   */
  function getSelectedBackupList(backupMgr) {
    return backupMgr.$table
      .$getSelection()
      .getSelectedElements(backupMgr.$tableModel.getRows())
      .map(function (row) {
        return {
          id: row.data("backupId"),
          fileName: row.data("fileName"),
          url: window.location.origin + "/cloudBackup/backups/" + row.data("backupId")
        };
      });
  }

  /**
   * Returns true if a job is running or canceling.
   *
   * @inner
   * @private
   * 
   * @param job The job to test to see if it's running.
   * @returns {Boolean} returns true if a backup job is currently running.
   */
  function isJobActive(job) {
    return job && (job.getJobState().is("running") || job.getJobState().is("canceling"));
  }

  /**
   * Show the backup job already running dialog.
   *
   * @inner
   * @private
   */
  function showBackupJobRunningDialog() {
    dialogs.showOk(lex.get("backupJobAlreadyRunning"));
  }

 /**
  * @class
  * @alias module:nmodule/cloudBackup/rc/BackupManager
  * @desc Manages Niagara Back ups.
  */
  var BackupManager = function () {
    var that = this,
        cmds = [];

    Widget.apply(that, arguments);
    subscriberMixIn(that);

    cmds.push(new Command({
      module: "cloudBackup",
      lex: "backup",
      func: function () {
        if (isJobActive(that.$currentJob)) {
          showBackupJobRunningDialog();
          return;
        }

        dialogs.showOkCancel({
          title: lex.get("backupNotes"),
          content: createBackupTemplate({
            master: lex.get("masterBackup"),
          })
        })
            .ok(function (dlg) {
              var notes = dlg.content().find(".backupNotes").val();
              var retention = dlg.content().find("input[value|=master]:checked").val();
              var createInfo = {'backupNotes' : notes, 'retentionPolicy' : retention};
              that.value().rpc("runBackup", createInfo);
            });
      }
    }));

    that.$deleteCommand = new Command({
      module: "cloudBackup",
      lex: "delete",
      enabled: false,
      func: function () {
        if (isJobActive(that.$currentJob)) {
          showBackupJobRunningDialog();
          return;
        }

        var backupList = getSelectedBackupList(that);

        if (!backupList.length) {
          return Promise.reject("No rows selected");
        }

        return dialogs.showYesNo(lex.get("delete.areYouSure"))
          .yes(function () {
            var backupIds = backupList.map(function (info) {
              return info.id;
            });

            that.value().rpc("delete", backupIds);        
         }).promise();
      }
    });
    cmds.push(that.$deleteCommand);

    that.$downloadCommand = new Command({
      module: "cloudBackup",
      lex: "download",
      enabled: false,
      func: function () {
        var downloadList = getSelectedBackupList(that);

        function download() {
          downloadList.forEach(function (info) {
            var downloadJq = $(downloadTemplate({
              url: info.url,
              key: info.key,
              decrypt: !!info.key
            }));

            $(document.body).append(downloadJq);
            
            try {
              downloadJq.submit();
            }
            finally {
              downloadJq.remove();
            }
          });
        }

        if (window.niagara.env.type === "wb") {
          that.value().rpc("triggerDownloadNotification", downloadList);
        }
        else {
          if (backupUtils.isSecure()) {
            dialogs.showOkCancel({
                title: lex.get("downloadDecrypted.title"),
                content: downloadDecryptedTemplate({
                  message: lex.get("downloadDecrypted.message"),
                  edist: lex.get("downloadDecrypted.edist"),
                  dist: lex.get("downloadDecrypted.dist")
                })
              })
              .ok(function (dlg) {
                if (dlg.content().find("input[value|=decrypted]:checked").val() === "decrypted") {
                  backupUtils.showPassphraseDialog(function (dlg, passphrase) {
                    // Request the key information and then compute it accordingly.
                    dialogs.showLoading(500, Promise.all(downloadList.map(function (info) {
                      return Promise.resolve($.get(info.url + "?filter=keyinfo"));
                    }))
                    .then(function (keyInfos) {
                      return Promise.all(keyInfos.map(function (ki) {
                        return backupUtils.pbkdf2(passphrase, ki.salt, ki.iterationCount, ki.keySize);
                      }));
                    })
                    .then(function (keys) {
                      keys.forEach(function (k, i) {
                        downloadList[i].key = k;
                      });

                      download();
                    }));  
                  });
                }
                else {
                  download();
                }                
              });
          }
          else {
            download();
          }
        }
      }
    });
    cmds.push(that.$downloadCommand);

    cmds.forEach(function (c) { that.getCommandGroup().add(c); });
  };
  
  BackupManager.prototype = Object.create(Widget.prototype);
  BackupManager.prototype.constructor = BackupManager;

  /**
   * Refresh the back up manager table.
   *
   * @returns {Promise} Returns a promise that's resolved once the table has been refreshed.
   */
  BackupManager.prototype.refreshTable = function () {
    // TODO: keep track of back up ids so we can keep selection after refresh?
    var list,
        that = this;

    // Only allow one refresh of the table at a time.
    if (that.$refreshPromise && that.$refreshPromise.isPending()) {
      return that.$refreshPromise;
    }    

    function unableToConnectToCloud(err) {
      dialogs.showOk((err && err.message) || lex.get("unableToConnectToCloud"));
    }

    return (that.$refreshPromise = dialogs.showLoading(0, Promise.all([
      that.value().rpc("listBackups"), that.value().rpc("getPolicy")])
      .spread(function (backupList, limitsPolicy) {
        if (!that.isInitialized()) {
          return;
        }

        updateSummary(that, backupList, limitsPolicy);
        list = backupList;
        return that.$tableModel.removeRows(that.$tableModel.getRows());
      }, unableToConnectToCloud)
      .then(function () {
        if (!that.isInitialized()) {
          return;
        }

        return that
          .$tableModel
          .insertRows(list.map(backupInfoJsonToRows));
      }))
      .promise());
  };


  /**
   * Get the element that holds the main table.
   * 
   * @private
   * @returns {jQuery}
   */
  BackupManager.prototype.$getMainTableElement = function () {
    return this.jq().children('.mainTable');
  };

  /**
   * Get the element that holds the command buttons.
   * 
   * @private
   * @returns {jQuery}
   */
  BackupManager.prototype.$getCommandContainerElement = function () {
    return this.jq().children('.commandContainer');
  };

  /**
   * Get the element that holds the job bar.
   * 
   * @private
   * @returns {jQuery}
   */
  BackupManager.prototype.$getJobBarElement = function () {
    return this.jq().find('.BackupManager-jobBar');
  };

  /**
   * Get the element that holds the summary.
   * 
   * @private
   * @returns {jQuery}
   */
  BackupManager.prototype.$getSummaryElement = function () {
    return this.jq().find('.BackupManager-summary');
  };

  /**
   * Get the element that holds backup information.
   * 
   * @private
   * @returns {jQuery}
   */
  BackupManager.prototype.$getInfoBarElement = function () {
    return this.jq().find('.BackupManager-infoBar');
  };

  /**
   * Back up Manager Table Column
   *
   * @inner
   * @private
   * @class
   */
  var BackupManagerColumn = function (name) {
    Column.apply(this, [name, { displayName: lex.get("column." + name) } ]);
  };

  BackupManagerColumn.prototype = Object.create(Column.prototype);
  BackupManagerColumn.prototype.constructor = BackupManager;

  /**
   * Access the data straight from the row.
   * 
   * @param row The row to access the information from.
   * @returns The value to display for the row and column.
   */
  BackupManagerColumn.prototype.getValueFor = function (row) {
    var name = this.getName(),
        value = row.data(name);

    if (name === "size") {
      return filesize(value, {base: 10});
    }
    else if (name === "timestamp") {
      var timezone = row.data("timeZone");
      return momentTimezone.utc(value).tz(timezone).format("lll z");
    }
    else if (name === "retention") {
      return (value.toLowerCase() === "master" ? "yes" : "no");
    }
    else {
      return value;
    }
  };

  /**
   * Build the backup table.
   *
   * @inner
   * @private
   * 
   * @param  {nmodule/cloudBackup/rc/BackupManager} backupMgr An instance of the back up manager.
   * @returns a promise that's resolves once the table has been created.
   */
  function buildTable(backupMgr) {
    var tableContainer = backupMgr.$getMainTableElement().children('.tableContainer'),
        timestampColumn = new BackupManagerColumn('timestamp'),
        fileNameColumn  = new BackupManagerColumn('fileName'),
        sizeColumn      = new BackupManagerColumn('size'),
      retentionColumn = new BackupManagerColumn('retention'),
        userColumn      = new BackupManagerColumn('remoteUser'),
        notesColumn     = new BackupManagerColumn('notes'),
      columns = [timestampColumn, fileNameColumn, sizeColumn, retentionColumn, userColumn, notesColumn],
        selection = new ListSelection();

    backupMgr.$tableModel = new TableModel({
      rows: [], 
      columns: columns
    });

    // Override the default handler to allow only one item to be selected in the list at a time.
    selection.defaultHandler = function (e) {
     var idx = $(this).index();
     if (e.which !== 3 || !selection.isSelected(idx)) {
       selection.select(idx);
     }
   };

   return fe.buildFor({
      dom: $('<table class="ux-table"></table>').appendTo(tableContainer),
      type: Table,
      value:  backupMgr.$tableModel,
      selection: selection
    }).then(function (table) {
      backupMgr.$table = table;
      var selection = table.$getSelection();

      // When the selection changes, update the commands.
      selection.on('changed', function () {
        backupMgr.$deleteCommand.setEnabled(!selection.isEmpty());
        backupMgr.$downloadCommand.setEnabled(!selection.isEmpty());
      });

      // When an entry is double clicked then show the notes in a dialog.
      table.jq().on('dblclick', 'tr', function (e) {
        var obj = table.getSubject($(e.target));

        if (obj && obj[0]) {
          dialogs.showOk({
            title: lex.get("backupNotes"),
            content: _.escape(obj[0].notes || "")
          });
        }
      });
    });
  }

  /**
   * Set up elements for the main table and command group.
   *
   * @param {jQuery} dom
   */
  BackupManager.prototype.doInitialize = function (dom) {
    var that = this;

    dom.html(backupMgrTemplate({
      loading: lex.get('loading')
    }));

    dom.addClass('BackupManager');

    return Promise.all([
      // Command container
      fe.buildFor({
        dom: $('<div/>').appendTo(that.$getCommandContainerElement()),
        type: CommandButtonGroup,
        value: that.getCommandGroup(),
        formFactor: Widget.formfactor.mini
      }),
      // Backup table
      buildTable(that),
      // Backup job bar
      fe.buildFor({
        dom: that.$getJobBarElement(),
        type: "cloudBackup:CloudCreateBackupJob",
        formFactor: Widget.formfactor.mini
      })
      .then(function (JobBarWidget) {
        that.$jobBar = JobBarWidget;
      })
    ]);
  };

  /**
   * Monitors a job's progress and updates the user interface accordingly.
   *
   * @inner
   * @private
   * 
   * @param  {nmodule/cloudBackup/rc/BackupManager} backupMgr An instance of the backup manager.
   * @param  job The job to monitor.
   * @returns {Promise} A promise that's resolved once the job is being monitored.
   */
  function monitorJob(backupMgr, job) {
    if (!backupMgr.$jobBar) {
      return Promise.reject();
    }

    if (backupMgr.$currentJob) {
      backupMgr.getSubscriber().unsubscribe(backupMgr.$currentJob);
    }

    backupMgr.$currentJob = job;
    backupMgr.$jobBar.load(job);

    return backupMgr.getSubscriber().subscribe(job);
  }

  /**
   * Load the table using data from the backup service.
   *
   * @param backupSrv The backup Service.
   * @returns {Promise}
   */
  BackupManager.prototype.doLoad = function (backupSrv) {
    var that = this,
        timeTrigger = backupSrv.getTimeTrigger();

    that.refreshTable()
        .then(baja.noop, function error() {
          updateSummary(that, [], {});
        });

    that.getSubscriber().attach({
      changed: function (prop, cx) {
        if (cx && cx.fromLoad) {
          return;
        }

        // If a cloud backup job completes then update the table.
        if (this.getType().is("cloudBackup:CloudBackupJob") &&
            prop.getName() === "jobState" &&
            !isJobActive(this)) {
          that.refreshTable();
        }
      },
      added: function (prop, cx) {
        if (cx && cx.fromLoad) {
          return;
        }

        var val = this.get(prop);

        // If a cloud backup job has been added then monitor it and refresh
        // the table if it's already completed.
        if (val.getType().is("cloudBackup:CloudBackupJob")) {
          monitorJob(that, val).then(function () {
            if (!isJobActive(val)) {
              that.refreshTable();
            }
          });
        }
      }
    });

    // Subscribe to the job service and find the last cloud job available.
    return Promise.all([
      baja.Ord.make("service:baja:JobService")
          .get({subscriber: that.getSubscriber()})
          .then(function (jobService) {
                  return that.value().rpc("getLastBackupJob").
                  then(function (jobName) {
                    if (jobName) {
                      var job = jobService.get(jobName);
                      if (job) {
                        return monitorJob(that, job);
                      }
                    }
                  })
           }),
      that.getSubscriber().subscribe(timeTrigger)
          .then(function () {
            var jq = that.jq(),
                timeTriggerLabel = timeTrigger.getDisplayName("triggerMode"),
                nextTriggerLabel = timeTrigger.getDisplayName("nextTrigger"),
                triggerLabelJq = jq.find(".BackupManager-triggerModeLabel"),
                triggerJq = that.jq().find(".BackupManager-triggerMode"),
                triggerNextTriggerLabelJq = jq.find(".BackupManager-triggerNextTriggerLabel"),
                nextTriggerJq = that.jq().find(".BackupManager-triggerNextTrigger"),
                nextTriggerOuterJq = that.jq().find(".BackupManager-triggerNextTriggerOuter");

            triggerLabelJq.text(timeTriggerLabel);
            triggerNextTriggerLabelJq.text(nextTriggerLabel);

            function saveFe(fe) {
              return function () {
                setTimeout(function () {
                  fe.save();
                }, 20);
              };
            }

            return Promise.all([
              // Build the field editor for editing the trigger mode.
              fe.buildFor({
                dom: triggerJq,
                complex: timeTrigger,
                slot: "triggerMode",
                formFactor: Widget.formfactor.mini
              }).then(function (fe) {
                that.$timeTriggerFe = fe;

                fe.jq().on(events.MODIFY_EVENT, saveFe(fe));
              }),
              // Build the field editor for editing the next trigger.
              fe.buildFor({
                dom: nextTriggerJq,
                complex: timeTrigger,
                slot: "nextTrigger",
                formFactor: Widget.formfactor.mini
              })
                  .then(function (fe) {
                    that.$nextTriggerFe = fe;

                    fe.validators().add(function (absTime) {
                      // Make sure the user can only select a future date and time!
                      return moment(absTime.getJsDate()).isBefore(moment()) ?
                          Promise.reject(lex.get("chooseDateTimeInFuture")) : Promise.resolve();
                    });

                    fe.jq().on(events.MODIFY_EVENT, saveFe(fe));

                    function triggerModeUpdate() {
                      nextTriggerOuterJq.toggle(!timeTrigger.getTriggerMode().is("manual"));
                    }

                    that.getSubscriber().attach("changed", function (prop) {
                      if (this.getType().is("cloudBackup:RandomizedTimeTrigger") && prop.getName() === "triggerMode") {
                        triggerModeUpdate();
                      }
                    });

                    triggerModeUpdate();
                  })
            ]);
          })
    ]);
  };

   /**
   * Update the height of the main table element so that the command buttons
   * are always visible.
   */
  BackupManager.prototype.doLayout = function () {
    var that = this;
    that.$getMainTableElement().css({
      top: that.$getInfoBarElement().outerHeight(),
      bottom: that.$getCommandContainerElement().outerHeight()
    });
  };

  /**
   * Destroy child editors.
   *
   * @returns {Promise}
   */
  BackupManager.prototype.doDestroy = function () {
    var that = this,
        promises = [];

    that.jq().removeClass('BackupManager');

    if (that.$table) {
      promises.push(that.$table.destroy());
    }

    if (that.$jobBar) {
      promises.push(that.$jobBar.destroy());
    }   

    if (that.$timeTriggerFe) {
      promises.push(that.$timeTriggerFe.destroy());
    } 

    if (that.$nextTriggerFe) {
      promises.push(that.$nextTriggerFe.destroy());
    } 

    return Promise.all(promises);
  };

  return BackupManager;
});

