/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 */

/**
 * @private
 * @module nmodule/jsonToolkit/rc/fe/SlotSelectionTypeEditor
 */
define([
    'baja!',
    'baja!jsonToolkit:SlotSelectionType,jsonToolkit:RelativeJsonSchema,jsonToolkit:JsonSchemaBoundObject,jsonToolkit:JsonSchema',
    'log!nmodule.jsonToolkit.rc.fe.SlotSelectionTypeEditor',
    'jquery',
    'Promise',
    'underscore',
    'nmodule/js/rc/switchboard/switchboard',
    'nmodule/webEditors/rc/fe/baja/util/compUtils',
    'nmodule/webEditors/rc/fe/baja/BaseEditor',
    'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber',
    'nmodule/webEditors/rc/fe/baja/FrozenEnumEditor',
    'nmodule/webEditors/rc/fe/fe',
    'hbs!nmodule/jsonToolkit/rc/fe/template/SlotSelectionTypeEditor',
    'css!nmodule/jsonToolkit/rc/jsonToolkit'
  ], function (
    baja,
    types,
    log,
    $,
    Promise,
    _,
    switchboard,
    compUtils,
    BaseEditor,
    DepthSubscriber,
    FrozenEnumEditor,
    fe,
    template
    ) {

  'use strict';

  var logError = log.severe.bind(log),
    logFine = log.fine.bind(log),
    closest = compUtils.closest,
    SLOT_SELECTION_TYPE_ENUM = baja.$('jsonToolkit:SlotSelectionType'),
    ALL_VISIBLE_SLOTS = SLOT_SELECTION_TYPE_ENUM.make(1),
    SELECTED_SLOTS = SLOT_SELECTION_TYPE_ENUM.make(3);


  /**
   * A field editor for working with 'jsonToolkit:SlotSelectionType' components.
   *
   * @private
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/jsonToolkit/rc/fe/SlotSelectionTypeEditor
   */
  var SlotSelectionTypeEditor = function SlotSelectionTypeEditor(params) {
    BaseEditor.call(this, _.extend({
      keyName: 'SlotSelectionTypeEditor'
    }, params));

    switchboard(this, {
      '$updateSlotSelection': { allow: 'oneAtATime', onRepeat: 'preempt' }
    });
  };
  SlotSelectionTypeEditor.prototype = Object.create(BaseEditor.prototype);
  SlotSelectionTypeEditor.prototype.constructor = SlotSelectionTypeEditor;

  /**
   * Creates a select dropdown.
   *
   * @param {JQuery} dom
   */
  SlotSelectionTypeEditor.prototype.doInitialize = function (dom) {
    var that = this;

    dom.addClass('SlotSelectionTypeEditor');

    dom.html(
      template()
    );

    dom.on('change', '.slot-selection-enum', function () {
      that.$updateSlotSelection().catch(logError);
      return false;
    });

    dom.on('click', '.add-slot-button', function () {
      that.$addSelectedSlot().catch(logError);
      return false;
    });

    dom.on('click', '.delete-slot-button', function (e) {
      that.$deleteSelectedSlot(e.target.getAttribute('data-slot-to-delete')).catch(logError);
      return false;
    });

    return fe.buildFor({
      dom: $('<span/>').appendTo(dom.find('.slot-selection-enum')),
      type: FrozenEnumEditor,
      value: ALL_VISIBLE_SLOTS, // default value, the actual value will be loaded in doLoad.
      formFactor: 'mini'
    })
      .then(function (enumEditor) {
        that.$enumEditor = enumEditor;
      });
  };

  /**
   * Assembles the SlotSelectionType Enum into a set of options to load into the select
   * dropdown.
   *
   * @param {baja.Enum} value a jsonToolkit:SlotSelectionType enum
   * @returns {Promise}
   */
  SlotSelectionTypeEditor.prototype.doLoad = function (value) {
    var that = this,
      sub = that.$getSubscriber();

    return Promise.all([
      that.$enumEditor.load(value),
      sub.subscribe(that.getComplex()).catch(logError)
    ])
      .then(function () {
        sub.attach('topicFired', _.bind(that.$changedHandler, that));

        // Don't resolve the binding for a relative schema
        if (that.$isRelativeSchema()) { return; }

        return that.getComplex().get('binding').get({ subscriber: sub })
          .then(function (binding) {
            that.$binding = binding;
          })
          .catch(logFine);
      })
      .then(function () {
        return that.$updateSlotSelection();
      });
  };

  /**
   * update following itemAdded or itemRemoved topics
   *
   * @private
   * @param {baja.Topic} topic
   */
  SlotSelectionTypeEditor.prototype.$changedHandler = function (topic) {
    // act on ListOf.itemAdded or ListOf.itemRemoved topics
    if (topic.getName() === 'itemAdded' || topic.getName() === 'itemRemoved') {
      this.$alreadySelectedSlots = null;  // force $alreadySelectedSlots to refresh
      this.$updateSlotSelection(true).catch(logError);
    }
  };

  /**
   * return true is the parent JsonSchema is relative
   *
   * @private
   * @returns {Boolean}
   */
  SlotSelectionTypeEditor.prototype.$isRelativeSchema = function () {
    var schemaParent = closest(this.getComplex(), 'jsonToolkit:JsonSchema');
    return schemaParent.getType().is('jsonToolkit:RelativeJsonSchema');
  };

  /**
   * get a subscriber for this editor
   *
   * @private
   * @returns {baja.DepthSubscriber}
   */
  SlotSelectionTypeEditor.prototype.$getSubscriber = function () {
    if (!this.$subscriber) {
      this.$subscriber = new DepthSubscriber(3);
    }
    return this.$subscriber;
  };

  /**
   * Get the slots that have already been selected - applies to the SelectedSlots
   * option only.
   * This value is stored on the parent complex in the 'slotList' slot.
   *
   * @private
   * @returns {Array.<String>}
   */
  SlotSelectionTypeEditor.prototype.$getAlreadySelectedSlots = function () {
    if (!this.$alreadySelectedSlots) {
      var slotList = this.getComplex().getSlotList(),
        alreadySelectedSlots = [];

      slotList.getVector().getSlots().properties().dynamic().each(function (prop) {
        alreadySelectedSlots.push(prop.$val);
      });
      this.$alreadySelectedSlots = alreadySelectedSlots;
    }
    return this.$alreadySelectedSlots;
  };

  /**
   * update when the SlotSelectionType Enum dropdown is changed
   *
   * @returns {Promise}
   */
  SlotSelectionTypeEditor.prototype.$updateSlotSelection = function () {
    var that = this,
      addSlotButtonElement = that.jq().find('.add-slot-button'),
      alreadySelectedSlotsElement = that.jq().find('.selected-slots'),
      addSlotDropdown = that.$getAddSlotSelect(),
      addSlotTextInput = that.$getAddSlotInput(),
      showSlotSelection = false,
      showDropDown = true;

    return that.$enumEditor.read()
      .then(function (currentSelection) {
        addSlotDropdown.empty();
        addSlotTextInput.val('');
        alreadySelectedSlotsElement.empty();

        if (currentSelection === SELECTED_SLOTS) {

          showSlotSelection = true;

          if (that.$isRelativeSchema()) {
            showDropDown = false;
            that.$getAlreadySelectedSlots().forEach(function (slotName) {
              alreadySelectedSlotsElement.append(
                '<div>' +
                '<span>' + slotName + '</span>' +
                '<span data-slot-to-delete="' + slotName + '" class="delete-slot-button icon-icons-x16-delete" />' +
                '</div>'
              );
            });
          } else {
            addSlotDropdown.append('<option class="ux-option" value=""></option>');

            if (that.$binding) {
              // populate the selected slots dropdown
              that.$binding.getSlots().properties().each(function (slot) {
                if (that.$getAlreadySelectedSlots().indexOf(slot.getName()) < 0) {
                  addSlotDropdown.append(
                    '<option class="ux-option" value="' + slot.getName() + '">' + slot.getName() + '</option>'
                  );
                } else {
                  alreadySelectedSlotsElement.append(
                    '<div>' +
                    '<span>' + slot.getName() + '</span>' +
                    '<span data-slot-to-delete="' + slot.getName() + '" class="delete-slot-button icon-icons-x16-delete" ' + '</div>'
                  );
                }
              });
            }
          }
        }

        addSlotDropdown.toggle(showSlotSelection && showDropDown);
        addSlotTextInput.toggle(showSlotSelection && !showDropDown);
        addSlotButtonElement.toggle(showSlotSelection);
        alreadySelectedSlotsElement.toggle(showSlotSelection);
      });
  };

  /**
   * add a selected slot, and update the editor
   *
   * @returns {Promise}
   */
  SlotSelectionTypeEditor.prototype.$addSelectedSlot = function () {
    if (this.isReadonly() || !this.isEnabled()) { return Promise.resolve(); }

    var addSlotDropdown = this.$getAddSlotSelect(),
      addSlotTextInput = this.$getAddSlotInput(),
      inputSelector = this.$isRelativeSchema() ? addSlotTextInput : addSlotDropdown,
      slotToAdd = inputSelector.val();

    if (!slotToAdd) { return Promise.resolve(); }

    this.$getAlreadySelectedSlots().push(slotToAdd);
    this.setModified(true);

    return this.$updateSlotSelection();
  };

  /**
   * delete a selected slot, and update the editor
   *
   * @returns {Promise}
   */
  SlotSelectionTypeEditor.prototype.$deleteSelectedSlot = function (slotToDelete) {

    if (this.isReadonly() || !this.isEnabled()) { return Promise.resolve(); }

    var alreadySelectedSlots = this.$getAlreadySelectedSlots(),
      indexToRemove = alreadySelectedSlots.indexOf(slotToDelete);

    if (indexToRemove < 0) { return Promise.resolve(); }

    alreadySelectedSlots.splice(indexToRemove, 1);

    this.setModified(true);

    return this.$updateSlotSelection();
  };

  /**
   * Read the individual child editors and assemble them into a `PermissionsMap`
   *
   * @returns {Promise.<baja.Enum>} promise to be resolved with a 'jsonToolkit:SlotSelectionType' value
   */
  SlotSelectionTypeEditor.prototype.doRead = function () {
    var that = this;
    return that.$enumEditor.read();
  };

  /**
   * save the SlotSelectionType value
   * @param {baja.Enum} readValue - a 'jsonToolkit:SlotSelectionType' value
   * @param {Object} params
   * @returns {Promise}
   */
  SlotSelectionTypeEditor.prototype.doSave = function (readValue, params) {
    var that = this,
      slotList = that.getComplex().get('slotList'),
      batch = new baja.comm.Batch(),
      promises = [];

    promises.push(slotList.invoke({ slot: 'clearList', batch: batch }));

    if (readValue === SELECTED_SLOTS) {
      that.$getAlreadySelectedSlots().forEach(function (selectedSlot) {
        promises.push(
          slotList.invoke({ slot: 'addItem', value: selectedSlot, batch: batch })
        );
      });
    }
    that.$alreadySelectedSlots = null; // force $alreadySelectedSlots to refresh

    return batch.commit(Promise.all(promises))
      .then(function () {
        return BaseEditor.prototype.doSave.apply(that, arguments);
      });
  };

  /**
   * Get the 'addSlot' select element.
   *
   * @private
   * @returns {JQuery}
   */
  SlotSelectionTypeEditor.prototype.$getAddSlotSelect = function () {
    return this.jq().find('select.slot-selection');
  };

  /**
   * Get the 'addSlot' input element.
   *
   * @private
   * @returns {JQuery}
   */
  SlotSelectionTypeEditor.prototype.$getAddSlotInput = function () {
    return this.jq().find('input.slot-selection');
  };

  /**
   * Enables or disables the select dropdown.
   *
   * @param {Boolean} enabled
   */
  SlotSelectionTypeEditor.prototype.doEnabled = function (enabled) {
    var disabled = this.isReadonly() || !enabled;

    this.$getAddSlotSelect().prop('disabled', disabled);
    this.$getAddSlotInput().prop('disabled', disabled);

    return this.getChildEditors().setAllEnabled(enabled);
  };

  /**
   * Disables or enables the select dropdown.
   *
   * @param {Boolean} readonly
   */
  SlotSelectionTypeEditor.prototype.doReadonly = function (readonly) {
    var disabled = !this.isEnabled() || readonly;

    this.$getAddSlotSelect().prop('disabled', disabled);
    this.$getAddSlotInput().prop('disabled', disabled);

    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Clean up after this editor
   * @returns {Promise}
   */
  SlotSelectionTypeEditor.prototype.doDestroy = function () {
    var that = this;

    that.jq().removeClass('SlotSelectionTypeEditor');

    return Promise.resolve(that.$subscriber && that.$subscriber.unsubscribeAll())
      .then(function () {
        return that.getChildWidgets().destroyAll();
      });
  };

  return (SlotSelectionTypeEditor);
});
