/**
 * @copyright 2019 Tridium, Inc. All Rights Reserved.
 */

/**
 * @private
 * @module nmodule/jsonToolkit/rc/fe/QueryStylePicker
 */
define([
    'baja!',
    'baja!jsonToolkit:JsonSchema,jsonToolkit:JsonSchemaBoundQueryResult,jsonToolkit:QueryResultWriter',
    'log!nmodule.jsonToolkit.rc.fe.QueryStylePicker',
    'Promise',
    'underscore',
    'nmodule/webEditors/rc/fe/baja/BaseEditor',
    'nmodule/webEditors/rc/fe/baja/util/typeUtils',
    'lex!jsonToolkit',
    'css!nmodule/jsonToolkit/rc/jsonToolkit'
  ], function (
    baja,
    types,
    log,
    Promise,
    _,
    BaseEditor,
    typeUtils,
    lexs
  ) {

  'use strict';

  var jsonToolkitLex = lexs[0],
    logFine = log.fine.bind(log),
    logError = log.severe.bind(log);

  /**
   * A field editor for working with the 'query' ('baja:String') slot on a
   * 'jsonToolkit:JsonSchemaBoundQueryResult' component
   *
   * @private
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/jsonToolkit/rc/fe/QueryStylePicker
   */
  var QueryStylePicker = function QueryStylePicker(params) {
    BaseEditor.call(this, _.extend({
    //BaseEditor.call(this, $.extend({
      keyName: 'QueryStylePicker'
    }, params));
  };
  QueryStylePicker.prototype = Object.create(BaseEditor.prototype);
  QueryStylePicker.prototype.constructor = QueryStylePicker;

  /**
   * Initializes this editor.
   *
   * @param {JQuery} dom
   */
  QueryStylePicker.prototype.doInitialize = function (dom) {
    var that = this;

    dom.addClass('QueryStylePicker');

    dom.html(
      '<span><select class="style-selection" /></span>' +
      '<span class="style-preview">' + _.escape(jsonToolkitLex.get('preview')) + '</span>'
    );

    dom.on('change', '.style-selection', function () {
      that.setModified(true);
      that.$populatePreview().catch(logError);
      return false;
    });
  };

  /**
   * loads the queryName into the editor
   *
   * @param {baja.Type} queryStyleType
   * @returns {Promise}
   */
  QueryStylePicker.prototype.doLoad = function (queryStyleType) {
    return this.$populateSelectOptions();
  };

  /**
   * populates the output style preview hover text
   *
   * @private
   * @returns {Promise}
   */
  QueryStylePicker.prototype.$populatePreview = function () {
    var previewElement = this.jq().find('.style-preview'),
      currentTypeSpec = this.$getSelect().val();

    // set the preview to blank by default
    previewElement.attr('title', '');

    if (!currentTypeSpec) {
      // no typeSpec, leave the preview as blank and log the error.
      logFine('no current typeSpec:');
      return Promise.resolve();
    }

    return baja.rpc({
      typeSpec: 'jsonToolkit:JsonToolkitRpcUtil',
      methodName: 'retrievePreviewText',
      args: [ currentTypeSpec ]
    })
      .then(function (previewText) {
        previewElement.attr('title', previewText);
      })
      .catch(function (err) {
        // leave the preview as blank and log the error.
        logFine('rpc lookup failed: ' + err);
      });
  };


  /**
   * populates the dropdown
   *
   * @private
   * @returns {Promise}
   */
  QueryStylePicker.prototype.$populateSelectOptions = function () {
    var that = this,
      selectElement = that.$getSelect();

    // clear out the dropdown options
    that.$getSelect().empty();

    // add a blank element to use if the current value no longer exists,
    // this element will be removed it the current value is an option.
    selectElement.append(
      '<option class="ux-option blank-option" value=""></option>'
    );
    // and make it the selected one.
    selectElement.val('');

    return that.$getQueryStyleOptions()
      .then(function (queryStyleOptions) {
        queryStyleOptions.forEach(function (queryStyleType) {
          selectElement.append(
            '<option class="ux-option" value="' + _.escape(queryStyleType.id) + '">' +
            _.escape(queryStyleType.displayName) + '</option>'
          );

          if (queryStyleType.typeSpec === String(that.value())) {
            // if this match the current value, set it as the selected option
            selectElement.val(queryStyleType.id);
            // remove the blank option
            that.jq().find('.blank-option').remove();
          }
        });

        return that.$populatePreview();
      });
  };

  /**
   * returns a list of available style types
   *
   * @private
   * @returns {Promise.<Array.<baja.Type>>}
   */
  QueryStylePicker.prototype.$getQueryStyleOptions = function () {
    // agents on BJsonSchemaBoundQueryResult extending BQueryResultWriter
    return baja.registry.getAgents('type:jsonToolkit:JsonSchemaBoundQueryResult', [ 'jsonToolkit:QueryResultWriter' ]);
  };

  /**
   * Reads the select element and return its baja.TypeSpec
   *
   * @returns {baja.TypeSpec}
   */
  QueryStylePicker.prototype.doRead = function () {
    var selected = this.$getSelect().val();

    return typeUtils.toTypeSpec(selected);
  };

  /**
   * Get the select element.
   *
   * @private
   * @returns {JQuery}
   */
  QueryStylePicker.prototype.$getSelect = function () {
    return this.jq().find('.style-selection');
  };

  /**
   * Enables or disables the select dropdown.
   *
   * @param {Boolean} enabled
   */
  QueryStylePicker.prototype.doEnabled = function (enabled) {
    this.$getSelect().prop('disabled', this.isReadonly() || !enabled);
  };

  /**
   * Disables or enables the select dropdown.
   *
   * @param {Boolean} readonly
   */
  QueryStylePicker.prototype.doReadonly = function (readonly) {
    this.$getSelect().prop('disabled', !this.isEnabled() || readonly);
  };

  /**
   * Clean up after this editor
   * @returns {Promise}
   */
  QueryStylePicker.prototype.doDestroy = function () {
    this.jq().removeClass('QueryStylePicker');
  };

  return (QueryStylePicker);
});
