/*global niagara*/

/**
 * @license Copyright 2011, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Mobile Px App.
 *
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(function (require) {
  "use strict";

  require('jquerymobile');

  require('css!mobile/px/px');

  var baja = require('baja!'),
      lexs = require('lex!mobile'),
      $ = require('jquery'),
      PxPage = require('mobile/px/PxPage'),
      Promise = require('Promise'),
      _ = require('underscore'),
      Command = require('bajaux/commands/Command'),
      pages = require('mobile/util/mobile/pages'),
      dialogs = require('mobile/util/mobile/dialogs'),
      commands = require('mobile/util/mobile/commands'),
      views = require('mobile/util/mobile/selectview'),
      mobileUtil = require('mobile/util/mobile/mobile'),
      initializeMobileHxPx = require('mobile/px/hxPx/MobileHxPxView'),
      pageStack = [],
      pageStackSize = 5,
      // The limit to the number of PxPage DOM elements to cache
  mobileLex = lexs[0]; ////////////////////////////////////////////////////////////////
  // PageStack
  ////////////////////////////////////////////////////////////////  

  /**
   * Add a page DOM element to the internal stack.
   * 
   * The page stack keeps track of the loaded page DOM elements. A DOM element 
   * will be removed from the DOM if the stack size is exceeded.
   * 
   * @param pageDom the DOM element for an entire Px page
   */


  function addToPageStack(pageDom) {
    var ord = pageDom.jqmData("pageData").ord,
        i; // If the pageDom already exists in the stack then remove it

    for (i = 0; i < pageStack.length; ++i) {
      if (pageStack[i].jqmData("pageData").ord === ord) {
        pageStack.splice(i, 1);
        break;
      }
    } // Push this page onto the stack


    pageStack.push(pageDom); // If the limit of the stack has been reached then remove the
    // oldest DOM element from the page

    if (pageStack.length > pageStackSize) {
      $(pageStack.shift()).remove();
    }
  } ////////////////////////////////////////////////////////////////
  // Pages
  ////////////////////////////////////////////////////////////////  


  pages.register(function ordMatcher(str) {
    // Handle all ORDs that match this String  
    if (window.niagara.env.type === "mobile") {
      return str.match(/^ord/);
    }

    return false;
  }, function pxPageHandler() {
    // All of these functions are handlers for the Pages framework

    /**
     * If the current PX page is modified, shows a confirmation dialog asking
     * whether to save changes before invoking the action.
     * 
     * @returns {Promise} a promise to be resolved if the user chose to
     * invoke the action, whether or not saving occurred. It is resolved with
     * the dialog invocation, enabling things like
     * <code>dlg.redirect(anotherPage);</code>. It will be resolved with `null`
     * if there was no dialog shown (the page was not modified). 
     * If the user clicked Cancel, it will never be resolved.
     */
    function confirmAbandonChanges(params) {
      var rootPage, rootWidget; // Capture any save changes if the user attempts to navigate away from an unsaved Px Page

      if (PxPage.isCurrentPxPageModified()) {
        rootPage = PxPage.getCurrentPxPage();
        rootWidget = rootPage.$rootWidget; // eslint-disable-next-line promise/avoid-new

        return new Promise(function (resolve, reject) {
          var dlg = dialogs.confirmAbandonChanges({
            redirect: params && params.redirect,
            viewName: rootPage.getDisplayName(),
            yes: function yes(cb) {
              // Save the Widgets
              // eslint-disable-next-line promise/catch-or-return
              rootWidget.saveWidgets().then(function () {
                resolve(dlg);
                cb.ok();
              });
            },
            no: function no(cb) {
              // Clear all modified changes
              rootWidget.clearWidgetTreeModified();
              resolve(dlg);
              cb.ok();
            }
          });
        });
      } else {
        return Promise.resolve();
      }
    }

    function decodeUrl(url) {
      return {
        ord: baja.Ord.make(url.href)
      };
    }

    function encodeUrl(pageData) {
      var ord = pageData.ord || niagara.view.ord,
          url = baja.Ord.make(ord).toUri();
      return url;
    }

    var pxHtml = "<div data-role='page' data-theme='c' id='px' >" + "<div data-role='header' data-theme='a'>" + "<h1 class='pxHeader profileHeader'>{title}</h1>" + "<a class='profileHeader profileHeaderBack' data-icon='arrow-l' data-iconpos='notext'></a>" + "<a class='commandsButton profileHeader' id='commandsButton' title='{menuBtn}' data-icon='bars' data-role='button' data-iconpos='notext'></a>" + "</div>" + "<div class='pxContent' data-role='content'>" + "</div>" + "</div>";

    function createPage(obj) {
      var pageData = obj.pageData,
          ord = pageData.ord; // TODO: is this legal?

      function loadPx(title, bson) {
        var pageDiv = $(pxHtml.patternReplace({
          title: mobileLex.get("loading"),
          menuBtn: mobileLex.get("menu")
        })).appendTo(mobileUtil.getPageContainer());
        pageDiv.find('.profileHeaderBack').off().on('click', function () {
          // eslint-disable-next-line promise/catch-or-return
          confirmAbandonChanges().then(function () {
            history.back();
          });
        }); // Add the pxPage to the page data      

        pageData.pxPage = new PxPage(pageDiv, bson, ord, title);
        obj.ok(pageDiv);
      } // If maps to the originally loaded Px Page


      if (ord.toString() === niagara.view.ord) {
        loadPx(niagara.view.px.displayName, niagara.view.px.bson);
      } else {
        // Make a Servlet call to the Px App to get the page data
        $.ajax({
          url: ord.toUri(),
          type: "POST",
          headers: {
            "niagara-mobile-rpc": "px"
          },
          success: function success(data) {
            loadPx(data.displayName, data.bson);
          }
        });
      }
    }

    function pagebeforehide(obj) {
      var pageData = obj.page.jqmData("pageData"),
          nextPageData = obj.nextPage.jqmData("pageData"),
          nextPxPage = nextPageData && nextPageData.pxPage ? nextPageData.pxPage : {}; // Only stop this page if the current page is Px and the next page isn't a dialog

      if (pageData && pageData.pxPage && obj.nextPage.jqmData("role") !== "dialog" && nextPxPage !== pageData.pxPage) {
        pageData.pxPage.stop();
      }
    }

    function pagelayout(obj) {
      var pageData = obj.page.jqmData("pageData"),
          pxPage = pageData && pageData.pxPage;

      if (pxPage) {
        // eslint-disable-next-line promise/catch-or-return
        pxPage.load().then(function () {
          pxPage.start();
        });
      }
    }

    function pageshow(obj) {
      addToPageStack(obj.page);
    }

    function pagebeforechange(obj) {
      var newId = obj.nextPage.attr('id'),
          currentId = obj.page.attr('id');

      if (PxPage.isCurrentPxPageModified() && newId !== currentId) {
        obj.event.preventDefault();
        confirmAbandonChanges({
          redirect: mobileUtil.decodePageId(newId)
        });
      }
    }

    function getCommands(obj) {
      var cmds = obj.commands,
          homeCmd = commands.getHomeCmd(),
          homeIndex = _.indexOf(cmds, homeCmd),
          logoutCmd = commands.getLogoutCmd(),
          logoutIndex = _.indexOf(cmds, logoutCmd),
          pxPage = PxPage.getCurrentPxPage();

      if (homeIndex >= 0) {
        homeCmd.toDisplayName(function (displayName) {
          cmds[homeIndex] = new Command(displayName, function () {
            return confirmAbandonChanges().then(function () {
              return homeCmd.invoke();
            });
          });
        });
      }

      if (logoutIndex >= 0) {
        logoutCmd.toDisplayName(function (displayName) {
          cmds[logoutIndex] = new Command(displayName, function () {
            return confirmAbandonChanges().then(function () {
              return logoutCmd.invoke();
            });
          });
        });
      }

      if (pxPage) {
        // Add refresh command
        cmds.splice(0, 0, new Command("%lexicon(mobile:refresh)%", function () {
          return pxPage.$rootWidget.refresh();
        })); // Only add the save command if the current view is modified

        if (PxPage.isCurrentPxPageModified()) {
          // Add save command
          cmds.splice(0, 0, new Command("%lexicon(mobile:save)%", function () {
            return pxPage.$rootWidget.saveWidgets().then(function () {
              // eslint-disable-next-line promise/avoid-new
              return new Promise(function (resolve, reject) {
                dialogs.ok({
                  title: mobileLex.get("saved"),
                  content: mobileLex.getSafe("pxPageSaved"),
                  ok: resolve
                });
              });
            });
          }));
        }
      }

      return cmds;
    } // Functions exported as public


    return {
      decodeUrl: decodeUrl,
      encodeUrl: encodeUrl,
      createPage: createPage,
      pagebeforehide: pagebeforehide,
      pagelayout: pagelayout,
      pageshow: pageshow,
      pagebeforechange: pagebeforechange,
      getCommands: getCommands
    };
  }()); // pxPageHandler
  ////////////////////////////////////////////////////////////////
  // Scroll Position Restore
  //////////////////////////////////////////////////////////////// 

  (function scrollPositionRestore() {
    var scrollDataKey = "scrollData";
    $(document).on("pagechange", function pagechange(e, data) {
      // Check we're restoring to a page. If so then attempt to restore the scroll positions  
      var toPage = data.toPage,
          scrollData;

      if (toPage && toPage instanceof $ && toPage.jqmData("role") === "page") {
        // Restore the scroll information once the page has shown
        scrollData = toPage.jqmData(scrollDataKey);

        if (scrollData) {
          $(document).one("silentscroll", function () {
            // On silent scroll event, schedule a scroll reset shortly afterwards
            // (must be greater than 20 ms)
            baja.clock.schedule(function () {
              window.scrollTo(scrollData.x, scrollData.y);
            }, 30);
          });
        }
      }
    });
    $(document).on("pagebeforechange", function pagebeforechange(e, data) {
      // In order to note the scroll positions so we can restore them between page
      // changes, // we need to pick up this event so we can note the scroll positions
      // down before anything on the page changes.
      var scrollData,
          toPage = data.toPage,
          toRole,
          fromPage = $.mobile.activePage,
          fromRole; // If we're going to a dialog from a page then note down the scroll positions    

      if (fromPage && fromPage instanceof $ && toPage && toPage instanceof $) {
        toRole = toPage.jqmData("role");
        fromRole = fromPage.jqmData("role"); // If we're going to a dialog then note the scroll position

        if (fromRole === "page" && toRole === "dialog") {
          scrollData = fromPage.jqmData(scrollDataKey); // Lazily create the scroll data if it doesn't exist

          if (!scrollData) {
            scrollData = {};
            fromPage.jqmData(scrollDataKey, scrollData);
          } // Note down the scroll positions


          scrollData.x = window.scrollX;
          scrollData.y = window.scrollY;
        } else if (fromRole === "page" && toRole === "page") {
          // If we're going from page to page then remove the scroll positions
          // because we don't want to restore in this case.
          fromPage.jqmRemoveData(scrollDataKey);
        }
      }
    });
  })(); // scrollPositionRestore
  ////////////////////////////////////////////////////////////////
  // Initialize
  //////////////////////////////////////////////////////////////// 
  // Handle when the user attempts to leave the page prematurely


  window.onbeforeunload = function () {
    if (PxPage.isCurrentPxPageModified()) {
      return mobileLex.get({
        key: 'message.viewModified',
        args: [PxPage.getCurrentPxPage().getDisplayName()]
      });
    }
  };

  function initializeUI() {
    //add in support for HxPx into app
    initializeMobileHxPx(); // Commands button

    $(document).on('onShowCommands', 'a.commandsButton', function () {
      var pxPage = PxPage.getCurrentPxPage();

      if (pxPage && window.niagara.env.type === "mobile") {
        views.getViewsCommand().setOrd(pxPage.getViewOrd());
      }
    });

    if (window.niagara.env.type === "mobile") {
      $(document).on('click', 'a.commandsButton', commands.showCommandsHandler);
    }

    function resetToViewOrd() {
      if (window.niagara.env.type === "mobile") {
        mobileUtil.changePage(baja.Ord.make(niagara.view.ord).toUri(), {
          transition: "none",
          changeHash: false
        });
      }
    }

    pages.register({
      pageshow: resetToViewOrd
    });
  }

  return initializeUI;
}); // pxAppView
