/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Color class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!', 'mobile/px/gx/gxStyles', 'mobile/util/color'], function (baja, gxStyles, colorUtil) {
  "use strict";

  var strictArg = baja.strictArg,
      colorCache = {},
      colorConstants = {},
      parseRgba = colorUtil.parseRgba,
      COLOR_CONSTANTS = colorUtil.COLOR_CONSTANTS;
  /**
   * @class Color.
   * 
   * This is currently a partial implementation of Niagara's 'gx:Color' Type.
   * 
   * When creating a Simple, always use the 'make' method instead of creating a new Object.
   *
   * @name Color
   * @extends baja.Simple
   */

  function Color(str, rgba) {
    baja.callSuper(Color, this, arguments);
    this.$str = strictArg(str, String);
    this.$rgba = strictArg(rgba, Object);
  }

  baja.subclass(Color, baja.Simple);

  function makeRgba() {
    return {
      red: 0,
      blue: 0,
      green: 0,
      alpha: 1,
      css: ""
    };
  }
  /**
   * Make a Color.
   * 
   * A Color can be made from a String (str) or from some rgba (red, green, blue or alpha) values.
   * If any of the RGBA values are specified, this will take precedence over a the String.
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} [obj.str] the String encoding for a Color.
   * @param {Number} [obj.red] the red color (0-255).
   * @param {Number} [obj.green] the green color (0-255).
   * @param {Number} [obj.blue] the blue color (0-255).
   * @param {Number} [obj.alpha] the alpha color (0-1).
   * @return {Color}
   */


  Color.make = function (obj) {
    obj = baja.objectify(obj, "str");
    var str = obj.str,
        rgba; // Create color from rgba if specified in Object Literal arguments

    if (obj.red !== undefined || obj.green !== undefined || obj.blue !== undefined || obj.alpha !== undefined) {
      rgba = makeRgba(); // TODO: Could check limits

      if (obj.red !== undefined) {
        rgba.red = Math.round(obj.red);
      }

      if (obj.green !== undefined) {
        rgba.green = Math.round(obj.green);
      }

      if (obj.blue !== undefined) {
        rgba.blue = Math.round(obj.blue);
      }

      if (obj.alpha !== undefined) {
        rgba.alpha = obj.alpha;
      } // Check for default


      if (rgba.red === 0 && rgba.green === 0 && rgba.blue === 0 && rgba.alpha === 1) {
        return Color.DEFAULT;
      }

      if (rgba.alpha === 1) {
        str = rgba.css = "rgb(" + rgba.red + "," + rgba.green + "," + rgba.blue + ")";
      } else {
        str = rgba.css = "rgba(" + rgba.red + "," + rgba.green + "," + rgba.blue + "," + rgba.alpha + ")";
      }
    } else {
      // Handle null
      if (str === "null" || str === "") {
        return Color.NULL;
      } // If this is a constant then return it


      var lowerCaseStr = str.toLowerCase();

      if (colorConstants.hasOwnProperty(lowerCaseStr)) {
        return colorConstants[lowerCaseStr];
      }
    } // Check the cache to see if it exists


    if (colorCache.hasOwnProperty(str)) {
      return colorCache[str];
    } // Create new color and add to the internal Cache


    var color = new Color(str, rgba || parseRgba(str));
    colorCache[str] = color;
    return color;
  };
  /**
   * Make a Color
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} obj.str the String encoding for a Color.
   * @return {Color}
   */


  Color.prototype.make = function (obj) {
    return Color.make.apply(Color, arguments);
  };
  /**
   * Decode a Color from a String
   *
   * @param {String} str
   * @return {Color}
   */


  Color.prototype.decodeFromString = function (str) {
    return Color.make(str);
  };
  /**
   * Encode the Color to a String
   *
   * @return {String}
   */


  Color.prototype.encodeToString = function () {
    return this.$str;
  }; // Color constants


  function constant(colName, str) {
    var c = new Color(colName, parseRgba(str));
    colorCache[str] = c;
    colorConstants[colName.toLowerCase()] = c;
    return c;
  } // Even though these are part of CSS3, we need to declare them so we can get access to their RGBA values.
  // The RGBA values are used in the spectrum bindings.


  for (var colorName in COLOR_CONSTANTS) {
    if (COLOR_CONSTANTS.hasOwnProperty(colorName)) {
      Color[colorName] = constant(colorName, COLOR_CONSTANTS[colorName]);
    }
  }
  /**
   * Default Color instance
   */


  Color.DEFAULT = Color.black;
  /**
   * Null Color instance
   */

  Color.NULL = new Color("", makeRgba());
  /**
   * Update an HTML DOM element.
   *
   * @param {Object} obj contains method arguments.
   */

  Color.prototype.update = function (obj) {
    gxStyles.applyColor(this, obj);
  };
  /**
   * Return the red part of the color
   *
   * @return {Number}
   */


  Color.prototype.getRed = function () {
    return this.$rgba.red;
  };
  /**
   * Return the green part of the color
   *
   * @return {Number}
   */


  Color.prototype.getGreen = function () {
    return this.$rgba.green;
  };
  /**
   * Return the blue part of the color
   *
   * @return {Number}
   */


  Color.prototype.getBlue = function () {
    return this.$rgba.blue;
  };
  /**
   * Return the alpha
   *
   * @return {Number}
   */


  Color.prototype.getAlpha = function () {
    return this.$rgba.alpha;
  };
  /**
   * Return the decoded CSS for the Color.
   */


  Color.prototype.getCss = function () {
    return this.$rgba.css || "";
  };

  return Color;
});
