/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview NumericToSimpleMap class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!'], function (baja) {
  "use strict";
  /**
   * @class Numeric To Simple Map.
   * 
   * A Component that represents a Niagara 'converters:NumericToSimpleMap' Type.
   * 
   * Please note, this isn't a complete implementation but will do for the Px App.
   *
   * @name NumericToSimpleMap
   * @extends baja.Simple
   */

  function NumericToSimpleMap(items, def) {
    baja.callSuper(NumericToSimpleMap, this, arguments);
    this.$items = items;
    this.$def = def;
  }

  baja.subclass(NumericToSimpleMap, baja.Simple);
  /**
   * Make a NumericToSimpleMap
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} obj.str the String encoding.
   * @return {NumericToSimpleMap}
   */

  NumericToSimpleMap.make = function (obj) {
    obj = baja.objectify(obj, "str");

    if (obj.str === "null") {
      return NumericToSimpleMap.DEFAULT;
    } // Get the prototype instance (used to decode everything else)


    var protoRes = /^([^ ]+) (.*)/.exec(obj.str);

    if (protoRes.length <= 1) {
      throw new Error("Cannot decode NumericToSimpleMap" + obj.str);
    } // Decode the prototype simple we're going to decode from


    var proto = baja.$(protoRes[1]); // Separate out the value pairs

    var valPairRegEx = /([^\;]*)=([^\;]*)/g;
    var valPairRes,
        defValue = null,
        value,
        numRegEx,
        numRes,
        items = [];
    valPairRes = valPairRegEx.exec(protoRes[2]);

    while (valPairRes) {
      // Decode the value from the String
      value = proto.decodeFromString(valPairRes[2]);

      if (valPairRes[1] === "default") {
        defValue = value;
      } else {
        // Decode the numeric range
        numRegEx = /([^:]*):([^:]*)/;
        numRes = numRegEx.exec(valPairRes[1]);

        if (!numRes || numRes && numRes.length !== 3) {
          throw new Error("Could not decode NumericToSimpleMap: " + valPairRes[1]);
        } // Add the item with the min and max range to our array


        items.push({
          min: baja.Double.DEFAULT.decodeFromString(numRes[1]).valueOf(),
          max: baja.Double.DEFAULT.decodeFromString(numRes[2]).valueOf(),
          value: value
        });
      }

      valPairRes = valPairRegEx.exec(protoRes[2]);
    }

    return new NumericToSimpleMap(items, defValue);
  };
  /**
   * Make a NumericToSimpleMap
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} obj.str the String encoding.
   * @return {NumericToSimpleMap}
   */


  NumericToSimpleMap.prototype.make = function (obj) {
    return NumericToSimpleMap.make.apply(NumericToSimpleMap, arguments);
  };
  /**
   * Decode a NumericToSimpleMap
   *
   * @param {String} str
   * @return {NumericToSimpleMap}
   */


  NumericToSimpleMap.prototype.decodeFromString = function (str) {
    return NumericToSimpleMap.make(str);
  };
  /**
   * Encode the NumericToSimpleMap to a String
   *
   * @return {String}
   */


  NumericToSimpleMap.prototype.encodeToString = function () {
    // TODO: Although the PxApp should never need to encode this, we should
    // probably implement this at some point.
    throw new Error("Currently not supported");
  };
  /**
   * Default NumericToSimpleMap instance
   */


  NumericToSimpleMap.DEFAULT = new NumericToSimpleMap([], null);
  /**
   * Equality test
   *
   * @param obj num
   * @return {Boolean}
   */

  NumericToSimpleMap.prototype.equals = function (obj) {
    var it, objIt, i;

    if (baja.hasType(obj) && obj.getType().equals(this.getType()) && obj.$items.length === this.$items.length) {
      // Compare items
      for (i = 0; i < this.$items.length; ++i) {
        it = this.$items[i];
        objIt = obj.$items[i];

        if (it.min !== objIt.min || it.max !== objIt.max || !it.value.equals(objIt.value)) {
          return false;
        }
      } // Compare default values


      if (this.$def === null) {
        return obj.$def === null;
      } else {
        if (obj.$def === null) {
          return false;
        } else {
          return obj.$def.equals(this.$def);
        }
      }
    }

    return false;
  };
  /**
   * Get a simple by its numeric value or return the default if no mapping.
   *
   * @param {Number} num
   * @return {baja.Simple} the value (or null if no default is specified).
   */


  NumericToSimpleMap.prototype.get = function (num) {
    var it, i;

    for (i = 0; i < this.$items.length; ++i) {
      it = this.$items[i];

      if (it.min <= num && num <= it.max) {
        return it.value;
      }
    }

    return this.$def;
  };

  return NumericToSimpleMap;
});
