/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview EnumToSimpleMap class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!'], function (baja) {
  "use strict";
  /**
   * @class Enum To Simple Map.
   * 
   * A Component that represents a Niagara 'converters:EnumToSimpleMap' Type.
   * 
   * Please note, this isn't a complete implementation but will do for the Px App.
   *
   * @name EnumToSimpleMap
   * @extends baja.Simple
   */

  function EnumToSimpleMap(items, def) {
    baja.callSuper(EnumToSimpleMap, this, arguments);
    this.$items = items;
    this.$def = def;
  }

  baja.subclass(EnumToSimpleMap, baja.Simple);
  /**
   * Make a EnumToSimpleMap
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} obj.str the String encoding.
   * @return {EnumToSimpleMap}
   */

  EnumToSimpleMap.make = function (obj) {
    /*jshint loopfunc: true */
    obj = baja.objectify(obj, "str");

    if (obj.str === "null") {
      return EnumToSimpleMap.DEFAULT;
    } // Get the prototype instance (used to decode everything else)


    var protoRes = /^([^ ]+) (.*)/.exec(obj.str);

    if (protoRes.length <= 1) {
      throw new Error("Cannot decode EnumToSimpleMap" + obj.str);
    } // Decode the prototype simple we're going to decode from


    var proto = baja.$(protoRes[1]); // Separate out the value pairs

    var valPairRegEx = /([^\;]*)=([^\;]*)/g;
    var valPairRes,
        defValue = null,
        value,
        items = [];
    valPairRes = valPairRegEx.exec(protoRes[2]);

    while (valPairRes) {
      // Decode the value from the String
      value = proto.decodeFromString(valPairRes[2]);

      if (valPairRes[1] === "default") {
        defValue = value;
      } else {
        //parse comma-delimited ordinals
        var ordinals = valPairRes[1].split(',');
        ordinals.forEach(function (ordinalStr) {
          items.push({
            ordinal: parseInt(ordinalStr, 10),
            value: value
          });
        });
      }

      valPairRes = valPairRegEx.exec(protoRes[2]);
    }

    return new EnumToSimpleMap(items, defValue);
  };
  /**
   * Make a EnumToSimpleMap
   *
   * @param {Object} obj Object Literal for the method's arguments.
   * @param {String} obj.str the String encoding.
   * @return {EnumToSimpleMap}
   */


  EnumToSimpleMap.prototype.make = function (obj) {
    return EnumToSimpleMap.make.apply(EnumToSimpleMap, arguments);
  };
  /**
   * Decode a EnumToSimpleMap
   *
   * @param {String} str
   * @return {EnumToSimpleMap}
   */


  EnumToSimpleMap.prototype.decodeFromString = function (str) {
    return EnumToSimpleMap.make(str);
  };
  /**
   * Encode the EnumToSimpleMap to a String
   *
   * @return {String}
   */


  EnumToSimpleMap.prototype.encodeToString = function () {
    // TODO: Although the PxApp should never need to encode this, we should
    // probably implement this at some point.
    throw new Error("Currently not supported");
  };
  /**
   * Default EnumToSimpleMap instance
   */


  EnumToSimpleMap.DEFAULT = new EnumToSimpleMap([], null);
  /**
   * Equality test
   *
   * @param obj num
   * @return {Boolean}
   */

  EnumToSimpleMap.prototype.equals = function (obj) {
    var it, objIt, i;

    if (baja.hasType(obj) && obj.getType().equals(this.getType()) && obj.$items.length === this.$items.length) {
      // Compare items
      for (i = 0; i < this.$items.length; ++i) {
        it = this.$items[i];
        objIt = obj.$items[i];

        if (it.ordinal !== objIt.ordinal || !it.value.equals(objIt.value)) {
          return false;
        }
      } // Compare default values


      if (this.$def === null) {
        return obj.$def === null;
      } else {
        if (obj.$def === null) {
          return false;
        } else {
          return obj.$def.equals(this.$def);
        }
      }
    }

    return false;
  };
  /**
   * Get a simple by its ordinal value or return the default if no mapping.
   *
   * @param {Number} ordinal
   * @return {baja.Simple} the value (or null if no default is specified).
   */


  EnumToSimpleMap.prototype.get = function (ordinal) {
    var it = this.$items,
        i;

    for (i = 0; i < it.length; ++i) {
      if (it[i].ordinal === ordinal) {
        return it[i].value;
      }
    }

    return this.$def;
  };

  return EnumToSimpleMap;
});
