/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Converter class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!', 'Promise'], function (baja, Promise) {
  "use strict";
  /**
   * @class Converter.
   * 
   * A Component that represents a Niagara 'baja:Converter' Type.
   * 
   * Converters are used to convert a value from one baja:Object Type to another. 
   *
   * @name Converter
   * @extends baja.Struct
   */

  function Converter() {
    baja.callSuper(Converter, this, arguments);
    this.$cVal = null;
  }

  baja.subclass(Converter, baja.Struct);
  /**
   * Update the converted by converting the first object to the second object.
   * 
   * Please note, this method should NOT be overridden.
   *
   * @see Converter#convert
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @param {Promise} promise to be resolved with the converted value.
   */

  Converter.prototype.update = function (from, to, target) {
    var that = this;
    return Promise.resolve(that.convert(from, to, target)).then(function (val) {
      // Update the cached converted value so it can be accessed synchronously
      that.$cVal = val;
      return val;
    });
  };
  /**
   * Convert the first object to the second object. This method is designed 
   * be to overridden by subclasses. It may return the converted value, or
   * a promise to be resolved with the same.
   *
   * @param {baja.Object} from.
   * @param {baja.Object} to.
   * @param {OrdTarget} target.
   * @returns {*|Promise}
   */


  Converter.prototype.convert = function (from, to, target) {
    return null;
  };
  /**
   * Return converted value. 
   * 
   * A converter stores the previously converted value it processed so it can access it
   * synchronously.
   *
   * @returns the converted value.
   */


  Converter.prototype.getConvertedValue = function () {
    return this.$cVal;
  };

  return Converter;
});
