/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview SetPointBinding class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!', 'Promise', 'mobile/util/mobile/dialogs', 'mobile/px/bindings/ValueBinding'], function (baja, Promise, dialogs, ValueBinding) {
  "use strict";

  var callSuper = baja.callSuper;

  function convertSetPointValue(from, to) {
    // If the same then easy as pie
    if (from.getType().equals(to.getType())) {
      return from;
    } // ------- Numeric -------
    // INumeric -> Numeric


    var x;

    if (from.getType().is("baja:INumeric") && to.getType().isNumber()) {
      x = Number.getNumberFromINumeric(from);

      switch (to.getDataTypeSymbol()) {
        case "i":
          return baja.Integer.make(x);

        case "l":
          return baja.Long.make(x);

        case "f":
          return baja.Float.make(x);

        case "d":
          return x;
      }
    } // INumeric -> StatusNumeric


    if (from.getType().is("baja:INumeric") && to.getType().is("baja:StatusNumeric")) {
      var statusNumeric = baja.$("baja:StatusNumeric");
      statusNumeric.setValue(Number.getNumberFromINumeric(from));
      return statusNumeric;
    } // ------- Boolean -------
    // IBoolean -> Boolean


    if (from.getType().is("baja:IBoolean") && to.getType().is("baja:Boolean")) {
      return Boolean.getBooleanFromIBoolean(from);
    } // IBoolean -> StatusBoolean


    if (from.getType().is("baja:IBoolean") && to.getType().is("baja:StatusBoolean")) {
      var statusBoolean = baja.$("baja:StatusBoolean");
      statusBoolean.setValue(Boolean.getBooleanFromIBoolean(from));
      return statusBoolean;
    } // ------- Enum -------
    // IEnum -> DynamicEnum


    if (from.getType().is("baja:IEnum") && to.getType().is("baja:DynamicEnum")) {
      return baja.Enum.getEnumFromIEnum(from);
    } // IEnum -> StatusEnum


    if (from.getType().is("baja:IEnum") && to.getType().is("baja:StatusEnum")) {
      var statusEnum = baja.$("baja:StatusEnum");
      statusEnum.setValue(baja.Enum.getEnumFromIEnum(from));
      return statusEnum;
    } // ------- String -------
    // String -> StatusString


    if (from.getType().is("baja:String") && to.getType().is("baja:StatusString")) {
      var statusString = baja.$("baja:StatusString");
      statusString.setValue(from.toString());
      return statusString;
    }

    baja.error("WARNING: No mechanism found to save");
    return null;
  }

  function verifySetPointBounds(val, facets) {
    if (!val.getType().isNumber() || facets === null) {
      return;
    } // Just incase this is a boxed type, get the inner value


    val = val.valueOf();
    var min = facets.get("min");

    if (min !== null) {
      min = min.valueOf();
    }

    var max = facets.get("max");

    if (max !== null) {
      max = max.valueOf();
    } // Check the bounds


    if (min !== null && val < min) {
      throw new Error();
    }

    if (max !== null && val > max) {
      throw new Error();
    }
  }

  function saveSetPointAction(params) {
    // TODO: Need to enforce facets here too
    var binding = params.binding,
        value = params.value,
        batch = params.batch,
        c = binding.$target.getComponent(),
        action = c.getSlot("set");

    if (!action || action && !action.isAction()) {
      return false;
    }

    if (!action.getParamType().equals(value.getType())) {
      return false;
    } // Invoke the Action (please note, this performs an asynchronous network call)


    return c.invoke({
      slot: action,
      value: value,
      batch: batch
    });
  }

  function saveSetPointProperty(params) {
    // Get the PropertyPath
    var binding = params.binding,
        value = params.value,
        batch = params.batch,
        target = binding.$target,
        c = target.getComponent(),
        path = target.propertyPath;

    if (!path || !path.length) {
      return;
    } // Make sure the Property is not readonly


    if ((baja.Flags.READONLY & c.getFlags(path[0])) === baja.Flags.READONLY) {
      return;
    } // Find appropriate Property on target


    var parent = c;
    baja.iterate(path, 0, path.length - 1, function (p) {
      parent = parent.get(p);
    }); // Check min/max

    verifySetPointBounds(value, c.getFacets(path[0])); // Make set (please note, this performs an asynchronous network call)

    return parent.set({
      slot: path[path.length - 1],
      value: value,
      batch: batch
    });
  }
  /**
   * @class SetPointBinding.
   * 
   * A Component that represents a Niagara 'kitPx:SetPointBinding' Type.
   *
   * @name SetPointBinding
   * @extends Binding
   */


  function SetPointBinding() {
    callSuper(SetPointBinding, this, arguments);
  }

  baja.subclass(SetPointBinding, ValueBinding);
  /**
   * Save the Widget Property driving the target
   *
   * @return {baja.Value} the value for the Widget Property (or null if there's no value available);
   */

  SetPointBinding.prototype.saveWidgetProperty = function () {
    var widget = this.getWidget(),
        prop = widget.getSlot(this.getWidgetProperty());

    if (prop === null) {
      return null;
    }

    return widget.get(prop);
  };
  /**
   * Save the Set Point Binding
   *
   * @param {Object} params
   * @param {baja.Value} params.value the value that will be saved.
   * @param {baja.comm.Batch} [params.batch] a batch to use to perform the
   * save. (Batch will be handled synchronously - safe to emit `COMMIT_READY`
   * afterwards.)
   */


  SetPointBinding.prototype.saveSetPoint = function (params) {
    var batch = params.batch,
        value = params.value,
        savePromise;

    if (!this.isBound()) {
      return Promise.resolve();
    }

    var targetVal = this.$target.getObject(); // If we are bound to a property under a Component,
    // then convert widget -> target and save Property

    if (!targetVal.getType().isComponent()) {
      var newVal = convertSetPointValue(value, targetVal);

      if (newVal === null) {
        return Promise.resolve();
      } // Try to save value as Property


      savePromise = saveSetPointProperty({
        binding: this,
        value: newVal,
        batch: batch
      });

      if (savePromise) {
        return savePromise;
      }
    }

    savePromise = saveSetPointAction({
      binding: this,
      value: value,
      batch: batch
    });

    if (savePromise) {
      return savePromise;
    }

    return Promise.reject(new Error('No mechanism found to save ' + this.getOrd())); // TODO: Implement save Action
  };
  /**
   * Handle an event from the Widget.
   *
   * @param {String} eventName.
   * @return {Boolean} return true if this binding has handled the event.
   */


  SetPointBinding.prototype.handleEvent = function (eventName) {
    // Please note, this binding overrides the default click behavior of
    // ValueBinding
    if (this.getWidgetEvent() === "actionPerformed") {
      try {
        var value = this.saveWidgetProperty();

        if (value === null) {
          return;
        }

        this.saveSetPoint({
          value: value
        });
      } catch (err) {
        dialogs.error(err);
      } // Save set point value


      return true;
    }

    return false;
  };
  /**
   * Called when the binding is updated.
   */


  SetPointBinding.prototype.update = function () {
    // Load the set point value
    var widget = this.getWidget(),
        prom;

    if (this.isBound()) {
      // handle SetPointFieldEditor special
      if (widget.getType().is("kitPx:SetPointFieldEditor")) {
        prom = widget.loadSetPoint(this);
      } else {
        var prop = widget.getSlot(this.getWidgetProperty());

        if (prop !== null) {
          // Get the current Widget value
          var widgetVal = widget.get(prop); // Get target value

          var targetVal = this.$target.getObject(); // convert target value

          var newVal = convertSetPointValue(targetVal, widgetVal); // If we have a solution then update the Widget Property

          if (newVal !== null) {
            prom = widget.set({
              slot: prop,
              value: newVal
            });
          }
        }
      }
    } // Call super


    return Promise.join(prom, callSuper("update", SetPointBinding, this, arguments));
  };
  /**
   * Return true if the Property is overridden.
   *
   * @param {baja.Property} prop the Property to test.
   * @return {Boolean} return true if the Property is overridden.
   */


  SetPointBinding.prototype.isOverridden = function (prop) {
    if (prop.getName() === this.getWidgetProperty()) {
      return true;
    } // Call super method


    return callSuper("isOverridden", SetPointBinding, this, arguments);
  };

  return SetPointBinding;
});
