/**
 * @license Copyright 2012, Tridium, Inc. All Rights Reserved.
 */

/**
 * @fileOverview Binding class
 * 
 * @author Gareth Johnson
 * @version 0.0.2.0
 */
define(['baja!', 'Promise'], function (baja, Promise) {
  "use strict";
  /**
   * @class Binding.
   * 
   * A Component that represents a Niagara 'bajaui:Binding' Type.
   * This Component's ORD is resolved and used to dynamically update a Widget
   * when a Px page is loaded.
   *
   * @extends baja.Component
   */

  function Binding() {
    baja.callSuper(Binding, this, arguments);
    this.$target = null;
  }

  baja.subclass(Binding, baja.Component);
  /**
   * Return the parent Widget.
   */

  Binding.prototype.getWidget = function () {
    return this.getParent();
  };
  /**
   * Return the ORD Target for this binding (null if not bound).
   */


  Binding.prototype.getTarget = function () {
    return this.$target;
  };
  /**
   * Called when the binding is first loaded.
   * @returns {Promise}
   */


  Binding.prototype.load = function () {
    // Widget Degradation
    var that = this,
        args = arguments,
        prom;

    if (!that.isBound()) {
      var widget = that.getWidget(),
          behavior = that.getDegradeBehavior();

      if (behavior.is("disable")) {
        widget.setEnabled(false);
        prom = widget.update(function (propName) {
          return propName === "enabled";
        });
      } else if (behavior.is("hide")) {
        widget.setVisible(false);
        prom = widget.update(function (propName) {
          return propName === "visible";
        });
      }
    }

    return Promise.resolve(prom).then(function () {
      return that.doLoad.apply(that, args);
    });
  };
  /**
   * Called when the binding is first loaded and is designed to be overridden.
   * It may or may not return a promise.
   * @returns {*|Promise}
   */


  Binding.prototype.doLoad = function () {};
  /**
   * Called when the binding is updated.
   * @returns {Promise}
   */


  Binding.prototype.update = function () {
    var that = this,
        widget = that.getWidget(); // Lazily create closure function for hasUpdate only once    

    if (!that.$widgetHasUpdate) {
      that.$widgetHasUpdate = function (propName) {
        var slot = widget.getSlot(propName);

        if (slot === null) {
          return false;
        }

        return that.isBound() && that.isOverridden(slot);
      };
    } // If there are no more converters to process then just update the Widget


    return widget.update(that.$widgetHasUpdate);
  };
  /**
   * Internal Framework Method
   *
   * @private
   */


  Binding.prototype.$fw = function (x, a, b, c, d) {
    if (x === "updateBinding") {
      this.$target = a; // Update the ORD Target for the binding

      return;
    }

    return Binding.$super.prototype.$fw.apply(this, arguments);
  };
  /**
   * Return the String for a binding.
   *
   * @return {String}
   */


  Binding.prototype.toString = function () {
    // Return the ORD for the binding (this is used when the page is loaded).
    return this.getOrd().toString();
  };
  /**
   * Return true if the Binding has a valid ORD Target.
   *
   * @return {Boolean}
   */


  Binding.prototype.isBound = function () {
    return !!this.$target;
  };
  /**
   * Return the override value for the given Property.
   *
   * @param {baja.Property} the Property to try and override
   * @return {baja.Value} the overridden value
   */


  Binding.prototype.getOnWidget = function (prop) {
    return null;
  };
  /**
   * Return true if the Property is overridden.
   *
   * @param {baja.Property} prop the Property to test.
   * @return {Boolean} true if overridden.
   */


  Binding.prototype.isOverridden = function (prop) {
    return false;
  };
  /**
   * Handle an event from the Widget.
   *
   * @param {String} eventName.
   * @return {Boolean} return true if this binding has handled the event.
   */


  Binding.prototype.handleEvent = function (eventName) {
    return false;
  };

  return Binding;
});
